#include "notifystate.h"
#include "symbian_auto_ptr.h"
#include "contextnotifyclientsession.h"
#include <gulicon.h>
#include <eikenv.h>
#include <bautils.h>
#include "app_context.h"

class CNotifyStateImpl: public CNotifyState, public MContextBase {
public:
	virtual ~CNotifyStateImpl();
private:
	CNotifyStateImpl(MApp_context& Context, const TDesC& aBitmapFilename);
	void ConstructL();

	void CheckedRunL();
	TInt CheckedRunError(TInt aError);
	void DoCancel();
	void SetCurrentState(TInt aIconId, TInt aMaskId);
	void StartL();
	void SetIcon();
	void Release();

	TFileName	iFilename;
	TInt		iNotifyId, iLocalId;

	enum TRunState { EIdle, EChanging };
	TRunState iRunState;
	TInt	iNextIconId, iNextMaskId;
	CGulIcon	*iIcon;
	RContextNotifyClientSession iNotifyClient;
	CLocalNotifyWindow* iLocalWindow;

	friend class CNotifyState;
};

EXPORT_C CNotifyState* CNotifyState::NewL(MApp_context& Context, const TDesC& aBitmapFilename)
{
	CALLSTACKITEM2_N(_CL("CNotifyState"), _CL("NewL"), &Context);

	auto_ptr<CNotifyStateImpl> ret(new (ELeave) CNotifyStateImpl(Context, aBitmapFilename));
	ret->ConstructL();
	return ret.release();
}

CNotifyState::CNotifyState() : CCheckedActive(EPriorityNormal, _L("CNotifyStateImpl"))
{
}

CNotifyStateImpl::CNotifyStateImpl(MApp_context& Context, const TDesC& aBitmapFilename) : 
	MContextBase(Context), iFilename(aBitmapFilename), iNotifyId(-1), iNextIconId(-1),
		iLocalId(-1)
{
}

void CNotifyStateImpl::Release()
{
	CALLSTACKITEM_N(_CL("CNotifyStateImpl"), _CL("Release"));

	Cancel();
	iNotifyClient.Close();
	iLocalWindow=CLocalNotifyWindow::Global();
	if (iLocalWindow && iLocalId>=0) iLocalWindow->RemoveIcon(iLocalId);
	iLocalId=-1;
	iNotifyId=-1;
	delete iIcon; iIcon=0;
}

CNotifyStateImpl::~CNotifyStateImpl()
{
	CALLSTACKITEM_N(_CL("CNotifyStateImpl"), _CL("~CNotifyStateImpl"));

	Release();
}

void CNotifyStateImpl::ConstructL()
{
	CALLSTACKITEM_N(_CL("CNotifyStateImpl"), _CL("ConstructL"));

	if (! BaflUtils::FileExists(Fs(), iFilename) ) {
		iFilename.Replace(0, 1, _L("e"));
		if (! BaflUtils::FileExists(Fs(), iFilename) ) User::Leave(KErrNotFound);
	}
	CActiveScheduler::Add(this);
	iLocalWindow=CLocalNotifyWindow::Global();
	StartL();
}

void CNotifyStateImpl::StartL()
{
	CALLSTACKITEM_N(_CL("CNotifyStateImpl"), _CL("StartL"));

	User::LeaveIfError( iNotifyClient.Connect() );
}

void CNotifyStateImpl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CNotifyStateImpl"), _CL("CheckedRunL"));

	iRunState=EIdle;
	if (iStatus!=KErrNone) 
		User::Leave(iStatus.Int());
}

TInt CNotifyStateImpl::CheckedRunError(TInt aError)
{
	CALLSTACKITEM_N(_CL("CNotifyStateImpl"), _CL("CheckedRunError"));

	Release();
	TRAPD(err, StartL());
	if (err==KErrNone) {
		SetIcon();
	}
	return err;
}

void CNotifyStateImpl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CNotifyStateImpl"), _CL("DoCancel"));

	iNotifyClient.Cancel();
}

void CNotifyStateImpl::SetCurrentState(TInt aIconId, TInt aMaskId)
{
	CALLSTACKITEM_N(_CL("CNotifyStateImpl"), _CL("SetCurrentState"));

	iNextIconId=aIconId; iNextMaskId=aMaskId;
	if (iRunState==EIdle) {
		SetIcon();
	}
}

void CNotifyStateImpl::SetIcon()
{
	CALLSTACKITEM_N(_CL("CNotifyStateImpl"), _CL("SetIcon"));

	delete iIcon; iIcon=0;

	iIcon=CEikonEnv::Static()->CreateIconL(iFilename, iNextIconId, iNextMaskId);

	iStatus=KRequestPending;
	iLocalWindow=CLocalNotifyWindow::Global();
	if (iLocalId>=0) {
		if (iLocalWindow) iLocalWindow->ChangeIconL(iIcon->Bitmap(), iIcon->Mask(), iLocalId);
	} else {
		if (iLocalWindow) iLocalId=iLocalWindow->AddIconL(iIcon->Bitmap(), iIcon->Mask());
	}
	if (iNotifyId>=0) {
		iNotifyClient.ChangeIcon(iIcon->Bitmap(), iIcon->Mask(), iNotifyId, iStatus);
	} else {
		iNotifyClient.AddIcon(iIcon->Bitmap(), iIcon->Mask(), iNotifyId, iStatus);
	}
	iRunState=EChanging;
	SetActive();
}
