/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/

/*
 * Concepts:
 * !Bluetooth scan!
 * !Switching Bluetooth on!
 * !Checking for existing Bluetooth connections!
 */

#include "discover.h"
#include "cl_settings.h"

#include <e32base.h>
#include <f32file.h>
#include <e32std.h>

#include "log_base_impl.h"
#include "timeout.h"


_LIT(KBtAccessMutex, "ContextBtMutex");

#ifndef NO_BTENG_H
#include <bteng.h>
#include <btmanclient.h>
#include <bttypes.h>
#else
#include <btmanclient.h>
#include <bttypes.h>
// reverse engineered from BTENG.LIB:
enum TBTDiscoverabilityMode { EBTDiscoverabilityMode0, EBTDiscoverabilityMode1 };
enum TBTSearchMode { EBTSearchMode0, EBTSearchMode1 };
class CBTMCMSettings : public CBase {
public:
        IMPORT_C static int  GetAllSettings(int &, enum TBTDiscoverabilityMode &, enum TBTSearchMode &, class TDes16 &, int &);
        IMPORT_C static int  GetDiscoverabilityModeL(enum TBTDiscoverabilityMode &);
        IMPORT_C static int  GetLocalBDAddress(class TBTDevAddr &);
        IMPORT_C static int  GetLocalBTName(class TDes16 &);
        IMPORT_C static int  GetPowerStateL(int &);
        IMPORT_C static int  GetSearchModeL(enum TBTSearchMode &);
        IMPORT_C static int  IsLocalNameModified(int &);
        IMPORT_C static class CBTMCMSettings *  NewL(class MBTMCMSettingsCB *);
        IMPORT_C static class CBTMCMSettings *  NewLC(class MBTMCMSettingsCB *);
        IMPORT_C int  SetDefaultValuesL(void);
        IMPORT_C int  SetDiscoverabilityModeL(enum TBTDiscoverabilityMode, int);
        IMPORT_C int  SetLocalBTName(class TDesC16 const &);
        IMPORT_C int  SetPowerStateL(int, int);
        IMPORT_C int  SetSearchModeL(enum TBTSearchMode);
};
#endif

#include <es_sock.h>
#include <bt_sock.h>

#include "csd_bluetooth.h"
#include "mutexrequest.h"
#include "reporting.h"

class CDiscoverImpl: public CDiscover, public CCheckedActive, public MTimeOut, public MSettingListener
{
public:
	~CDiscoverImpl();
	
	static CDiscoverImpl* NewL(MApp_context& Context);
private:
	void ConstructL();
	CDiscoverImpl(MApp_context& Context);
	virtual void DoCancel();
	virtual void CheckedRunL();
	virtual const TBTDevAddr& GetOwnAddress() { return own_address; }
	virtual void SettingChanged(TInt Setting);
private:
	enum comm_state { IDLE, WAITING_ON_MUTEX, GETTING_LOCAL, FIRST_SEARCH, NEXT_SEARCH,
		WAITING_ON_BT_BUSY };
	comm_state current_state;
	CBBBtDeviceList* names;
	CBBBtDeviceList* prev_env;

	RHostResolver hr; bool hr_is_open;
	RSocket socket; bool socket_is_open;
	TBTDevAddr own_address; TPckgBuf<TBTDevAddr> iDevAddrPckg;
	RSocketServ socket_server; bool socket_server_is_open;
	TInquirySockAddr addr;
	TNameEntry name;
	TProtocolDesc pInfo;
	TInt	iInterval;

	RTimer timer; bool timer_is_open;
	CTimeOut*	iTimeOut;
	TInt		iTimeOutCount;
	TBool		iFirst;
	CMutexRequest*	iMutex;
	void ReleaseMutex();

	TTime		iPrevPost;
#ifndef NO_BTENG_H
	RBTDeviceHandler iBtHandler;
#endif

	void release();
	void finished();
	void Restart();
	void expired(CBase* source);
	
	friend class CDiscover;
};

EXPORT_C CDiscover* CDiscover::NewL(MApp_context& Context)
{
	CALLSTACKITEM2_N(_CL("CDiscover"), _CL("NewL"),  &Context);

	auto_ptr<CDiscoverImpl> ret(new (ELeave) CDiscoverImpl(Context));
	ret->ConstructL();
	return ret.release();
}

EXPORT_C CDiscover::~CDiscover()
{
}

CDiscover::CDiscover(MApp_context& Context) : Mlog_base_impl(Context, KCSDBtList, KBluetoothTuple)
{
}

CDiscoverImpl::CDiscoverImpl(MApp_context& Context) : CDiscover(Context),
CCheckedActive(CCheckedActive::EPriorityIdle, _L("CDiscoverImpl")), 
current_state(IDLE), iFirst(ETrue) { }

void CDiscoverImpl::ConstructL() 
{
	CALLSTACKITEM_N(_CL("CDiscoverImpl"), _CL("ConstructL"));

	names=CBBBtDeviceList::NewL();
	prev_env=CBBBtDeviceList::NewL();

	Settings().NotifyOnChange(SETTING_BT_SCAN_INTERVAL, this);

	Mlog_base_impl::ConstructL();
	iTimeOut=CTimeOut::NewL(*this);

	CActiveScheduler::Add(this); // add to scheduler

	Restart();
}

void CDiscoverImpl::Restart()
{
	if (prev_env->Count()>0) {
		prev_env->Reset();
		iEvent.iData()=prev_env;
		iEvent.iStamp()=GetTime();
		iPrevPost=GetTime();
		post_new_value(iEvent);
	}
	Cancel();
	release();
	iTimeOut->Reset();
	names->Reset();
	current_state=IDLE;

	iInterval=5*60;
	Settings().GetSettingL(SETTING_BT_SCAN_INTERVAL, iInterval);
	if (iInterval==0) return;

	User::LeaveIfError(timer.CreateLocal()); timer_is_open=true;
#ifndef __WINS__
	User::LeaveIfError(socket_server.Connect()); socket_server_is_open=true;
	TProtocolName aName=_L("BTLinkManager");
	User::LeaveIfError(socket_server.FindProtocol(aName, pInfo));
	User::LeaveIfError(socket.Open(socket_server, KBTAddrFamily, KSockSeqPacket, KL2CAP));	
	User::LeaveIfError(hr.Open(socket_server,pInfo.iAddrFamily,pInfo.iProtocol)); hr_is_open=true;

	auto_ptr<CBTMCMSettings> BtSettings(CBTMCMSettings::NewL(0));
	TBool power;
	BtSettings->GetPowerStateL(power);
	if (!power) {
		//FIXME: post_new_value(_L("Powering up Bt"), Mlogger::INFO);
		BtSettings->SetPowerStateL(ETrue, EFalse);
	}

	socket.Ioctl(KHCILocalAddressIoctl, iStatus, &iDevAddrPckg, KSolBtHCI);
	SetActive();
	current_state=GETTING_LOCAL;
#else // __WINS__
	TTimeIntervalMicroSeconds32 w(5*1000*1000);
	timer.After(iStatus, w);
	SetActive();
#endif
}

CDiscoverImpl* CDiscoverImpl::NewL(MApp_context& Context)
{
	CALLSTACKITEM2_N(_CL("CDiscoverImpl"), _CL("NewL"),  &Context);

	auto_ptr<CDiscoverImpl> ret(new (ELeave) CDiscoverImpl(Context));
	ret->ConstructL();
	return ret.release();
}

void CDiscoverImpl::release()
{	
	CALLSTACKITEM_N(_CL("CDiscoverImpl"), _CL("release"));

	if (socket_is_open) socket.Close();
	socket_is_open=false;

	if (timer_is_open) timer.Close();
	timer_is_open=false;
	if (hr_is_open) hr.Close();
	hr_is_open=false;
	if (socket_server_is_open) socket_server.Close();
	socket_server_is_open=false;
}

void CDiscoverImpl::ReleaseMutex()
{
	if (!iMutex) return;

	delete iMutex; iMutex=0;
	Reporting().DebugLog(_L("discover::releasemutex"));
}

CDiscoverImpl::~CDiscoverImpl()
{
	CALLSTACKITEM_N(_CL("CDiscoverImpl"), _CL("~CDiscoverImpl"));

	Settings().CancelNotifyOnChange(SETTING_BT_SCAN_INTERVAL, this);

	delete iTimeOut;
	Cancel();
	release();

	ReleaseMutex();

	delete names; delete prev_env;
}	

void CDiscoverImpl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CDiscoverImpl"), _CL("DoCancel"));

	switch(current_state) {
	case IDLE:
		timer.Cancel();
		break;
	case GETTING_LOCAL:
		socket.CancelIoctl();
		break;
	case WAITING_ON_MUTEX:
		ReleaseMutex();
		break;
	default:
		hr.Cancel();
		break;
	}
}


void CDiscoverImpl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CDiscoverImpl"), _CL("CheckedRunL"));

	iTimeOut->Reset();
	TBuf<30> msg;
	TInt async_ret=iStatus.Int();
	if (async_ret!=KErrNone && async_ret!=KErrHostResNoMoreResults ) {
		_LIT(stat_err, "error: %d at state %d");
		msg.Format(stat_err, iStatus.Int(), current_state);
		Reporting().DebugLog(msg);
		post_error(msg, iStatus.Int(), GetTime());
		if (current_state!=GETTING_LOCAL) {
			ReleaseMutex();
			Restart();
			return;
		}
	}
#ifndef __WINS__
	switch(current_state) {
	case GETTING_LOCAL:
		{
		if (socket_is_open) socket.Close();
		socket_is_open=false;

		if (iInterval>0) {
			TTimeIntervalMicroSeconds32 w;
			if (iFirst)
				w=TTimeIntervalMicroSeconds32(3*1000*1000);
			else
				w=TTimeIntervalMicroSeconds32(iInterval*1000*1000);
			iFirst=EFalse;
			timer.After(iStatus, w);
			SetActive();
			current_state=IDLE;
		}

		if (async_ret==KErrNone) {
			own_address=iDevAddrPckg();
			CBBSensorEvent e(KCSDBtOwnAddress, KOwnBluetoothTuple); e.iData.SetOwnsValue(EFalse);
			bb_auto_ptr<TBBBtDeviceInfo> di(new (ELeave) TBBBtDeviceInfo(iDevAddrPckg, _L(""),  0, 0, 0));
			names->AddItemL(di.get());
			di.release();
			e.iStamp()=GetTime();
			e.iData()=names;

			iPrevPost=GetTime();
			post_new_value(e);
		}
		break;
		}
	case FIRST_SEARCH:
		{
		if (iStatus.Int()!=KErrHostResNoMoreResults) {
			names->Reset();
			TInquirySockAddr btaddr(name().iAddr);
			bb_auto_ptr<TBBBtDeviceInfo> di(new (ELeave) TBBBtDeviceInfo(btaddr, name().iName));
			names->AddItemL(di.get());
			di.release();

			hr.Next(name, iStatus);
			iTimeOut->Wait(3*60);
			SetActive();
			current_state=NEXT_SEARCH;
		} else {
			ReleaseMutex();
			Reporting().DebugLog(_L("discover: finished scan"));
			if (prev_env->Count()>0 || iPrevPost+TTimeIntervalMinutes(5)<GetTime()) {
				prev_env->Reset();
				iPrevPost=GetTime();
				post_new_value(prev_env);
			}
			current_state=IDLE;
			iTimeOutCount=0;
			if (iInterval>0) {
				TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
				timer.After(iStatus, w);
				SetActive();
			}
		}
		}
		break;
	case NEXT_SEARCH:
		{
		if (iStatus.Int()!=KErrHostResNoMoreResults) {
			TInquirySockAddr btaddr(name().iAddr);
			bb_auto_ptr<TBBBtDeviceInfo> di(new (ELeave) TBBBtDeviceInfo(btaddr, name().iName));
			names->AddItemL(di.get());
			di.release();
			
			hr.Next(name, iStatus);
			iTimeOut->Wait(3*60);
			SetActive();
		} else {
			ReleaseMutex();
			Reporting().DebugLog(_L("discover: finished scan"));
			if (prev_env->Equals(names)) {
				iEvent.iPriority()=CBBSensorEvent::UNCHANGED_VALUE;
				iEvent.iData()=names;
				iEvent.iStamp()=GetTime();
				iPrevPost=GetTime();
				post_new_value(iEvent);
			} else {
				iEvent.iData()=names;
				iEvent.iPriority()=CBBSensorEvent::VALUE;
				iEvent.iStamp()=GetTime();
				post_new_value(iEvent);
				CBBBtDeviceList* tmp=prev_env;
				prev_env=names; names=tmp;
			}
			current_state=IDLE;
			iTimeOutCount=0;
			if (iInterval>0) {
				TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
				timer.After(iStatus, w);
				SetActive();
			}
		}
		}
		break;
	case IDLE:
		{
		TTimeIntervalMicroSeconds32 w(120*1000*1000);
		current_state=WAITING_ON_MUTEX;
		Reporting().DebugLog(_L("discover::wait_on_mutex"));
		iMutex=CMutexRequest::NewL(AppContext(), KBtAccessMutex, w, &iStatus);
		SetActive();
		break;
		}
	case WAITING_ON_MUTEX:
		Reporting().DebugLog(_L("discover:got_mutex"));
	case WAITING_ON_BT_BUSY:
		{
		if (iStatus!=KErrNone) {
			ReleaseMutex();
			current_state=IDLE;
			if (iInterval>0) {
				TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
				timer.After(iStatus, w);
				SetActive();
			}
			return;
		}
		TBool connections=EFalse;
#ifndef NO_BTENG_H
		iBtHandler.AreExistingConnectionsL(connections);
#endif
		if (connections) {
			if (current_state==WAITING_ON_BT_BUSY) {
				ReleaseMutex();
				current_state=IDLE;
				Reporting().DebugLog(_L("discover: BT connections in use"));
				if (iInterval>0) {
					TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
					timer.After(iStatus, w);
					SetActive();
				}
				post_error(_L("BT connections in use"), KErrInUse);
			} else {
				current_state=WAITING_ON_BT_BUSY;
				TTimeIntervalMicroSeconds32 w(5*1000*1000);
				timer.After(iStatus, w);
				SetActive();
			}			
		} else {
			if (iInterval>0) {
				Reporting().DebugLog(_L("discover: starting scan"));
				addr.SetIAC(KGIAC);
				//addr.SetAction(KHostResInquiry|KHostResName|KHostResIgnoreCache);
				// don't ask for host names for quicker and more robust
				// inquiry
				addr.SetAction(KHostResInquiry|KHostResIgnoreCache);
				hr.GetByAddress(addr, name, iStatus);

				SetActive();
			
				iTimeOut->Wait(3*60);
				current_state=FIRST_SEARCH;
			}
		}
		break;
		}
	}
#else
	names->Reset();
	CBBSensorEvent e(KCSDBtList, KBluetoothTuple); e.iData.SetOwnsValue(EFalse);
	//bb_auto_ptr<TBBBtDeviceInfo> di(new (ELeave) TBBBtDeviceInfo(_L8("\x01\x20\xe0\x4c\x71\xb9"), _L(""),  2, 3, 4));

	_LIT8(mac, "\x00\x12\x62\xe3\x53\xb1");
	bb_auto_ptr<TBBBtDeviceInfo> di(new (ELeave) 
		TBBBtDeviceInfo(mac, _L(""),  2, 3, 4));
	names->AddItemL(di.get());
	di.release();
	e.iData()=names;
	e.iStamp()=GetTime();

	post_new_value(e);

	TTimeIntervalMicroSeconds32 w(10*1000*1000);
	timer.After(iStatus, w);
	SetActive();
	//FIXME: post_new_value(_L("0020e04c71b8 [GLOMGOLD-25,1:1:2] 0002ee51c437 [Reno7650,2:3:4] 0002ee51c438 [Reno7651,2:3:4]"));

#endif

}

void CDiscoverImpl::expired(CBase* /*source*/)
{
	CALLSTACKITEM_N(_CL("CDiscoverImpl"), _CL("expired"));
	if (iTimeOutCount>5) {
		post_error(_L("too many timeouts in Bt discovery"), KErrTimedOut, GetTime());
		User::Leave(-1029);
	}
	iTimeOutCount++;
	post_error(_L("timeout in Bt discover"), KErrTimedOut, GetTime());
	Restart();
}

void CDiscoverImpl::SettingChanged(TInt /*Setting*/)
{
	TInt prev=iInterval;
	iInterval=5*60;
	Settings().GetSettingL(SETTING_BT_SCAN_INTERVAL, iInterval);
	if (prev==0 && iInterval!=0 && current_state==IDLE) Restart();
}
