#include "log_alarm.h"

#include "symbian_auto_ptr.h"
#include "checkedactive.h"

#define MAX_ERRORS	10

#include "csd_presence.h"
#include <t32alm.h>

/*
 * Concepts:
 * !Accessing clock alarms!
 */

class CLogAlarmImpl : public CLogAlarm {
public:
	virtual ~CLogAlarmImpl();
private:
	CLogAlarmImpl(MApp_context& Context);
	void ConstructL();

	void CheckedRunL();
	void DoCancel();
	void GetAlarm();
	void NotifyOfChange();

	RAlarmServer	iAlarmServer;
	TBBTime		iValue;
	TInt		iErrorCount;

	friend class CLogAlarm;
};

EXPORT_C CLogAlarm* CLogAlarm::NewL(MApp_context& Context)
{
	CALLSTACKITEM2_N(_CL("CLogAlarm"), _CL("NewL"), &Context);

	auto_ptr<CLogAlarmImpl> ret(new (ELeave) CLogAlarmImpl(Context));
	ret->ConstructL();
	return ret.release();
}

_LIT(KClass, "CLogAlarm");

CLogAlarm::CLogAlarm(MApp_context& Context) : CCheckedActive(EPriorityNormal, KClass),
	Mlog_base_impl(Context, KAlarm, KAlarmTuple) { }

CLogAlarmImpl::~CLogAlarmImpl()
{
	CALLSTACKITEM_N(_CL("CLogAlarmImpl"), _CL("~CLogAlarmImpl"));
	Cancel();
	iAlarmServer.Close();
}

_LIT(KAlarmTime, "alarm_due");

CLogAlarmImpl::CLogAlarmImpl(MApp_context& Context) : 
	CLogAlarm(Context), iValue(KAlarmTime) { }

void CLogAlarmImpl::ConstructL()
{
	CALLSTACKITEM_N(_CL("CLogAlarmImpl"), _CL("ConstructL"));
	Mlog_base_impl::ConstructL();
	CActiveScheduler::Add(this);

	User::LeaveIfError(iAlarmServer.Connect());

	iEvent.iData()=&iValue; iEvent.iData.SetOwnsValue(EFalse);

	GetAlarm();
	post_new_value(&iValue);
	NotifyOfChange();
}

void CLogAlarmImpl::GetAlarm()
{
	CALLSTACKITEM_N(_CL("CLogAlarmImpl"), _CL("GetAlarm"));
	TAlarmInfo info;
	TAlarmSetState state;

	TInt i, err=-1;
	auto_ptr<CAlarmIdArray> ids(new (ELeave) CAlarmIdArray(8));
	iAlarmServer.AlarmArrayPopulateL(*ids.get(), RAlarmServer::EArrayNext, 8);
	iValue()=TTime(0);
	for (i=0; i<ids->Count(); i++) {
		err=iAlarmServer.AlarmInfo(info, RAlarmServer::EInfoClock, ids->At(i));
		if (err==KErrNone ) {
			state=iAlarmServer.AlarmState(info.iAlarmId);
			if (state!=EAlarmNotSet && state!=EAlarmDisabled)
				break;
		}
		if (err!=KErrNotFound) {
			User::LeaveIfError(err);
		}
	}
	if (err==KErrNone) {
		/* AlarmTime is when the alarm will next go off,
		 * e.g., after a snooze
		 */
		iValue()=info.iAlarmTime;
	}
}

void CLogAlarmImpl::NotifyOfChange()
{
	CALLSTACKITEM_N(_CL("CLogAlarmImpl"), _CL("NotifyOfChange"));
	iAlarmServer.NotifyOnChange(iStatus);
	SetActive();
}

void CLogAlarmImpl::DoCancel()
{
	CALLSTACKITEM_N(_CL("CLogAlarmImpl"), _CL("DoCancel"));
	iAlarmServer.NotifyOnChangeCancel();
}

void CLogAlarmImpl::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CLogAlarmImpl"), _CL("CheckedRunL"));
	if (iStatus.Int() < 0) {
		if (iErrorCount++ > MAX_ERRORS) User::Leave(iStatus.Int());
	} else {
		iErrorCount=0;
	}

	GetAlarm();
	post_new_value(&iValue);
	NotifyOfChange();
}