/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


/*
 * Concepts:
 * !Creating a server!
 */

//CContextServer.cpp
#include "ver.h"
#include "ContextServer.h"
#include "ContextServerSession.h"
#include "ContextCommon.h"
#include <E32SVR.H>
#include <basched.h>

#include "app_context.h"
#include "app_context_impl.h"
#include <flogger.h>
#include "callstack.h"

void Log(const TDesC& msg);

bool ClientAlive(TInt ThreadId);

#pragma warning(disable:4706)

CContextServer::CContextServer(TInt aPriority, MApp_context& Context) : CServer(aPriority), MContextBase(Context)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("CContextServer"));
}

CContextServer::~CContextServer()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("~CContextServer"));

	delete iPresenceInfo;
	delete iMessageHolder;
	delete iJabberClient;
}

CContextServer* CContextServer::NewL(MApp_context& Context)
{
	CALLSTACKITEM2_N(_CL("CContextServer"), _CL("NewL"), &Context);

	CContextServer* contextServer = CContextServer::NewLC(Context);
    	CleanupStack::Pop(contextServer);
    	return contextServer;
}

CContextServer* CContextServer::NewLC(MApp_context& Context)
{
	CALLSTACKITEM2_N(_CL("CContextServer"), _CL("NewLC"), &Context);

	CContextServer* contextServer = new (ELeave) CContextServer(EPriorityNormal, Context);
    	CleanupStack::PushL(contextServer);
    	contextServer->ConstructL();
    	return contextServer;
}

void CContextServer::ConstructL()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("ConstructL"));

	iPresenceInfo = CPresenceInfo::NewL();
	iMessageHolder = CMessageHolder::NewL();

	iJabberClient = CJabber::NewL(*this, AppContext());
	//iHttp = CHttp::NewL(*this, AppContext());
	//iHttp->GetL(1, _L("http://www.cs.helsinki.fi/u/petit/test.txt"));
	//iHttp->GetL(1, _L("http://www.cs.helsinki.fi/u/petit/test.txt"), TTime(0), 10, 100);

	StartL(KContextServerName);
}

CSharableSession* CContextServer::NewSessionL(const TVersion& aVersion) const
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("NewSessionL"));

	// check version
	if (!User::QueryVersionSupported(TVersion(KContextServMajorVersionNumber,
                                                KContextServMinorVersionNumber,
                                                KContextServBuildVersionNumber),
                                                aVersion))
	{
		User::Leave(KErrNotSupported);
	}
	
	// create new session
	RThread client = Message().Client();
	return CContextServerSession::NewL(client, *const_cast<CContextServer*> (this));
}


void CContextServer::IncrementSessions()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("IncrementSessions"));

    	iSessionCount++;
}

void CContextServer::DecrementSessions()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("DecrementSessions"));

    	iSessionCount--;
    	if (iSessionCount <= 0) {
      		iSessionCount =0;
		TRAPD(err, iJabberClient->Disconnect(EFalse));
		CActiveScheduler::Stop();
	}
}

TInt CContextServer::CheckedRunError(TInt aError)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("CheckedRunError"));

	if (aError == KErrBadDescriptor)
	{
        	// A bad descriptor error implies a badly programmed client, so panic it;
        	// otherwise report the error to the client
        	PanicClient(Message(), ECSBadDescriptor);
	}
	else
	{
		Message().Complete(aError);
	}

	//
	// The leave will result in an early return from CServer::RunL(), skipping
	// the call to request another message. So do that now in order to keep the
	// server running.
	ReStart();

	return KErrNone;	// handled the error fully
}

void CContextServer::PanicClient(const RMessage& aMessage, TContextServPanic aPanic)
{
	CALLSTACKITEMSTATIC_N(_CL("CContextServer"), _CL("PanicClient"));

	aMessage.Panic(KContextServer, aPanic);
}

void CContextServer::PanicServer(TContextServPanic aPanic)
{
	CALLSTACKITEMSTATIC_N(_CL("CContextServer"), _CL("PanicServer"));

    	User::Panic(KContextServer, aPanic);
}

void CContextServer::RunL()
{
	AppContext().CallStackMgr().ResetCallStack();


	TRAPD(err, CServer::RunL());
	if (err!=KErrNone) {
		TBuf<30> msg;
		msg.Format(_L("Error in RunL: %d"), err);
		Log(msg);
		CheckedRunError(err);
	}

	AppContext().CallStackMgr().ResetCallStack();
}

void CContextServer::ThreadFunctionL()
{
	#if defined (__WINS__)
	UserSvr::ServerStarted();
	#endif

	__UHEAP_MARK;

	CApp_context* c=CApp_context::NewL(false, _L("ContextServer"));
	CleanupStack::PushL(c);
	{

	// make sure the call stack item is not on the stack
	// after app context is deleted

    	// Construct active scheduler
    	CActiveScheduler* activeScheduler = new (ELeave) CBaActiveScheduler;

    	CleanupStack::PushL(activeScheduler) ;

    	// Install active scheduler
    	// We don't need to check whether an active scheduler is already installed
    	// as this is a new thread, so there won't be one
    	CActiveScheduler::Install(activeScheduler);

    	// Construct our server
	CContextServer::NewLC(*c);    

	RSemaphore semaphore;
	if (semaphore.CreateGlobal(KContextServerSemaphoreName, 0)!=KErrNone) {
		User::LeaveIfError(semaphore.OpenGlobal(KContextServerSemaphoreName));
	}
	
	// Semaphore opened ok
	semaphore.Signal();
	semaphore.Close();

	// Start handling requests
	CActiveScheduler::Start();
	CleanupStack::PopAndDestroy(2); // CServer, activeScheduler, 
	}
		 
	CleanupStack::PopAndDestroy(1); // appcontext

	__UHEAP_MARKEND;

}

TInt CContextServer::ThreadFunction(TAny* /*aNone*/)
{
    	CTrapCleanup* cleanupStack = CTrapCleanup::New();
	if (cleanupStack == NULL)
	{
      	PanicServer(ECSCreateTrapCleanup);
	}

	TRAPD(err, ThreadFunctionL());
    	if (err != KErrNone)
	{
        	PanicServer(ECSSrvCreateServer);
	}
	
    	delete cleanupStack;
    	cleanupStack = NULL;

	return KErrNone;
}


//---------------------------------------------------------------------------

void CContextServer::TerminateContextServer()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("TerminateContextServer"));

	iJabberClient->Disconnect(EFalse);
	NotifySessions(ETerminated);
	CActiveScheduler::Stop();
}

void CContextServer::ConnectToPresenceServer(TDesC & username, TDesC & password, TDesC & server, TUint32 accessPoint)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("ConnectToPresenceServer"));

	iAccessPoint = accessPoint;
	iUsername.Copy(username);
	iPassword.Copy(password);
	iServer.Copy(server);

	iJabberClient->ConnectL(username,password,server,accessPoint);
}

void CContextServer::ConnectToPresenceServer(const RMessage &aMessage)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("ConnectToPresenceServer"));

	const TAny* clientsDescriptor0 = aMessage.Ptr0();
	TBuf<30> username;
	aMessage.ReadL(clientsDescriptor0, username);

	const TAny* clientsDescriptor1 = aMessage.Ptr1();
	TBuf<30> password;
	aMessage.ReadL(clientsDescriptor1, password);

	const TAny* clientsDescriptor2 = aMessage.Ptr2();
	TBuf<30> server;
	aMessage.ReadL(clientsDescriptor2, server);

	const TAny* clientsDescriptor3 = aMessage.Ptr3();
	TUint32 accessPointId;
	TPckg<TUint32> apPackage(accessPointId);
	aMessage.ReadL(clientsDescriptor3, apPackage,0);

	ConnectToPresenceServer(username,password,server,accessPointId);
}

void CContextServer::NotifySessions(TEvent aEvent)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("NotifySessions"));

	CContextServerSession* session=0;

	iSessionIter.SetToFirst();
	while( (session = reinterpret_cast<CContextServerSession*>(iSessionIter++)) ) {
		session->NotifyEvent(aEvent);
	}
}

void CContextServer::SuspendConnection()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("SuspendConnection"));

	//NotifySessions(EDisconnected);
	iJabberClient->Disconnect(ETrue);
}

void CContextServer::ResumeConnection()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("ResumeConnection"));

	iJabberClient->ConnectL(iUsername,iPassword,iServer,iAccessPoint);
}

CPresenceInfo *  CContextServer::GetPresenceInfo()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("GetPresenceInfo"));

	return iPresenceInfo;
}

CJabber * CContextServer::GetJabberClient()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("GetJabberClient"));

	return iJabberClient;
}


// from MConnectionOpenerObserver --------------------------------------------------------------------
/*void CContextServer::success(CBase* source)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("success"));

	iJabberClient->ConnectL(iUsername,iPassword,iServer);
}

void CContextServer::error(CBase* source, TInt code, const TDesC& reason)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("error"));

	Log(reason);
}

void CContextServer::info(CBase* source, const TDesC& msg)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("info"));

	Log(msg);
}*/

void CContextServer::NotifyJabberStatus(TInt st)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("NotifyJabberStatus"));

	switch (st)
	{
	case MJabberObserver::EJabberConnected:
                NotifySessions(EConnected);
		break;
	case MJabberObserver::EJabberDisconnected:
		NotifySessions(EDisconnected);
		break;
	case MJabberObserver::EIdentificationFailed:
		{
			CContextServerSession* session=0;
			iSessionIter.SetToFirst();
			while( (session = reinterpret_cast<CContextServerSession*>(iSessionIter++)) ) {
				session->ReportError(ECSIdentificationError, _L(""), _L(""));
			}
		}
		
	}
}

void CContextServer::NotifyNewPresenceInfo(const TDesC & from, const TDesC & info, 
					   const TTime& send_timestamp)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("NotifyNewPresenceInfo"));

#ifdef __WINS__
	RDebug::Print(from.Left(128));
	RDebug::Print(info.Left(128));

	//DEBUG
	TBuf<10> buf;
	send_timestamp.FormatL(buf, _L("%H%T%S"));
	RDebug::Print(buf);
#endif

	if (from.Compare(KThisUser)==0) {
		TInt slashpos;
		slashpos=iJabberClient->iFullNick.Locate('/');
		if (slashpos>0) {
			iPresenceInfo->Update(iJabberClient->iFullNick.Left(slashpos), 
				info, send_timestamp);
		}
	} else {
		iPresenceInfo->Update(from,info,send_timestamp);
	}

	NotifySessions(ENewPresenceInfo);
}
/*
void CContextServer::ReportError(MJabberObserver::TErrorType aErrorType, const TDesC & aErrorCode, const TDesC & aErrorValue)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("ReportError"));

	CContextServerSession* session=0;
	
	iSessionIter.SetToFirst();
	switch (aErrorType)
	{
		case EIdentificationFailed:
			while( (session = reinterpret_cast<CContextServerSession*>(iSessionIter++)) ) {
				session->ReportError(EIdentificationError, aErrorCode, aErrorValue);
			}
			break;
		default:
			while( (session = reinterpret_cast<CContextServerSession*>(iSessionIter++)) ) {
				session->ReportError(ERequestCompleted, aErrorCode, aErrorValue);
			}
			break;
	}
		

	
	ERequestCompleted = 1,
	EIdentificationError = -1,
	EServerUnreachable = -2,
	EBufferTooSmall = -3,
	EContextServerTerminated = -4

	
}
*/


void CContextServer::CancelRequest(const RMessage &aMessage)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("CancelRequest"));

	switch(aMessage.Function()) {
	case EConnectToPresenceServer:
	case EResumeConnection:
		iJabberClient->Disconnect(ETrue);
		break;
	}
}

CMessageHolder  * CContextServer::GetMessageHolder()
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("GetMessageHolder"));

	return iMessageHolder;
}

void CContextServer::NotifyNewMessage(const TDesC & from, const TDesC& subject, const TDesC & message)
{
	CALLSTACKITEM_N(_CL("CContextServer"), _CL("NotifyNewMessage"));

	RDebug::Print(_L("NewMessage"));
	RDebug::Print(from.Left(128));
	RDebug::Print(subject.Left(128));
	RDebug::Print(message.Left(128));
	iMessageHolder->AppendMessageL(from, subject, message);
	NotifySessions(ENewMessage);
}


///
/*
void CContextServer::NotifyNewHeader(const CHttpHeader &aHeader)
{
	RDebug::Print(_L("NotifyNewHeader!\n"));
	RDebug::Print(_L("%d"), aHeader.iChunkStart);
	RDebug::Print(_L("%d"), aHeader.iChunkEnd);
	RDebug::Print(_L("%d"), aHeader.iSize);
}

void CContextServer::NotifyNewBody(const TDesC8 &data)
{
	RDebug::Print(_L("NotifyNewBody!\n"));

}*/
