/*
 * $Id: CodeChecker.h,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef CODECHECKER_H
#define CODECHECKER_H

// INCLUDES

#include <e32base.h>
#include "BigInteger.h"

// FORWARD DECLARATIONS

class CCodeInfo;

// CLASS DECLARATION

/**
 * The TCodeChecker class implements a linear (83,76,3) decoder.
 * It corrects a single bit error in an 83-bit code word x using the
 * parity check matrix H in the following way:<br><br>
 *
 * Hx<sup>T</sup> = s<sup>T</sup><br><br>
 *
 * H [7x83], x<sup>T</sup> [83x1], s<sup>T</sup> [7x1],
 * H = (A<sup>T</sup> | I<sub>7</sub>)<br>
 *
 */
class CCodeChecker : public CBase {

public:

	CCodeChecker();
	void ConstructL();
	virtual ~CCodeChecker();
	
	/**
	 * Determine the bit error position of a code word.
	 * @param code	the code word to be checked
	 * @return error position [0..82] or -1 if no error
	 */
	TInt ErrorPosition(CBigInteger* code) const;

	/**
	 * Decode the given code word.
	 * @param code the code word to decode
	 */
	void Decode(CBigInteger* code);

	/**
	 * Check the given code for errors and update argument with decoded value.
	 * @param code the code word to check and decode if correct
	 */
	TBool DoErrorCheck(CCodeInfo* ci);

private:

	/** Parity check matrix. */
	static const TUint32 H[7][3];

	CBigInteger* mask1;
	CBigInteger* mask2;
	CBigInteger* mask3;
	CBigInteger* tmp1;
	CBigInteger* tmp2;

	/**
	 * Determine the bit error position of a code word.
	 * @param code	the code word to be checked
	 * @return error position [0..82] or -1 if no error
	 */
	TInt ErrorPosition(TUint32* code) const;

};

#endif

// End of File
