/*
 * $Id: RecognitionAlgorithm.h,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef RECOGNITIONALGORITHM_H
#define RECOGNITIONALGORITHM_H

// INCLUDES

#include <w32std.h>  // for CWsBitmap
#include "CodeInfo.h"
#include "ClusterInfo.h"

// FORWARD DECLARATIONS

class TClusterInfo;
class CFbsBitmap;

// CLASS DECLARATION

/**
 * Visual code recognizer.
 */
class CRecognitionAlgorithm : public CBase {

public:

	CRecognitionAlgorithm();
	void ConstructL();
	~CRecognitionAlgorithm();

	RPointerArray<CCodeInfo>* RecognizeL(const CFbsBitmap* aBmp);
	RPointerArray<CCodeInfo>* RecognizeL(const CFbsBitmap* aBmp, const TRect& aArea);

	inline RPointerArray<CCodeInfo>* GetCodes() { return codes; };
	void ClearL();

private:

	void CorrectBarrelDistortion();
	void DoThresholdingL();
	inline void ProcessForegroundPixelL(TInt pIndex, TInt top, TInt left);
	void FindClustersL();
	void DoRecognitionL();

	TClusterInfo* SearchCorner(TInt x, TInt y, const TClusterInfo& kernel, TInt max_s, TReal maxcorr) const;
	TBool IsLongCorner(const TClusterInfo& cl, const TClusterInfo& kernel) const;
	TBool IsCorner(const TClusterInfo& cl, const TClusterInfo& kernel) const;
	TBool IsKernel(const TClusterInfo& cl) const;

	/** Squared distance of (x1,y1) and (x2,y2). */
	inline TReal D2(TInt x1, TInt y1, TInt x2, TInt y2) const {
		TInt x = x1 - x2;
		TInt y = y1 - y2;
		return x * x + y * y;
	}

/**************************************************************************/

private:

	// CONSTANTS

	// these are all important parameters to be set for the recognition

	/**
	 * Used to correct the "erosion" effect (stretch for MINOR axis).
	 * Find good constant -> 1.02 seems OK (the effect is not too strong though)
	 */
	static const TReal stretch;

	/**
	 *  Number of units which are allowed for searching the corners.
	 * (units = max. distance from estimated position)
	 */
	static const TReal units;

	/**
	 * Number of units that the corners may be corrected.
	 * If the jump would be larger, no correction is applied.
	 */
	static const TReal maxcorrect;

	/** Moment range for major guide bar. */
	static const TReal majorbar_min;
	static const TReal majorbar_max;

	/** Moment range for minor guide bar. */
	static const TReal minorbar_min;
	static const TReal minorbar_max;

	/** Moment range for corner stones (other than guide bars). */
	static const TReal corner_min;
	static const TReal corner_max;

	/** Most extreme ratio allowed for left and top edge. */
	static const TReal minratio;

	/** Set this in DEGREES. */
	static const TReal anglediff;

	/** Range of angles allowed for top left corner (undistorted: 90). */
	static const TReal minangle;
	static const TReal maxangle;

/**************************************************************************/

private:

	// MEMBER VARIABLES
	
	/** Image to find visual codes in. */
	const CFbsBitmap* iBmp;

	/** Image size. */
	TInt w, h;

	TRect iArea;
	
	/** Pixel data. Contains the cluster label of each pixel. */
	TInt* p;
	TInt pSize;

	/** Gray values of the previous scan line. */
	TInt* prev_gn;
	TInt prev_gnSize;

	/** A single horizontal line of image pixels. */
	TUint8* scanline;
	TInt scanlineSize;

	/** Used to store the label equivalences (index 0 unused). */
	TInt* equals;
	TInt equalsSize;

	/** Clusters found. */
	TInt currentCluster;
	TClusterInfo* clusters;  
	TInt clusterCount;

	/** Codes found. */
	RPointerArray<CCodeInfo>* codes;

	/** Checks code validity. */
	CCodeChecker* codeChecker;

	/** Counter for debugging purposes. */
	TInt debugCounter;
	TReal debugTReal1, debugTReal2;

	/** For visualizing certain points during debugging. */
	RArray<TPoint> debugPoints;

};

#endif

// End of File
