/*
 * $Id: Warper.h,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef WARPER_H
#define WARPER_H

// INCLUDES

#include <e32base.h>
#include "RealPoint.h"

// CONSTANTS

// FORWARD DECLARATIONS

// CLASS DECLARATION

/**
 * Transforms image to code coordinates and vice versa. Implements a
 * projective transform (a homography) between the image plane and the
 * code plane. A unit in the code coordinate system is equivalent to
 * a single code bit element.
 */
class TWarper {

public:
	/**
	 * Create warper object with the given code coordinates and initialize
	 * the coefficients.
	 *
	 * @param x0 x-coordinate of upper left corner stone (code coordinate (0,0))
	 * @param y0 y-coordinate of upper left corner stone (code coordinate (0,0))
	 * @param x1 x-coordinate of upper right corner stone (code coordinate (10,0))
	 * @param y1 y-coordinate of upper right corner stone (code coordinate (10,0))
	 * @param x2 x-coordinate of smaller guide bar (code coordinate (8,10))
	 * @param y2 y-coordinate of smaller guide bar (code coordinate (8,10))
	 * @param x3 x-coordinate of lower left corner stone (code coordinate (0,10))
	 * @param y3 y-coordinate of lower left corner stone (code coordinate (0,10))
	 * @param unitSquare if true, computes the coefficients for the unit square;
	 *                   if false, computes the coefficients for the code quadrangle
	 */
	TWarper(TInt x0, TInt y0, TInt x1, TInt y1, TInt x2, 
			TInt y2, TInt x3, TInt y3, TBool unitSquare);

	/**
	 * Copy constructor.
	 */
	TWarper(const TWarper& w);

	/**
	 * Warp from code coordinates to image coordinates.
	 * @param u x-coordinate in code coordinate system
	 * @param v y-coordinate in code coordinate system
	 * @return Corresponding image coordinates.
	 */
	TPoint Warp(TReal u, TReal v) const;

	/**
	 * Backwarp from image coordinates to code coordinates.
	 * @param x x-coordinate in image coordinate system
	 * @param y y-coordinate in image coordinate system
	 * @return Corresponding code coordinates.
	 */
	TPoint Backwarp(TReal x, TReal y) const;

	/**
	 * Precise warping from code coordinates to image coordinates.
	 * @param u x-coordinate in code coordinate system
	 * @param v y-coordinate in code coordinate system
	 * @return Corresponding image coordinates.
	 */
	TRealPoint WarpPrecise(TReal u, TReal v) const;

	/**
	 * Precise backwarping from image coordinates to code coordinates.
	 * @param x x-coordinate in image coordinate system
	 * @param y y-coordinate in image coordinate system
	 * @return Corresponding code coordinates.
	 */
	TRealPoint BackwarpPrecise(TReal x, TReal y) const;

	/**
	 * Append the state of this warper (i.e. the values of 
	 * the coefficients) to the buffer argument.
	 * @param aBuffer serialization buffer
	 */
	void Serialize(TDes8& aBuffer) const;

private: 
	
	/** Coefficients for warping from code to image coordinates. */
	TReal a, b, c, d, e, f, g, h;

	/** Coefficients for backwarping from image to code coordinates. */
	TReal A, B, C, D, E, F, G, H, I;

};

#endif

// End of File
