/*
 * $Id: BigInteger.cpp,v 1.5 2005/06/02 15:15:44 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "BigInteger.h"



CBigInteger::CBigInteger() 
{
	iBits = 0;
	iBuffer = NULL;
}



void CBigInteger::ConstructL(TInt aBits) 
{
	if (iBuffer != NULL) return;
	if (aBits <= 0) User::Leave(KErrArgument);
	iBits = aBits;
	iBuffer = HBufC8::NewMaxL((iBits + 7) >> 3);
	TPtr8 ptr = iBuffer->Des();
	ptr.FillZ();
}


//@todo: check if better call this SetL
void CBigInteger::ConstructL(const CBigInteger& bi)
{
	if (iBuffer != NULL) return;
	iBits = bi.iBits;
	iBuffer = HBufC8::NewMaxL((iBits + 7) >> 3);
	*iBuffer = (const HBufC8&)(*(bi.iBuffer));
}



void CBigInteger::Set(const TDesC& hexNum) 
{
	TPtr8 ptr = iBuffer->Des();
	TInt n = iBuffer->Length();
	TInt j = hexNum.Length() - 1;

	for (TInt i = 0; i < n && j >= 0; i++) {
		TInt x = Hex2Num(hexNum[j--]);
		if (j >= 0) x |= (Hex2Num(hexNum[j--]) << 4);
		ptr[i] = (TUint8)x;
	}
}



void CBigInteger::Set(TInt aI)
{
	TInt n1 = iBuffer->Length();
	TInt n2 = sizeof(TInt);
	TInt n = n1 < n2 ? n1 : n2;
	TPtr8 ptr = iBuffer->Des();

	TInt sign = aI < 0 ? 0xff : 0;
	TInt i;

	for (i = 0; i < n; i++) {
		ptr[i] = (TUint8)aI;
		aI >>= 8;
	}

	for (i = n; i < n1; i++) {
		ptr[i] = (TUint8)sign;
	}

	TInt bitPos = iBits % 8;
	if (bitPos != 0) {
		TUint8 mask = (TUint8)(~(0xff << bitPos));
		ptr[n1-1] &= mask;
	}
}



inline TInt CBigInteger::Hex2Num(TUint x) const {
	if (x >= '0' && x <= '9') return x - '0';
	if (x >= 'A' && x <= 'F') return x - 'A' + 0xa;
	if (x >= 'a' && x <= 'f') return x - 'a' + 0xa;
	return 0;
}



void CBigInteger::Copy(const CBigInteger* aSrc)
{
	TPtr8 ptrDst = iBuffer->Des();
	TPtr8 ptrSrc = aSrc->iBuffer->Des();
	ptrDst.FillZ();

	TInt nDst = iBuffer->Length();
	TInt nSrc = aSrc->iBuffer->Length();
	TInt n = nDst < nSrc ? nDst : nSrc;	

	for (TInt i = 0; i < n; i++) {
		ptrDst[i] = ptrSrc[i];
	}
}



CBigInteger::~CBigInteger()
{
	delete iBuffer;
}



void CBigInteger::ShiftLeft() 
{
	TInt n = iBuffer->Length();
	TPtr8 ptr = iBuffer->Des();

	for (TInt i = n-1; i > 0; i--) {
		ptr[i] = (TUint8)((ptr[i] << 1) | ((ptr[i-1] & 0x80) ? 1 : 0));
	}
	ptr[0] = (TUint8)(ptr[0] << 1);
}



void CBigInteger::ShiftRight() 
{
	TInt n = iBuffer->Length();
	TPtr8 ptr = iBuffer->Des();

	for (TInt i = 0; i < n-1; i++) {
		ptr[i] = (TUint8)(((ptr[i] >> 1) & 0x7f) | ((ptr[i+1] & 1) ? 0x80 : 0));
	}
	ptr[n-1] = (TUint8)((ptr[n-1] >> 1) & 0x7f);
}


	
void CBigInteger::ShiftLeft(TInt shift)
{
	if (shift <= 0) return;
	TInt s1 = shift >> 3;
	TInt s2 = shift % 8;

	TInt n = iBuffer->Length();
	TPtr8 ptr = iBuffer->Des();

	for (TInt i = n-1; i >= 0; i--) {
		TInt x = (i-s1 < 0) ? 0 : (ptr[i-s1] & 0xff) << s2;
		TInt y = (i-s1-1 < 0) ? 0 : (ptr[i-s1-1] & 0xff) >> (8 - s2);
		ptr[i] = (TUint8)(x | y);
	}
	
	/*
	for (TInt i = 0; i < shift; i++) {
		ShiftLeft();
	}
	*/
}



void CBigInteger::ShiftRight(TInt shift)
{
	if (shift <= 0) return;
	TInt s1 = shift >> 3;
	TInt s2 = shift % 8;

	TInt n = iBuffer->Length();
	TPtr8 ptr = iBuffer->Des();

	for (TInt i = 0; i < n; i++) {
		TInt x = (i+s1 >= n) ? 0 : (ptr[i+s1] & 0xff) >> s2;
		TInt y = (i+s1+1 >= n) ? 0 : (ptr[i+s1+1] & 0xff) << (8 - s2);
		ptr[i] = (TUint8)(x | y);
	}

	/*
	for (TInt i = 0; i < shift; i++) {
		ShiftRight();
	}
	*/
}



void CBigInteger::SetLSB() 
{
	TPtr8 ptr = iBuffer->Des();
	ptr[0] |= 1;
}



TInt CBigInteger::GetBit(TInt i) const 
{
	TPtr8 ptr = iBuffer->Des();
	TInt mask = 1 << (i % 8);
	return (ptr[i >> 3] & mask) ? 1 : 0;
}



void CBigInteger::ClearBit(TInt i) 
{
	TPtr8 ptr = iBuffer->Des();
	TInt mask = 0xff ^ (1 << (i % 8));
	ptr[i >> 3] = (TUint8)(ptr[i >> 3] & mask);
}



void CBigInteger::SetBit(TInt i) 
{
	TPtr8 ptr = iBuffer->Des();
	TInt mask = 1 << (i % 8);
	ptr[i >> 3] = (TUint8)(ptr[i >> 3] | mask);
}



void CBigInteger::And(const CBigInteger* aBigInt)
{
	TInt n1 = iBuffer->Length();
	TInt n2 = aBigInt->iBuffer->Length();
	TInt n = n1 < n2 ? n1 : n2;
	
	TPtr8 ptr1 = iBuffer->Des();
	TPtr8 ptr2 = aBigInt->iBuffer->Des();
	TInt i;

	for (i = 0; i < n; i++) {
		ptr1[i] &= ptr2[i];
	}
	if (n1 > n2) {
		for (; i < n1; i++) {
			ptr1[i] = 0;
		}
	}
}



void CBigInteger::Or(const CBigInteger* aBigInt)
{
	TInt n1 = iBuffer->Length();
	TInt n2 = aBigInt->iBuffer->Length();
	TInt n = n1 < n2 ? n1 : n2;
	
	TPtr8 ptr1 = iBuffer->Des();
	TPtr8 ptr2 = aBigInt->iBuffer->Des();

	for (TInt i = 0; i < n; i++) {
		ptr1[i] |= ptr2[i];
	}
}



void CBigInteger::Xor(const CBigInteger* aBigInt)
{
	TInt n1 = iBuffer->Length();
	TInt n2 = aBigInt->iBuffer->Length();
	TInt n = n1 < n2 ? n1 : n2;
	
	TPtr8 ptr1 = iBuffer->Des();
	TPtr8 ptr2 = aBigInt->iBuffer->Des();

	for (TInt i = 0; i < n; i++) {
		ptr1[i] ^= ptr2[i];
	}
}



void CBigInteger::Add(const CBigInteger* aBigInt)
{
	TInt n1 = iBuffer->Length();
	TInt n2 = aBigInt->iBuffer->Length();
	TInt n = n1 < n2 ? n1 : n2;

	TPtr8 ptr1 = iBuffer->Des();
	TPtr8 ptr2 = aBigInt->iBuffer->Des();

	TInt carry = 0;
	TInt bitPos = (aBigInt->iBits + 7) % 8;
	TUint8 mask = (TUint8)(1 << bitPos);
	TInt sign = (ptr2[n2-1] & mask) != 0 ? 0xff : 0;
	TInt i;

	for (i = 0; i < n; i++) {
		TInt sum = (ptr1[i] + ptr2[i] + carry) & 0x1ff;
		carry = sum >> 8;
		ptr1[i] = (TUint8)sum;
	}

	for (i = n; i < n1; i++) {
		TInt sum = (ptr1[i] + sign + carry) & 0x1ff;
		carry = sum >> 8;
		ptr1[i] = (TUint8)sum;
	}

	bitPos = iBits % 8;
	if (bitPos != 0) {
		TUint8 mask = (TUint8)(~(0xff << bitPos));
		ptr1[n1-1] &= mask;
	}
}



void CBigInteger::Add(TInt aI)
{
	TInt n1 = iBuffer->Length();
	TInt n2 = sizeof(TInt);
	TInt n = n1 < n2 ? n1 : n2;
	TPtr8 ptr = iBuffer->Des();

	TInt carry = 0;
	TInt sign = aI < 0 ? 0xff : 0;
	TInt i;

	for (i = 0; i < n; i++) {
		TUint sum = (ptr[i] + (aI & 0xff) + carry) & 0x1ff;
		ptr[i] = (TUint8)sum;
		carry = sum >> 8;
		aI >>= 8;
	}

	for (i = n; i < n1; i++) {
		TUint sum = (ptr[i] + sign + carry) & 0x1ff;
		carry = sum >> 8;
		ptr[i] = (TUint8)sum;
	}

	TInt bitPos = iBits % 8;
	if (bitPos != 0) {
		TUint8 mask = (TUint8)(~(0xff << bitPos));
		ptr[n1-1] &= mask;
	}
}



TBool CBigInteger::Equals(const CBigInteger* aBigInt) const
{
	TInt n1 = iBuffer->Length();
	TInt n2 = aBigInt->iBuffer->Length();
	TInt i = n1 - 1;

	const TUint8* ptr1 = iBuffer->Ptr();
	const TUint8* ptr2 = aBigInt->iBuffer->Ptr();

	if (n1 > n2) {
		for (i = n1 - 1; i >= n2; i--) {
			if (ptr1[i] != 0) return EFalse;
		}
	} else if (n2 > n1) {
		for (i = n2 - 1; i >= n1; i--) {
			if (ptr2[i] != 0) return EFalse;
		}
	}

	for (; i >= 0; i--) {
		if (ptr1[i] != ptr2[i]) return EFalse;
	}

	return ETrue;
}



const TUint8* CBigInteger::GetBuffer() const
{
	return iBuffer->Ptr();
}



EXPORT_C TInt CBigInteger::ToInt() const
{
	TInt result = 0;

	TInt n = iBuffer->Length();
	if (n > sizeof(TInt)) n = sizeof(TInt);
	const TUint8* ptr = iBuffer->Ptr();

	for (TInt i = n-1; i >= 0; i--) {
		result = (result << 8) | (ptr[i] & 0xff);
	}

	return result;
}


EXPORT_C TInt64 CBigInteger::ToInt64() const
{
	TUint low = 0, high = 0;

	TInt n = iBuffer->Length();
	if (n > 8) n = 8;
	const TUint8* ptr = iBuffer->Ptr();

	TInt i;

	for (i = n-1; i >= 4; i--) {
		high = (high << 8) | (ptr[i] & 0xff);
	}
	if (n > 4) n=4;
	for (i = n-1; i >= 0; i--) {
		low = (low << 8) | (ptr[i] & 0xff);
	}

	return TInt64(high, low);
}


EXPORT_C void CBigInteger::AppendToString(TDes& aString) const 
{
	aString.Append(_L("0x"));	

	TInt n = iBuffer->Length();

	for (TInt i = n-1; i >= 0; i--) {
		TPtr8 ptr = iBuffer->Des();
		aString.AppendNum((ptr[i] >> 4) & 0xf, EHex);
		aString.AppendNum(ptr[i] & 0xf, EHex);
	}
}



