/*
 * $Id: CodeInfo.cpp,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <e32math.h>
#include "CodeInfo.h"
#include "BigInteger.h"



CCodeInfo::CCodeInfo(
	TInt x1, TInt y1, TInt x2, TInt y2, 
	TInt x3, TInt y3, TInt x4, TInt y4,
	TInt x5, TInt y5, TInt iw, TInt ih, 
	CBigInteger* c, TWarper* w) 
	: 
	x1(x1), y1(y1), x2(x2), y2(y2), 
	x3(x3), y3(y3), x4(x4), y4(y4), 
	x5(x5), y5(y5), iw(iw), ih(ih)
{
	code = c;
	warper = w;
	errorCheckPassed = EFalse;
	iRotation = INVALID_ROTATION;
	iTilting.SetXY(0xffff, 0xffff);
}



CCodeInfo::CCodeInfo() 
	: 
	x1(0), y1(0), x2(0), y2(0), 
	x3(0), y3(0), x4(0), y4(0), 
	x5(0), y5(0), iw(0), ih(0)
{
	code = NULL;
	warper = NULL;
	errorCheckPassed = EFalse;
	iRotation = INVALID_ROTATION;
	iTilting.SetXY(INVALID_ROTATION, INVALID_ROTATION);
}



void CCodeInfo::ConstructL(const CCodeInfo& ci) {
	x1 = ci.x1; y1 = ci.y1;
	x2 = ci.x2; y2 = ci.y2;
	x3 = ci.x3; y3 = ci.y3;
	x4 = ci.x4; y4 = ci.y4;
	x5 = ci.x5; y5 = ci.y5;
	iw = ci.iw; ih = ci.ih;
	code = new (ELeave) CBigInteger();
	code->ConstructL(*ci.code);
	warper = new (ELeave) TWarper(*ci.warper);
	errorCheckPassed = ci.errorCheckPassed; 
	iRotation = ci.iRotation;
	iTilting = ci.iTilting;
}




CCodeInfo::~CCodeInfo() {
	delete code;
	delete warper;
}



TInt CCodeInfo::GetRotation()
{
	if (iRotation != INVALID_ROTATION) return iRotation;

	TPoint o = warper->Warp(0,0);
	TPoint x = warper->Warp(100,0);

	TReal alpha = 0.0;

	if (o.iY == x.iY && o.iX < x.iX) {
		alpha = 0.0;
	} else if (o.iX == x.iX) {
		if (x.iY < o.iY) {
			alpha = KPi / 2.0;
		} else {
			alpha = 3.0 * KPi / 2.0;
		}
	} else if (o.iX < x.iX) {
		if (o.iY > x.iY) {
			Math::ATan(alpha, ((TReal)o.iY-x.iY) / ((TReal)x.iX-o.iX));
		} else { // o.iY <= x.iY
			Math::ATan(alpha, ((TReal)x.iY-o.iY) / ((TReal)x.iX-o.iX));
			alpha = 2*KPi - alpha;
		}
	} else { // o.iX > x.iX
		if (o.iY > x.iY) {
			Math::ATan(alpha, ((TReal)o.iY-x.iY) / ((TReal)o.iX-x.iX));
			alpha = KPi - alpha;
		} else { // o.iY <= x.iY
			Math::ATan(alpha, ((TReal)x.iY-o.iY) / ((TReal)o.iX-x.iX));
			alpha = KPi + alpha;
		}
	}

	alpha *= KRadToDeg;
	return (TInt)alpha;
}



TPoint CCodeInfo::GetTilting() 
{
	if (iTilting.iX != INVALID_ROTATION) return iTilting;

	TRealPoint oi, oc, pi, pc;
	TReal li;
	TReal dl, dr, dt, db;

	TInt iw2 = iw >> 1;
	TInt ih2 = ih >> 1;

	oi.SetXY(iw2, ih2);
	oc = GetCodeCoordinatesPrecise(iw2, ih2);

	pi = GetImageCoordinatesPrecise(oc.iX - 1, oc.iY);
	li = Distance(oi, pi);
	pi.SetXY((pi.iX - oi.iX) / li, (pi.iY - oi.iY) / li);
	pc = GetCodeCoordinatesPrecise(iw2 + ih * pi.iX, ih2 + ih * pi.iY);
	dl = Distance(oc, pc);

	pc = GetCodeCoordinatesPrecise(iw2 - ih * pi.iX, ih2 - ih * pi.iY);
	dr = Distance(oc, pc);

	pi = GetImageCoordinatesPrecise(oc.iX, oc.iY - 1);
	li = Distance(oi, pi);
	pi.SetXY((pi.iX - oi.iX) / li, (pi.iY - oi.iY) / li);
	pc = GetCodeCoordinatesPrecise(iw2 + ih * pi.iX, ih2 + ih * pi.iY);
	dt = Distance(oc, pc);

	pc = GetCodeCoordinatesPrecise(iw2 - ih * pi.iX, ih2 - ih * pi.iY);
	db = Distance(oc, pc);

	TReal tiltingX = dl / dr;
	TReal tiltingY = dt / db;

	tiltingX = 1.64177 * (tiltingX - 1) / (tiltingX + 1);

	TReal angle;
	Math::ATan(angle, tiltingX);
	tiltingX = angle * KRadToDeg;

	tiltingY = 1.64177 * (tiltingY - 1) / (tiltingY + 1);
	Math::ATan(angle, tiltingY);
	tiltingY = angle * KRadToDeg;

	iTilting.SetXY((TInt)tiltingX, (TInt)tiltingY);
	return iTilting;
}



TReal CCodeInfo::Distance(const TRealPoint& p, const TRealPoint& q) 
{
	TReal dx = p.iX - q.iX;
	TReal dy = p.iY - q.iY;
	TReal sqrt;
	Math::Sqrt(sqrt, dx*dx + dy*dy);
	return sqrt;
}
