/*
 * $Id: VisualCodeSystem.cpp,v 1.3 2005/06/02 15:15:44 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "VisualCodeSystem.h"



CVisualCodeSystem::CVisualCodeSystem(MApp_context& aContext) : MContextBase(aContext)
{
	iPictureTaker = NULL;
	iRecognizer = NULL;
	iDoLQRecognition = FALSE;
}



void CVisualCodeSystem::ConstructL()
{
	if (iPictureTaker == NULL) {
		CPictureTaker* pt = new (ELeave) CPictureTaker(AppContext(), *this);
		CleanupStack::PushL(pt);
		pt->ConstructL();
		CleanupStack::Pop(pt);
		iPictureTaker = pt;
	}

	if (iRecognizer == NULL) {
		CRecognitionAlgorithm* ra = new (ELeave) CRecognitionAlgorithm();
		CleanupStack::PushL(ra);
		ra->ConstructL();
		CleanupStack::Pop(ra);
		iRecognizer = ra;
	}
}



CVisualCodeSystem::~CVisualCodeSystem() 
{
	iObservers.Close();
	delete iPictureTaker;
	delete iRecognizer;
}


//@todo: may leave!
void CVisualCodeSystem::Start(
	TBool aSendUpdates, 
	TBool aSendTranslation,
	TBool aSendRotation,
	TBool aDoLQRecognition
	/*, 
	const CCoeCtrl* aDrawTarget, 
	TBool aShowCross,
	TBool aShowMagnification,
	*/)
{
	iDoLQRecognition = aDoLQRecognition;
	iRecognizer->ClearL();
	iPictureTaker->StartL(aSendUpdates, aSendTranslation, aSendRotation);
}



void CVisualCodeSystem::AddObserver(const MVisualCodeObserver* aObserver)
{
	if (iObservers.Find(aObserver) == KErrNotFound) {
		iObservers.Append(aObserver);
	}	
}



void CVisualCodeSystem::RemoveObserver(const MVisualCodeObserver* aObserver)
{
	TInt i = iObservers.Find(aObserver);
	if (i != KErrNotFound) {
		iObservers.Remove(i);
	}	
}



void CVisualCodeSystem::PictureUpdateL(
		CFbsBitmap& aPicture, 
		const TPoint& aTranslation,
		TInt aMinDiffTranslation,
		TInt aRotation,
		TInt aMinDiffRotation)
{
	if (iDoLQRecognition) {
		RPointerArray<CCodeInfo>* codes;
		if (aTranslation.iX == 0 && aTranslation.iY == 0) {
			codes = iRecognizer->RecognizeL(&aPicture);
		} else {
			codes = GetCodes(); // use old code values
		}
		TInt n = iObservers.Count();
		for (TInt i = 0; i < n; i++) {
			iObservers[i]->PictureUpdate(aPicture, codes, 
			aTranslation, aMinDiffTranslation, 
			aRotation, aMinDiffRotation);
		}
	} else {
		TInt n = iObservers.Count();
		for (TInt i = 0; i < n; i++) {
			iObservers[i]->PictureUpdate(aPicture, NULL, 
				aTranslation, aMinDiffTranslation, 
				aRotation, aMinDiffRotation);
		}
	}
}



void CVisualCodeSystem::PictureTakenL(CFbsBitmap* aPicture)
{
	RPointerArray<CCodeInfo>* codes = iRecognizer->RecognizeL(aPicture);
	TInt n = iObservers.Count();
	for (TInt i = 0; i < n; i++) {
		iObservers[i]->PictureTaken(aPicture, codes);
	}
}



TInt CVisualCodeSystem::FindClosestCode(const TPoint& target) const
{
	if (GetCodes() == NULL) return -1;
	TInt n = GetCodes()->Count();
	if (n <= 0) return -1;

	TInt minDist = 0;
	TInt minI = -1;

	for (TInt i = 0; i < n; i++) {
		CCodeInfo* ci = (*GetCodes())[i];
		TPoint codeCenter = ci->GetImageCoordinates(TPoint(5,5));
		TInt dx = codeCenter.iX - target.iX;
		TInt dy = codeCenter.iY - target.iY;
		TInt dist = dx * dx + dy * dy;
		if (minI == -1 || dist < minDist) {
			minDist = dist;
			minI = i;
		}
	}

	return minI;
}