/*
 * $Id: Warper.cpp,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <e32math.h>
#include "Warper.h"



TWarper::TWarper(TInt x0, TInt y0, TInt x1, TInt y1, 
				 TInt x2, TInt y2, TInt x3, TInt y3, TBool unitSquare) 
{
	TReal dx1 = x1 - x2;
	TReal dy1 = y1 - y2;
	TReal dx2 = x3 - x2;
	TReal dy2 = y3 - y2;

	TReal sx, sy;
	if (unitSquare) {
		sx = x0 - x1 + x2 - x3; // for unit square
		sy = y0 - y1 + y2 - y3; // ((0,0), (1,0), (1,1), (0,1))
	} else {
		sx = 0.8*x0 - 0.8*x1 + x2 - x3; // for code quadrangle
		sy = 0.8*y0 - 0.8*y1 + y2 - y3; // ((0,0), (1,0), (0.8,1), (0,1))
	}

	TReal ghDenom = dx1*dy2 - dy1*dx2;
	if (ghDenom == 0.0) {
		g = 0.0;
		h = 0.0;
	} else {
		ghDenom = 1.0 / ghDenom;
		g = (sx*dy2 - sy*dx2) * ghDenom;
		h = (sy*dx1 - sx*dy1) * ghDenom;
	}

	a = x1 - x0 + g*x1;
	b = x3 - x0 + h*x3;
	c = x0;
	d = y1 - y0 + g*y1;
	e = y3 - y0 + h*y3;
	f = y0;

	A = e-f*h;
	B = c*h-b;
	C = b*f-c*e;
	D = f*g-d;
	E = a-c*g;
	F = c*d-a*f;
	G = d*h-e*g;
	H = b*g-a*h;
	I = a*e-b*d;
}



TWarper::TWarper(const TWarper& w) 
{
	a = w.a; b = w.b; c = w.c; d = w.d; 
	e = w.e; f = w.f; g = w.g; h = w.h;

	A = w.A; B = w.B; C = w.C; D = w.D; 
	E = w.E; F = w.F; G = w.G; H = w.H; 
	I = w.I;
}



TPoint TWarper::Warp(TReal u, TReal v) const {
	u *= 0.1; v *= 0.1; // 0.1: scale to code bit element size

	TReal denom = g*u + h*v + 1;
	if (denom == 0.0) return TPoint(0, 0);
	denom = 1.0 / denom;

	TReal xr = (a*u + b*v + c) * denom;
	Math::Round(xr, xr, 0);
	TInt x = (TInt)xr; //(TInt)Math.round((a*u + b*v + c) / denom);

	TReal yr = (d*u + e*v + f) * denom;
	Math::Round(yr, yr, 0);
	TInt y = (TInt)yr; //(TInt)Math.round((d*u + e*v + f) / denom);

	return TPoint(x, y);
}



TPoint TWarper::Backwarp(TReal x, TReal y) const {
    TReal denom = 0.1 * (G*x + H*y + I); // 0.1: scale to code bit element size
    if (denom == 0.0) return TPoint(0, 0);
	denom = 1.0 / denom;

	TReal ur = (A*x + B*y + C) * denom;
	Math::Round(ur, ur, 0);
    TInt u = (TInt)ur; //(TInt)Math.round((A*x + B*y + C) / denom);

	TReal vr = (D*x + E*y + F) * denom;
	Math::Round(vr, vr, 0);
    TInt v = (TInt)vr; // (TInt)Math.round((D*x + E*y + F) / denom);

    return TPoint(u, v);
}



TRealPoint TWarper::WarpPrecise(TReal u, TReal v) const {
	u *= 0.1; v *= 0.1; // 0.1: scale to code bit element size
	TReal denom = g*u + h*v + 1;
	if (denom == 0.0) return TRealPoint(0, 0);
	denom = 1.0 / denom;

	TReal xr = (a*u + b*v + c) * denom;
	TReal yr = (d*u + e*v + f) * denom;

	return TRealPoint(xr, yr);
}



TRealPoint TWarper::BackwarpPrecise(TReal x, TReal y) const {
	TReal denom = 0.1 * (G*x + H*y + I); // 0.1: scale to code bit element size
    if (denom == 0.0) return TRealPoint(0, 0);
	denom = 1.0 / denom;

	TReal ur = (A*x + B*y + C) * denom;
	TReal vr = (D*x + E*y + F) * denom;

    return TRealPoint(ur, vr);
}



void TWarper::Serialize(TDes8& aBuffer) const
{
	/** Coefficients for warping from code to image coordinates. */
	aBuffer.Append((TUint8*)&a, sizeof(TReal));
	aBuffer.Append((TUint8*)&b, sizeof(TReal));
	aBuffer.Append((TUint8*)&c, sizeof(TReal));
	aBuffer.Append((TUint8*)&d, sizeof(TReal));
	aBuffer.Append((TUint8*)&e, sizeof(TReal));
	aBuffer.Append((TUint8*)&f, sizeof(TReal));
	aBuffer.Append((TUint8*)&g, sizeof(TReal));
	aBuffer.Append((TUint8*)&h, sizeof(TReal));

	/** Coefficients for backwarping from image to code coordinates. */
	aBuffer.Append((TUint8*)&A, sizeof(TReal));
	aBuffer.Append((TUint8*)&B, sizeof(TReal));
	aBuffer.Append((TUint8*)&C, sizeof(TReal));
	aBuffer.Append((TUint8*)&D, sizeof(TReal));
	aBuffer.Append((TUint8*)&E, sizeof(TReal));
	aBuffer.Append((TUint8*)&F, sizeof(TReal));
	aBuffer.Append((TUint8*)&G, sizeof(TReal));
	aBuffer.Append((TUint8*)&H, sizeof(TReal));
}
