/*
 * $Id: PictureTaker.cpp,v 1.1.1.1 2004/08/06 10:53:19 mraento Exp $
 *
 * Visual Codes for Symbian OS
 * Copyright (C) 2004 Beat Gfeller, Michael Rohs (rohs@inf.ethz.ch)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

// INCLUDES

#include <e32math.h>
#include <w32std.h>

#include "PictureTaker.h"
#include "VisualCodeSystem.h"

// CONSTANTS

// FORWARD DECLARATIONS

// CLASS DEFINITION



CPictureTaker::CPictureTaker(CVisualCodeSystem& aObserver)
: CActive(EPriorityHigh), iObserver(aObserver)
{
	iCamera = NULL;
	iCameraIsOn = EFalse;
	iTakePictureStop = EFalse;

	iBmp = NULL;
	iBitmapInitialized = EFalse;

	iSendUpdates = TRUE;
	iSendTranslation = FALSE;
	iSendRotation = FALSE;

	iFrameCounter = 0;
	iTranslation.SetXY(0,0);
	iRotation = 0;
	iMinDiffTranslation = 0;
	iMinDiffRotation = 0;

	scanline = NULL;
	blocks1 = NULL;
	blocks2 = NULL;
	currentBlock = 0;

	debugTInt = 0;
}



void CPictureTaker::ConstructL() 
{
	if (iCamera != NULL) return;

	iCamera = new (ELeave) RCameraServ;
	User::LeaveIfError(iCamera->Connect());

	iBmp = new (ELeave) CWsBitmap;

	scanline = new (ELeave) TBuf8<3*160>;
	blocks1 = new (ELeave) TInt[7][10];
	blocks2 = new (ELeave) TInt[7][10];

	CActiveScheduler::Add(this);  
}



CPictureTaker::~CPictureTaker() 
{
	Cancel();

	delete iCamera;
	iCamera = NULL;

	delete iBmp;

	delete scanline;
	delete [] blocks1;
	delete [] blocks2;
}



void CPictureTaker::DoCancel() 
{
	CameraOffL();
	iCamera->Close();
	iCameraIsOn = EFalse;
	iTakePictureStop = EFalse;

	iBmp->Reset();
	iBitmapInitialized = EFalse;
}



void CPictureTaker::RunL() 
{
	if (iSendUpdates && iBitmapInitialized) {
		if (iSendTranslation || iSendRotation) {
			iFrameCounter++;
			if ((iFrameCounter % 3) == 0) {
				GetImageMovement(*iBmp);
			}
		} else {
			iObserver.PictureUpdateL(*iBmp, TPoint(0,0), 0, 0, 0);
		}
	}

	if (iTakePictureStop) { // take high-quality picture and exit
		iTakePictureStop = EFalse;

		User::LeaveIfError(iCamera->SetImageQuality(RCameraServ::EQualityHigh));

		iCamera->GetImage(iStatus, *iBmp);
		User::WaitForRequest(iStatus); 

		//mr CameraOff(); // @todo: switch camera off after some idle time...

		iObserver.PictureTakenL(iBmp);

	} else { // take low-quality picture and continue

		// in the next call to RunL(), the image will be initialised and can be drawn
		iBitmapInitialized = ETrue;

		iCamera->GetImage(iStatus, *iBmp);

		SetActive();
	}
}



void CPictureTaker::StartL(
	TBool aSendUpdates, 
	TBool aSendTranslation,
	TBool aSendRotation
) 
{
	if (IsActive()) { // if image taking is active, ignore new request
		iTakePictureStop = EFalse;
		return; 
	}

	iSendUpdates = aSendUpdates;
	iSendTranslation = aSendTranslation;
	iSendRotation = aSendRotation;

	User::LeaveIfError(iCamera->SetImageQuality(RCameraServ::EQualityLow));

	// @todo: check if camera is really on. it is automatically 
	//        turned off after a few minutes if it is not used
	CameraOnL();

	iBmp->Reset();
	iBitmapInitialized = EFalse;
	iTakePictureStop = EFalse;

	iFrameCounter = 0;

	iTranslation.SetXY(0,0);
	iRotation = 0;
	iMinDiffTranslation = 0;
	iMinDiffRotation = 0;

	currentBlock = 0;

	debugTInt = 0;
	
	RunL(); // start taking low quality pictures
}



void CPictureTaker::SetSendTranslation(TBool aSendTranslation)
{
	iSendTranslation = aSendTranslation;
	if (!iSendTranslation) {
		iTranslation.SetXY(0,0);
		iMinDiffTranslation = 0;
		if (!iSendRotation) {
			iFrameCounter = 0;
			currentBlock = 0;
		}
	}
}



void CPictureTaker::SetSendRotation(TBool aSendRotation)
{
	iSendRotation = aSendRotation;
	if (!iSendRotation) {
		iRotation = 0;
		iMinDiffRotation = 0;
		if (!iSendTranslation) {
			iFrameCounter = 0;
			currentBlock = 0;
		}
	}
}



void CPictureTaker::CameraOnL() 
{
	if (!iCameraIsOn) {
		TRequestStatus status(KErrNone);

		iCamera->TurnCameraOn(status);
		User::WaitForRequest(status);

		if(status.Int() != KErrNone) {
			iCamera->Close();
			User::Leave(status.Int());
		}

		User::LeaveIfError(iCamera->SetLightingConditions(RCameraServ::ELightingNormal));  
		iCameraIsOn = ETrue;
	}
}



void CPictureTaker::CameraOffL() 
{
	if (iCameraIsOn) {
		User::LeaveIfError(iCamera->TurnCameraOff());
		iCameraIsOn = EFalse;
	}
}



void CPictureTaker::StopL() 
{
	iTakePictureStop = ETrue;
}



TBool CPictureTaker::GetImageMovement(CFbsBitmap& iBmp) 
{
	TDisplayMode mode = EColor16M; //iBmp.DisplayMode();

	TInt w = 160; // 160/16 = 10 blocks //@todo: make dynamic
	TInt h = 112; // 112/16 = 7 blocks
	// TInt blockSize = 16;
	TInt bw = 10;
	TInt bh = 7;
	//TInt bx = 0;
	TInt by = 0;

	TInt (*b1)[10];
	TInt (*b2)[10];

	if ((currentBlock % 2) == 0) {
		b1 = blocks1;
		b2 = blocks2;
	} else {
		b1 = blocks2;
		b2 = blocks1;
	}
	currentBlock++;

	for (TInt j=0; j<bh; j++) {
		for (TInt i=0; i<bw; i++) {
			b1[j][i] = 0;
		}
	}
	// memset(block, 0, ...)?

	// compute grey values of blocks
#if 0
	for (TInt y=1; y<h; y+=4) {
		iBmp.GetScanLine(*scanline, TPoint(0,y), w, mode);
		by = y >> 4;
		for (TInt px=1, x=3; px<w; px+=4) {
			x++; // TInt blue = (*scanline)[x++];
			TInt green = (*scanline)[x++] & 0xff;
			TInt red = (*scanline)[x++] & 0xff;
			x += 9; // skip three pixels
			TInt grey = (red+green) >> 1;
			b1[by][px>>4] += grey;
		}
	}
#else
	for (TInt y=1; y<h; y+=2) {
		iBmp.GetScanLine(*scanline, TPoint(0,y), w, mode);
		by = y >> 4;
		for (TInt px=1, x=3; px<w; px+=2) {
			x++; // TInt blue = (*scanline)[x++];
			TInt green = (*scanline)[x++] & 0xff;
			TInt red = (*scanline)[x++] & 0xff;
			x += 3;
			TInt grey = (red+green) >> 1;
			b1[by][px>>4] += grey;
		}
	}
#endif

	if (currentBlock == 1) return FALSE; // other block not valid yet
	
	// compute differences, find minimal difference

	// translational differences

	if (iSendTranslation) {
		TInt minDiff = -1;
		TInt minDiffX = 0;
		TInt minDiffY = 0;

		for (TInt oy=-3; oy<=3; oy++) {
			for (TInt ox=-3; ox<=3; ox++) {
				TInt count = 0;
				TInt diff = 0;
				for (TInt y=0; y<bh; y++) {
					for (TInt x=0; x<bw; x++) {
						TInt x2 = x+ox;
						TInt y2 = y+oy;
						if (x2 >= 0 && x2 < bw && y2 >= 0 && y2 < bh) { 
							diff += Abs(b2[y][x] - b1[y2][x2]);
							count++;
						}
					}
				}
				diff /= count;
				if (minDiff < 0 || diff < minDiff) {
					minDiff = diff;
					minDiffX = ox;
					minDiffY = oy;
				}
			}
		}

		iTranslation.SetXY(-minDiffX, -minDiffY);
		iMinDiffTranslation = minDiff;

	} else {
		iTranslation.SetXY(0,0);
		iMinDiffTranslation = 0;
	}

	// rotational differences

	if (iSendRotation) {
		TInt minDiffRotation = -1;
		TInt minDiffAngle = 0;

		for (TInt alpha = -24; alpha <= 24; alpha += 6) {
			TInt count = 0;
			TInt diff = 0;
			
			TReal alphaRad = alpha * KPi / 180.0;
			TReal sinAlpha;
			TReal cosAlpha;
			Math::Sin(sinAlpha, alphaRad);
			Math::Cos(cosAlpha, alphaRad);
			
			for (TInt y=0; y<bh; y++) {
				for (TInt x=0; x<bw; x++) {
					TInt dx = x - bw/2;
					TInt dy = y - bh/2;

					TReal xr = (dx*cosAlpha - dy*sinAlpha);
					TReal yr = (dy*cosAlpha + dx*sinAlpha);
					Math::Round(xr, xr, 0);
					Math::Round(yr, yr, 0);
					TInt x2 = bw/2 + (TInt)xr;
					TInt y2 = bh/2 + (TInt)yr;

					if (x2 >= 0 && x2 < bw && y2 >= 0 && y2 < bh) { 
						diff += Abs(b2[y][x] - b1[y2][x2]);
						count++;
					}
				}
			}
			diff /= count;
			if (minDiffRotation < 0 || diff < minDiffRotation) {
				minDiffRotation = diff;
				minDiffAngle = alpha;
			}

		}

		iRotation = minDiffAngle;
		iMinDiffRotation = minDiffRotation;

	} else {
		iRotation = 0;
		iMinDiffRotation = 0;
	}

	iObserver.PictureUpdateL(iBmp, iTranslation, iMinDiffTranslation, iRotation, iMinDiffRotation);

	return TRUE;
}
