/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/




// INCLUDE FILES
#include "Context_logContainer.h"

#include <eiklabel.h>  // for example label control
#include <eikenv.h>

#include "cl_settings.h"
#include "basestack.h"
#include "symbian_auto_ptr.h"
#include <aknmessagequerydialog.h> 
#include <context_log.rsg>

#include "concretedata.h"
#include "csd_cell.h"
#include "csd_base.h"
#include "csd_bluetooth.h"
#include "csd_gps.h"
#include "csd_loca.h"
#include "csd_profile.h"
#include "csd_unread.h"
#include "csd_presence.h"

enum KEYCODES {
	JOY_LEFT = 0xF807,
	JOY_RIGHT = 0xF808,
	JOY_UP = 0xF809,
	JOY_DOWN = 0xF80A,
	JOY_CLICK = 0xF845,
	KEY_CALL = 0xF862,
	KEY_CANCEL = 0xF863
};

// ================= MEMBER FUNCTIONS =======================

void CContext_logContainer::register_source(const TDesC& name)
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("register_source"));

	if (! name.Compare(KCell) ) {
		register_source(KCellName);
	}

	int this_label_index=label_store->Count()+1;

	auto_ptr<CEikLabel> label(new (ELeave) CEikLabel);
	label->SetContainerWindowL( *this );
	label->SetFont(CEikonEnv::Static()->DenseFont());
	label->SetTextL( name );
	log_labels->AppendL(name);
	label->SetExtent( TPoint(10, this_label_index*12-8), TSize(50, 12));
	label_store->AppendL(label.get());
	label.release();
	
	label.reset(new (ELeave) CEikLabel);
	label->SetContainerWindowL( *this );
	label->SetFont(CEikonEnv::Static()->DenseFont());
	label->SetTextL( _L("") );
	label->SetExtent( TPoint(65, this_label_index*12-8), TSize(100, 12));
	value_store->AppendL(label.get());
	label.release();

	if (iCursorPos==-1) iCursorPos=0;

	DrawNow();
}

TKeyResponse CContext_logContainer::OfferKeyEventL(const TKeyEvent &aKeyEvent, TEventCode aType)
{
	if (iCursorPos==-1) return EKeyWasNotConsumed;

	if (aKeyEvent.iCode==JOY_CLICK) {
		CEikLabel* label=value_store->At(iCursorPos);
		const TDesC *text=label->Text();
		HBufC* buf=HBufC::NewLC(text->Length());
		*buf=*text;

		label=label_store->At(iCursorPos);
		text=label->Text();
		HBufC* caption=HBufC::NewLC(text->Length());
		*caption=*text;

		CAknMessageQueryDialog *note=CAknMessageQueryDialog::NewL(*buf);
		note->SetHeaderText(*caption);
		note->ExecuteLD(R_LOGVIEW_EVENT_DIALOG);
		CleanupStack::PopAndDestroy(2);

	} else if (aKeyEvent.iCode==JOY_UP) {
		if (iCursorPos>0) --iCursorPos;
		DrawNow();
	} else if (aKeyEvent.iCode==JOY_DOWN ) {
		if (iCursorPos < value_store->Count()-1 ) ++iCursorPos;
		DrawNow();
	} else {
		return EKeyWasNotConsumed;
	}
	return EKeyWasConsumed;

}

void delete_bufc(void* data)
{
	HBufC* p;
	p=(HBufC*)data;
	delete p;
}

CContext_logContainer::CContext_logContainer(MApp_context& Context) : MContextBase(Context),
	iCursorPos(-1)
{

}

void CContext_logContainer::NewSensorEventL(const TTupleName& , const TDesC& , const CBBSensorEvent& aEvent)
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("NewSensorEventL"));

	const TDesC& name=aEvent.Name();
	TBuf<100> value; 
	if (aEvent.iData()) {
		const MBBData* d=aEvent.iData();
		TRAPD(err, d->IntoStringL(value) );
	}

	CEikLabel* label=find_value_label(name);
	if (!label) {
		register_source(name);
		label=find_value_label(name);
		if (!label) return;
	}
	if (! name.Compare(KCell) && aEvent.iPriority()==CBBSensorEvent::VALUE) {
		const TBBCellId *cell=bb_cast<TBBCellId>(aEvent.iData());
		if (cell) {
			TInt id=cell->iMappedId();
			HBufC* user_given_name=0;
			user_given_name=(HBufC*)cellid_names->GetData(id);
			CEikLabel* name_label=find_value_label(KCellName);
			if (user_given_name) {
				current_name=*user_given_name;
				if (name_label) {
					name_label->SetTextL(*user_given_name);
					if (id==iCurrentLoc->CurrentBaseId()) 
						name_label->SetUnderlining(ETrue);
					else
						name_label->SetUnderlining(EFalse);
				}
			} else {
				TBuf<200> tmp;
				if (current_name.Length()) {
					tmp.Format(_L("last: %S"), &current_name);
				}
				if (name_label) name_label->SetTextL(tmp);
				name_label->SetUnderlining(EFalse);
			}
		}
	}
	if (! name.Compare(KBase) && aEvent.iPriority()==CBBSensorEvent::VALUE) {
#ifdef __WINS__
		TBuf<20> t;
		_LIT(KFormatTxt,"%04d%02d%02dT%02d%02d%02d ");
		TDateTime dt;
		dt=aEvent.iStamp().DateTime();
		t.Format(KFormatTxt, dt.Year(), (TInt)dt.Month()+1, (TInt)dt.Day()+1,
			dt.Hour(), dt.Minute(), dt.Second());
		RDebug::Print(t); RDebug::Print(value);
#endif
		if (value.Length()==0 && !(label->Text()->Left(5).Compare(_L("last:")))) return;
	}
	label->SetTextL(value);
	DrawNow();
}

CEikLabel*	CContext_logContainer::find_value_label(const TDesC& name)
{
	TInt pos;
	TInt ret=log_labels->Find(name, pos);
	if (ret==0) 
		return value_store->At(pos);
	return 0;
}

void CContext_logContainer::add_cellid_name(const TBBCellId& Cell, const TDesC& name)
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("add_cellid_name"));

	HBufC* n=HBufC::NewL(name.Length());
	CleanupStack::PushL(n);
	*n=name;
	TInt id=Cell.iMappedId();
	cellid_names->AddDataL(id, (void*)n, true); 
	CleanupStack::Pop(); // n

	CEikLabel* label=find_value_label(KCellName);
	CEikLabel* id_label=find_value_label(KCell);
	TBuf<100> cellid; TRAPD(err, Cell.IntoStringL(cellid));
	if (label && id_label && (! id_label->Text()->Compare(cellid))) {
		label->SetTextL(name);
		DrawNow();
	}
	label=find_value_label(KBase);
	if (id==iCurrentLoc->CurrentBaseId() && label) {
		label->SetTextL(name);
		DrawNow();
	}
	if (id==iCurrentLoc->CurrentBaseId()) {
		TTime now=GetTime();
		iCurrentLoc->now_at_location(&Cell, id, true, false, now);
	}
}

bool CContext_logContainer::is_named(const TInt id)
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("is_named"));

	if (cellid_names->GetData(id)) return true;
	return false;
}


CCircularLog*	CContext_logContainer::BaseLog()
{
	return iCurrentLoc->BaseLog();
}

void CContext_logContainer::ConstructL(const TRect& aRect)
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("ConstructL"));

	Mlogger::ConstructL(AppContextAccess());

	cellid_names=CGenericIntMap::NewL();
	cellid_names->SetDeletor(delete_bufc);

	iCurrentLoc=CCurrentLoc::NewL(AppContext(), cellid_names);

	SubscribeL(KCellIdTuple);
	SubscribeL(KBaseTuple);
	SubscribeL(KProfileTuple);
	SubscribeL(KBluetoothTuple);
	SubscribeL(KOwnBluetoothTuple);
	SubscribeL(KGpsTuple);
	SubscribeL(KLocaMessageStatusTuple);
	SubscribeL(KAlarmTuple);
	SubscribeL(KUnreadTuple);

	CreateWindowL();
	
	label_store=new (ELeave) CArrayFixFlat<CEikLabel*>(1);
	value_store=new (ELeave) CArrayFixFlat<CEikLabel*>(1);

	log_labels=new (ELeave) CDesCArrayFlat(8);
	
	status_label=new (ELeave) CEikLabel;
	status_label->SetContainerWindowL(*this);
	status_label->SetFont(CEikonEnv::Static()->DenseFont());
	status_label->SetTextL( _L("starting") );
	status_label->SetExtent(TPoint(10, 116), status_label->MinimumSize());
	
	err_label=new (ELeave) CEikLabel;
	err_label->SetContainerWindowL(*this);
	err_label->SetFont(CEikonEnv::Static()->DenseFont());
	err_label->SetTextL( _L("    ") );
	err_label->SetExtent(TPoint(10, 128), err_label->MinimumSize());

	SetRect(aRect);
	ActivateL();
}

// Destructor
CContext_logContainer::~CContext_logContainer()
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("~CContext_logContainer"));

	delete iLabel;
	delete status_label;
	delete err_label;
	if (label_store) {
		TInt count=label_store->Count();
		for (int i=count-1; i>=0; i--)  {
			delete (*label_store)[i];
		}
		delete label_store;
	}
	if (value_store) {
		TInt count=value_store->Count();
		for (int i=count-1; i>=0; i--)  {
			delete (*value_store)[i];
		}
		delete value_store;
	}
	delete log_labels;
	delete cellid_names;
	delete iCurrentLoc;
}

void CContext_logContainer::set_status(const TDesC& status)
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("set_status"));

	status_label->SetTextL(status);
	status_label->SetExtent(TPoint(10, 116), status_label->MinimumSize());
	DrawNow();
}

void CContext_logContainer::set_error(const TDesC& err)
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("set_error"));

	err_label->SetTextL(err);
	err_label->SetExtent(TPoint(10, 128), err_label->MinimumSize());
	DrawNow();
}

// ---------------------------------------------------------
// CContext_logContainer::SizeChanged()
// Called by framework when the view size is changed
// ---------------------------------------------------------
//
void CContext_logContainer::SizeChanged()
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("SizeChanged"));

	status_label->SetExtent(TPoint(10, 116), status_label->MinimumSize());
	err_label->SetExtent(TPoint(10, 128), status_label->MinimumSize());
	
	if (!label_store) return;
	
	for (int i=0; i<label_store->Count(); i++)  {
		(*label_store)[i]->SetExtent( TPoint(10, (i)*12-8), TSize(50, 12) );
		(*value_store)[i]->SetExtent( TPoint(65, (i)*12-8), 
			(*value_store)[i]->MinimumSize() );
	}	
}

// ---------------------------------------------------------
// CContext_logContainer::CountComponentControls() const
// ---------------------------------------------------------
//
TInt CContext_logContainer::CountComponentControls() const
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("CountComponentControls"));

	if (!label_store) return 2;
	
	return label_store->Count()+value_store->Count()+2; // return nbr of controls inside this container
}

// ---------------------------------------------------------
// CContext_logContainer::ComponentControl(TInt aIndex) const
// ---------------------------------------------------------
//
CCoeControl* CContext_logContainer::ComponentControl(TInt aIndex) const
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("ComponentControl"));

	if (aIndex==0) {
		return status_label;
	}
	--aIndex;

	if (aIndex==0) {
		return err_label;
	} else if (label_store && aIndex>0 && aIndex <= label_store->Count()) {
		return (*label_store)[aIndex-1];
	} else if (label_store && aIndex>label_store->Count() && aIndex <= 
		label_store->Count()+value_store->Count()) {
		return (*value_store)[aIndex-label_store->Count()-1];
	} else {
		return NULL;
	}
}


// ---------------------------------------------------------
// CContext_logContainer::Draw(const TRect& aRect) const
// ---------------------------------------------------------
//
void CContext_logContainer::Draw(const TRect& aRect) const
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("Draw"));

	CWindowGc& gc = SystemGc();
	// TODO: Add your drawing code here
	// example code...
	gc.SetPenStyle(CGraphicsContext::ENullPen);
	gc.SetBrushColor(KRgbGray);
	gc.SetBrushStyle(CGraphicsContext::ESolidBrush);
	gc.DrawRect(aRect);

	if (iCursorPos!=-1) {
		TRect r(TPoint(0, (iCursorPos+1)*12-8), TSize( Rect().Width(), 12));
		gc.SetBrushColor(KRgbWhite);
		gc.DrawRect(r);
	}

}

// ---------------------------------------------------------
// CContext_logContainer::HandleControlEventL(
//     CCoeControl* aControl,TCoeEvent aEventType)
// ---------------------------------------------------------
//
void CContext_logContainer::HandleControlEventL(
						CCoeControl* /*aControl*/,TCoeEvent /*aEventType*/)
{
	CALLSTACKITEM_N(_CL("CContext_logContainer"), _CL("HandleControlEventL"));

	// TODO: Add your control event handler code here
}

_LIT(CLASS_NAME, "CContext_logContainer");

CCurrentLoc* CContext_logContainer::CurrentLoc()
{
	return iCurrentLoc;
}
// End of File  
