#include "btlist.h"

#include "db.h"
#include "symbian_auto_ptr.h"
#include <bttypes.h>
#include "list.h"
#include <badesca.h>

class CBTDeviceListImpl : public CBTDeviceList, public MContextBase, public MDBStore {
public:
	virtual ~CBTDeviceListImpl();
private:
	CBTDeviceListImpl(MApp_context& Context, RDbDatabase& Db);
	void ConstructL(const TDesC& aTableName);

	virtual MDesCArray* NameArray();
	virtual MDesC8Array* AddrArray();
	
	void AddObserver(MListObserver* aObserver);
	void RemoveObserver(MListObserver* aObserver);

	TInt AddDeviceL(const TDesC& Name, const TDesC8& Address);
	void RemoveDeviceL(TInt Idx);
	void MoveUpL(TInt Idx);
	void MoveDownL(TInt Idx);

	TBool ContainsDevice(const TDesC8& Address);
	TInt  FindbyReadable(const TDesC& Readable);

	void ContentsChanged();

	friend class CBTDeviceList;

	CDesC16Array * iBTDevNames;
	CDesC16Array * iBTDevReadable;
	CDesC8Array *  iBTDevAddrs;
	CList<MListObserver*> *iObservers;
	TInt iMaxIndex;
};

CBTDeviceList* CBTDeviceList::NewL(MApp_context& Context, RDbDatabase& Db, const TDesC& aTableName)
{
	auto_ptr<CBTDeviceListImpl> ret(new (ELeave) CBTDeviceListImpl(Context, Db));
	ret->ConstructL(aTableName);
	return ret.release();
}

CBTDeviceList::~CBTDeviceList()
{
}

CBTDeviceListImpl::~CBTDeviceListImpl()
{
	delete iBTDevNames;
	delete iBTDevAddrs;
	delete iBTDevReadable;
	delete iObservers;
}

CBTDeviceListImpl::CBTDeviceListImpl(MApp_context& Context, RDbDatabase& Db) : MContextBase(Context),
	MDBStore(Db)
{
}

void CBTDeviceListImpl::ConstructL(const TDesC& aTableName)
{
	TInt cols[] = { EDbColInt32, EDbColText, EDbColText8, -1 };
	TInt idx[] = { 1, -1 };

	MDBStore::ConstructL(cols, idx, false, aTableName);

	iBTDevNames = new (ELeave) CDesC16ArrayFlat(8);
	iBTDevAddrs = new (ELeave) CDesC8ArrayFlat(8);
	iBTDevReadable = new (ELeave) CDesC16ArrayFlat(8);

	iObservers=CList<MListObserver*>::NewL();

	iMaxIndex=-1;

	while ( iTable.NextL() ) {
		TBuf<50> name; TBuf8<10> addr;
		iTable.GetL();
		iMaxIndex=iTable.ColInt32(1);
		name=iTable.ColDes16(2);
		addr=iTable.ColDes8(3);

		iBTDevNames->AppendL(name);
		iBTDevAddrs->AppendL(addr);

		if (addr.Length()==6) {
			TBTDevAddr a(addr);
			TBuf<20> readable;
			a.GetReadable(readable);
			iBTDevReadable->AppendL(readable);
		} else {
			iBTDevReadable->AppendL(_L(""));
		}

	}
}


MDesCArray* CBTDeviceListImpl::NameArray()
{
	return iBTDevNames;
}

MDesC8Array* CBTDeviceListImpl::AddrArray()
{
	return iBTDevAddrs;
}

void CBTDeviceListImpl::AddObserver(MListObserver* aObserver)
{
	iObservers->AppendL(aObserver);
}

void CBTDeviceListImpl::RemoveObserver(MListObserver* aObserver)
{
	CList<MListObserver*>::Node *i, *prev, *tmp;
	i=iObservers->iFirst; prev=0;
	while (i) {
		if (i->Item==aObserver) {
			if (prev) {
				prev->Next=i->Next;
			} else {
				iObservers->iFirst=i->Next;
			}
			if (i==iObservers->iCurrent) iObservers->iCurrent=prev;
			tmp=i;
			i=i->Next;
			delete tmp;
		} else {
			prev=i;
			i=i->Next;
		}
	}
}

TInt CBTDeviceListImpl::AddDeviceL(const TDesC& Name, const TDesC8& Address)
{
	if (ContainsDevice(Address)) return -1;

	iTable.InsertL();
	iTable.SetColL(1, ++iMaxIndex);
	iTable.SetColL(2, Name.Left(50));
	iTable.SetColL(3, Address);
	PutL();

	iBTDevNames->AppendL(Name);
	iBTDevAddrs->AppendL(Address);
	if (Address.Length()==6) {
		TBTDevAddr a(Address);
		TBuf<20> readable;
		a.GetReadable(readable);
		iBTDevReadable->AppendL(readable);
	} else {
		iBTDevReadable->AppendL(_L(""));
	}

	ContentsChanged();

	return iMaxIndex;
}

void CBTDeviceListImpl::RemoveDeviceL(TInt Idx)
{
	TDbSeekKey k(Idx);
	if (iTable.SeekL(k)) {
		iTable.DeleteL();
	}
	iBTDevNames->Delete(Idx);
	iBTDevNames->Compress();
	iBTDevAddrs->Delete(Idx);
	iBTDevAddrs->Compress();
	iBTDevReadable->Delete(Idx);
	iBTDevReadable->Compress();
	ContentsChanged();
}

void CBTDeviceListImpl::MoveUpL(TInt Idx)
{
	if (Idx==0) return;

	TDbSeekKey k(Idx);
	if (iTable.SeekL(k)) {
		iTable.UpdateL();
		iTable.SetColL(1, Idx-1);
		PutL();
		iTable.PreviousL();
		iTable.UpdateL();
		iTable.SetColL(1, Idx);
		PutL();
	} else {
		User::Leave(KErrNotFound);
	}

	iBTDevNames->InsertL( Idx-1, (*iBTDevNames)[Idx] );
        iBTDevNames->Delete(Idx+1);
	iBTDevNames->Compress();
	iBTDevAddrs->InsertL( Idx-1, (*iBTDevAddrs)[Idx] );
	iBTDevAddrs->Delete(Idx+1);
	iBTDevAddrs->Compress();

	iBTDevReadable->InsertL( Idx-1, (*iBTDevReadable)[Idx] );
	iBTDevReadable->Delete(Idx+1);
	iBTDevReadable->Compress();

	ContentsChanged();
}

void CBTDeviceListImpl::MoveDownL(TInt Idx)
{
	if (Idx==iMaxIndex) return;

	TDbSeekKey k(Idx);
	if (iTable.SeekL(k)) {
		iTable.UpdateL();
		iTable.SetColL(1, Idx+1);
		PutL();
		iTable.NextL();
		iTable.UpdateL();
		iTable.SetColL(1, Idx);
		PutL();
	} else {
		User::Leave(KErrNotFound);
	}

	iBTDevNames->InsertL( Idx, (*iBTDevNames)[Idx+1] );
        iBTDevNames->Delete(Idx+2);
	iBTDevNames->Compress();
	iBTDevAddrs->InsertL( Idx, (*iBTDevAddrs)[Idx+1] );
	iBTDevAddrs->Delete(Idx+2);
	iBTDevAddrs->Compress();

	iBTDevReadable->InsertL( Idx, (*iBTDevReadable)[Idx+1] );
	iBTDevReadable->Delete(Idx+2);
	iBTDevReadable->Compress();

	ContentsChanged();
}

TBool CBTDeviceListImpl::ContainsDevice(const TDesC8& Address)
{
	for (int i=0; i<iBTDevAddrs->Count(); i++) {
		if ( ! ((*iBTDevAddrs)[i].Compare(Address)) ) {
			return ETrue;
		}
	}
	return EFalse;
}

void CBTDeviceListImpl::ContentsChanged()
{
	CList<MListObserver*>::Node *i;
	i=iObservers->iFirst;
	while (i) {
		i->Item->ContentsChanged();
		i=i->Next;
	}
}

TInt  CBTDeviceListImpl::FindbyReadable(const TDesC& Readable)
{
	for (int i=0; i<iBTDevReadable->Count(); i++) {
		if ( ! ((*iBTDevReadable)[i].Compare(Readable)) ) {
			return i;
		}
	}
	return KErrNotFound;
}
