/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/

/*
 * Concepts
 * !Threading!
 * !Capturing keypresses!
 * !UI in EXE!
 * !Hiding a WindowGroup!
 */
#include <aknnotewrappers.h>
#include "keycapture.h"
#include "independent.h"
#include <apgtask.h> // Going background 
#include <bautils.h>
#include <w32std.h>
#include "screen.h"
#include <viewcli.h> //CVwsSessionWrapper
#include <apgwgnam.h>
#include "viewids.h"
#include "cl_settings.h"

#include "symbian_auto_ptr.h"
#include "raii_w32std.h"
#include "raii_e32std.h"
#include "raii_f32file.h"
#include "raii_array.h"

#include <hal.h>

#include <sysutil.h>

enum JOYSTICK_EVENTS {
	JOY_LEFT = 0xF807,
	JOY_RIGHT = 0xF808,
	JOY_UP = 0xF809,
	JOY_DOWN = 0xF80A,
	JOY_CLICK = EKeyDevice3
};

#include "context_uids.h"
const TUid KUidPhone = { 0x100058b3 };
const TUid KUidMenu = {  0x101f4cd2 };

#ifndef __WINS__
_LIT(contextbook_filen, "c:\\system\\apps\\contextbook\\contextbook.app");
_LIT(contextcontacts_filen, "c:\\system\\apps\\contextcontacts\\contextcontacts.app");
_LIT(contextcalllog_filen, "c:\\system\\apps\\contextcalllog\\contextcalllog.app");
#else
_LIT(contextcontacts_filen, "z:\\system\\apps\\contextcontacts\\contextcontacts.app");
_LIT(contextbook_filen, "z:\\system\\apps\\contextbook\\contextbook.app");
_LIT(contextcalllog_filen, "z:\\system\\apps\\contextcalllog\\contextcalllog.app");
#endif

#define DO_RIGHTKEY 1
#define DO_TIMER 1

const TUid KCameraUid = { 0x1000593F };
const TUid KCamera2Uid = { 0x101f857a}; //6630
const TUid KVideoUid = { 0x101fa14a }; 
const TUid KVideo2Uid = { 0x101f857A}; //6630

void capturekeyL(RAWindowGroup& wg, RAArray<TInt>& ids, TUint keycode, TUint scancode)
{
	TInt id=wg.CaptureKey(keycode, 0, 0);
	User::LeaveIfError(id);
	User::LeaveIfError(ids.Append(id));
	id=wg.CaptureKeyUpAndDowns(scancode, 0, 0);
	User::LeaveIfError(id);
	User::LeaveIfError(ids.Append(id));
}

void capture_keysL(RAWindowGroup& wg, RAArray<TInt>& ids, TBool aCaptureRightKey)
{
	if (ids.Count()>0) return;

	capturekeyL(wg, ids, EKeyLeftArrow, EStdKeyLeftArrow);
	capturekeyL(wg, ids, EKeyRightArrow, EStdKeyRightArrow);
	capturekeyL(wg, ids, JOY_CLICK, EStdKeyDevice3);
	capturekeyL(wg, ids, EKeyUpArrow, EStdKeyUpArrow);
	capturekeyL(wg, ids, EKeyDownArrow, EStdKeyDownArrow);
	capturekeyL(wg, ids, EKeyYes, EStdKeyYes);
	capturekeyL(wg, ids, EKeyNo, EStdKeyNo);

	if (aCaptureRightKey) capturekeyL(wg, ids, EKeyDevice1, EStdKeyDevice1);
}

void cancel_keysL(RAWindowGroup& wg, RAArray<TInt>& ids)
{
	int i=0;
	while (i<ids.Count()) {
		wg.CancelCaptureKey(ids[i++]);
		wg.CancelCaptureKeyUpAndDowns(ids[i++]);
	}
	ids.Reset();
}

void do_start_keycapture(TAny* aPtr)
{
	RAWsSession ws; ws.ConnectLA();

	worker_info *wi=(worker_info*)aPtr;
	TKeycaptureArgs *args=(TKeycaptureArgs*)wi->worker_args;

	TRequestStatus status;
	
	RAArray<TInt> ids;
	RAWindowGroup wg(ws); wg.ConstructLA((TUint32)&wg, EFalse);

	TBool aDoRightKey=EFalse;

#ifdef DO_RIGHTKEY
#  ifndef NO_PRESENCE
	aDoRightKey=args->right_softkey_mapped;
#  endif
#endif //DO_RIGHTKEY
	capture_keysL(wg, ids, aDoRightKey);

	bool book_exists=false;
	bool log_exists=false;
	TVwsViewId callid, recentid,context_logid;

	{
		RAFs fsSession; fsSession.ConnectLA();

		// ContextBook activation
		if (BaflUtils::FileExists(fsSession, contextbook_filen)) {
			callid=TVwsViewId(KUidcontextbook, TUid::Uid(1));
			recentid=TVwsViewId(KUidcontextbook, TUid::Uid(2));
			book_exists=true;
		} 
		// ContextCallLog activation
		if (BaflUtils::FileExists(fsSession, contextcalllog_filen)) {
			recentid = TVwsViewId(KUidContextCallLog, TUid::Uid(8));
			log_exists=true;
		} 
		// ContextContact activation
		if (BaflUtils::FileExists(fsSession, contextcontacts_filen)) {
			callid=TVwsViewId(KUidContextContacts, TUid::Uid(1));
			book_exists=true;
		}
		
	}

#ifdef DO_TIMER
	RATimer timer; timer.CreateLocalLA();
	TBool timer_is_active=EFalse;
	TRequestStatus timer_status=KRequestPending;
	TWsEvent prev_keypress;
	TTimeIntervalMicroSeconds32 key_repeat_rate(250*1000);
	TTimeIntervalMicroSeconds32 key_repeat_initial(500*1000);
#endif

	auto_ptr<screen> screenhelp(new (ELeave) screen);
	screenhelp->ConstructL(&ws);

	auto_ptr<CVwsSessionWrapper> vws(CVwsSessionWrapper::NewL());

	wg.EnableFocusChangeEvents();
	ws.EventReady(&status);

	wg.SetOrdinalPosition(-1);
	wg.EnableReceiptOfFocus(EFalse);
	auto_ptr<CApaWindowGroupName> wn(CApaWindowGroupName::NewL(ws));
	wn->SetHidden(ETrue);
	wn->SetWindowGroupName(wg);

	TBool is6630=EFalse;
	TInt mach;
	if ( HAL::Get(HALData::EMachineUid, mach)==KErrNone ) {
		if (mach==0x101FBB55) is6630=ETrue;
	}
	//TRequestStatus parent_thread_status;
	RAThread parent; parent.OpenLA(wi->parent_threadid);
	/*
		We'd like to actually react to parent thread death
		and try to log it, put once the main thread of a process
		is killed all other threads are too, that means this thread :-)
	*/

	TInt wgid=ws.GetFocusWindowGroup();
	auto_ptr<CApaWindowGroupName> gn(CApaWindowGroupName::NewL(ws, wgid));

	for(;;) {
		User::WaitForAnyRequest();
		
		if ( (*wi->do_stop)!=KRequestPending) {
			break;
		}
		
#ifdef DO_TIMER
		if (timer_status.Int()==KErrNone) {
			if (timer_is_active) {
				ws.SendEventToWindowGroup(wgid, prev_keypress);
				timer.After(timer_status, key_repeat_rate);
			}
		}
		if (timer_status.Int()==KErrCancel) {
			// skip
		}
#endif
		if (status.Int()==KErrNone) {
			TWsEvent e;
			ws.GetEvent(e);
			if (e.Type() != EEventFocusGroupChanged && *(args->iKeyStatus)==KRequestPending) {
				TRequestStatus *sp=args->iKeyStatus;
				parent.RequestComplete(sp, KErrNone);
			}
			
#ifdef DO_TIMER
			if (timer_is_active) {
				timer.Cancel();
				User::WaitForRequest(timer_status);
				timer_status=KRequestPending;
				timer_is_active=EFalse;
			}
#endif
			if (e.Type() == EEventFocusGroupChanged) {
				wgid=ws.GetFocusWindowGroup();
				gn.reset(CApaWindowGroupName::NewL(ws, wgid));
				RThread me;
				if (		gn->AppUid()==KCameraUid ||
						gn->AppUid()==KCamera2Uid ||
						gn->AppUid()==KVideoUid ||
						gn->AppUid()==KVideo2Uid ) {

					// we can't afford to do work
					// when the camera app is being used,
					// because it otherwise can fail with -9
					//
					me.SetPriority(EPriorityAbsoluteBackground);
					if (timer_is_active) {
						timer.Cancel();
						User::WaitForRequest(timer_status);
						timer_status=KRequestPending;
						timer_is_active=EFalse;
					}
					cancel_keysL(wg, ids);
				} else {
					me.SetPriority(EPriorityAbsoluteForeground);
					capture_keysL(wg, ids, aDoRightKey);
				}
				ws.EventReady(&status);
				continue;
			}

			TInt c;
			TKeyEvent* aKeyEvent=e.Key();
			c=aKeyEvent->iCode;

#ifdef DO_TIMER
			prev_keypress=e;
			if (e.Type()==EEventKey) {
				timer.After(timer_status, key_repeat_initial);
				timer_is_active=ETrue;
			} else {
				ws.SendEventToWindowGroup(wgid, e);
				ws.EventReady(&status);
				continue;
			}
#endif

			// There's no Phone app on the emulator, so
			// we test it with Menu; on the phone the joystick
			// is used for the menu so we cannot use it
#ifndef __WINS__
			if (!is6630 && 
				e.Type()==EEventKey && aKeyEvent->iRepeats==0 &&
				gn->AppUid()==KUidPhone && 
				! screenhelp->dialog_on_screen()) {
#else
			if (e.Type()==EEventKey && aKeyEvent->iRepeats==0 &&
				gn->AppUid()==KUidMenu && 
				! screenhelp->dialog_on_screen() /*&& book_exists*/) {
#endif				
				switch(aKeyEvent->iScanCode) {
#ifdef DO_RIGHTKEY
				case EStdKeyDevice1:
					vws->ActivateView(TVwsViewId(KUidcontext_log, KUserViewId), TUid::Null(),KNullDesC8);
					break;
#endif
				case EStdKeyDevice3:
					if (callid!=TVwsViewId())
						vws->ActivateView(callid,TUid::Null(),KNullDesC8);
					else
						ws.SendEventToWindowGroup(wgid, e);
					break;
				case EStdKeyYes:
					if (recentid!=TVwsViewId())
						vws->ActivateView(recentid,TUid::Null(),KNullDesC8);
					else
						ws.SendEventToWindowGroup(wgid, e);
					break;
				default:
					ws.SendEventToWindowGroup(wgid, e);
					break;
				}
			} else 	{
				// otherwise send event to foreground app
				ws.SendEventToWindowGroup(wgid, e);
			}
			ws.EventReady(&status);
		}
		if ( (*wi->do_stop)!=KRequestPending) {
			break;
		}
	}
	ws.EventReadyCancel();
#ifdef DO_TIMER
	if (timer_is_active) timer.Cancel();
#endif
}

TInt start_keycapture(TAny* aPtr)
{

        CTrapCleanup *cl;
        cl=CTrapCleanup::New();

        TInt err=0;
        TRAP(err,
                do_start_keycapture(aPtr));

	delete cl;

	TTimeIntervalMicroSeconds32 w(50*1000);
	User::After(w);
	worker_info* wi=(worker_info*)aPtr;
	wi->stopped(err);
        return err;
}
