#include "loca_logic.h"

#include "cbbsession.h"
#include "db.h"
#include "csd_loca.h"
#include <e32math.h>
#include "csd_event.h"
#include "cl_settings.h"
#include "i_logger.h"
#include "app_context_impl.h"
#include "reporting.h"

_LIT(KLocaLogic, "localogic");

struct TDevStats {
	TTime	iLastSeen; 
	TTime	iLastSeenContinuously; 
	TTime	iLastSeenContinuouslyBegin; 
	TTime	iPreviousSeen; 
	TInt	iFailureCount; 
	TInt	iLocalFailureCount;
	TInt	iSuccessCount; 
	TTime	iPreviousLocalSuccess; 
	TTime	iPreviousRemoteSuccess; 
	TTime	iPreviousLocalFailure; 
	TTime	iPreviousRemoteFailure;
};

class CDevStats : public CBase, public MContextBase, public MDBStore {
public:
	static CDevStats* NewL(MApp_context& aContext, RDbDatabase& aDb);

	void GetStatsL(const TBBBtDeviceInfo& aDevice,
		TDevStats& aStats);

	void SetStatsL(const TBBBtDeviceInfo& aDevice,
		const TDevStats& aStats);
private:
	enum TColumns {
		EDevAddr = 1,
		ELastSeen,
		EFailureCount,
		ESuccessCount,
		ELastSeenContinuously,
		ELastSeenContinuouslyBegin,
		EPreviousSeen,
		EPreviousLocalSuccess,
		EPreviousRemoteSuccess,
		EPreviousLocalFailure,
		EPreviousRemoteFailure,
		ELocalFailureCount,
	};

	TTime	GetColTime(TColumns aColumn);
	TInt	GetColInt(TColumns aColumn);
	TBool	SeekToDevL(const TBBBtDeviceInfo& aDevice);
	CDevStats(MApp_context& aContext, RDbDatabase& aDb);
	void ConstructL();
};

class CLocaRemoteEvents : public CBase, public MContextBase,
	public Mlogger {
public:
	static CLocaRemoteEvents* NewL(MApp_context& aContext,
		CLocaLogic* aLogic);
	~CLocaRemoteEvents() { }
private:
	CLocaRemoteEvents(MApp_context& aContext,
		CLocaLogic* aLogic);
	void ConstructL();

	virtual void NewValueL(TUint aId, const TTupleName& aName, const TDesC& aSubName, 
			const MBBData* aData);

	virtual void NewSensorEventL(const TTupleName& , 
		const TDesC& , const CBBSensorEvent& ) { }

	CLocaLogic*	iLogic;
};

CLocaRemoteEvents* CLocaRemoteEvents::NewL(MApp_context& aContext, CLocaLogic* aLogic)
{
	auto_ptr<CLocaRemoteEvents> ret(new (ELeave) CLocaRemoteEvents(aContext, aLogic));
	ret->ConstructL();
	return ret.release();
}

CLocaRemoteEvents::CLocaRemoteEvents(MApp_context& aContext, CLocaLogic* aLogic) :
	MContextBase(aContext), iLogic(aLogic) { }

void CLocaRemoteEvents::ConstructL()
{
	Mlogger::ConstructL(AppContextAccess());
	SubscribeL(KRemoteLocaLogicTuple);
}

void CLocaRemoteEvents::NewValueL(TUint aId, const TTupleName& , const TDesC& , 
		const MBBData* aData)
{
	const CBBSensorEvent* e=bb_cast<CBBSensorEvent>(aData);
	if (!e) return;
	const TBBLocaMsgStatus* s=bb_cast<TBBLocaMsgStatus>(e->iData());
	if (!s) return;

	TBBBtDeviceInfo dev(s->iRecipientAddress(), KNullDesC, 0, 0, 0);

	if (s->iSucceeded()) {
		iLogic->Success(dev, s->iMessageId(), s->iAtTime(), EFalse);
	} else {
		iLogic->Failed(dev, s->iMessageId(), s->iAtTime(),
			CLocaLogic::EUnknown, EFalse);
	}
	TRAPD(err, iBBSubSessionNotif->DeleteL(aId));
	if (err!=KErrNone) {
		err=KErrNone;
	}
}

class CLocaLogicImpl : public CLocaLogic, public MContextBase,
	public MSettingListener {
private:

	CLocaLogicImpl(MApp_context& Context, RDbDatabase& aDb);
	void ConstructL();
	~CLocaLogicImpl();

	// name and title max length 240
	virtual void GetMessage(const TBBBtDeviceInfo& aDevice,
		TInt	aEnvironmentCount,
		const TTime& aAtTime,
		TBool& doSend, 
		TInt& aMessageCode, TDes& aWithName,
		TDes& aWithTitle, HBufC8*& aBody);

	virtual void Failed(const TBBBtDeviceInfo& aDevice,
		TInt aMessageCode,
		const TTime& aAtTime,
		TSendFailure aErrorCode,
		TBool	aLocal);
	virtual void Success(const TBBBtDeviceInfo& aDevice,
		TInt aMessageCode,
		const TTime& aAtTime,
		TBool	aLocal);

	virtual void SettingChanged(TInt Setting);
	void GetSettingsL();

	RDbDatabase&	iDb;
	CDevStats*	iDevStats;

	TInt		iMaxFailures, iMaxSuccesses;

	TInt		iMessageIntervals[30];

	TBBLocaMsgStatus	iMsgStatus;
	CBBSensorEvent		iEvent;
	CBBSubSession*		iBBSubSession;
	CLocaRemoteEvents*	iRemote;

	friend CLocaLogic;
	friend auto_ptr<CLocaLogicImpl>;
};

CLocaLogic* CLocaLogic::NewL(MApp_context& Context, RDbDatabase& aDb)
{
	CALLSTACKITEM_N(_CL("CLocaLogic"), _CL("NewL"));


	auto_ptr<CLocaLogicImpl> ret(new (ELeave) CLocaLogicImpl(Context, aDb));
	ret->ConstructL();
	return ret.release();
}

CLocaLogicImpl::CLocaLogicImpl(MApp_context& Context, RDbDatabase& aDb) :
	MContextBase(Context), iDb(aDb), iEvent(KLocaMsgStatus, KLocaMessageStatusTuple, 
		Context.BBDataFactory()) { }

void CLocaLogicImpl::ConstructL()
{
	CALLSTACKITEM_N(_CL("CLocaLogicImpl"), _CL("ConstructL"));


	iDevStats=CDevStats::NewL(AppContext(), iDb);
	iRemote=CLocaRemoteEvents::NewL(AppContext(), this);

	iBBSubSession=BBSession()->CreateSubSessionL(0);

	iEvent.iData()=&iMsgStatus;
	iEvent.iData.SetOwnsValue(EFalse);
	iEvent.iPriority()=CBBSensorEvent::VALUE;

	GetSettingsL();

	int i;
	iMessageIntervals[0]=15;
	iMessageIntervals[1]=60;
	for (i=2; i<30; i++) {
		iMessageIntervals[i]=2;
	}
	Settings().NotifyOnChange(SETTING_LOCA_BLUEJACK_MAX_MESSAGES, this);
	Settings().NotifyOnChange(SETTING_LOCA_BLUEJACK_MAX_RETRIES, this);
}

void CLocaLogicImpl::SettingChanged(TInt )
{
	GetSettingsL();
}

void CLocaLogicImpl::GetSettingsL()
{
	iMaxFailures=100;
	iMaxSuccesses=30;
	Settings().GetSettingL(SETTING_LOCA_BLUEJACK_MAX_MESSAGES, iMaxSuccesses);
	Settings().GetSettingL(SETTING_LOCA_BLUEJACK_MAX_RETRIES, iMaxFailures);

	if (iMaxFailures < 0) iMaxFailures=0;
	if (iMaxSuccesses < 0) iMaxSuccesses=0;
	if (iMaxSuccesses>30) iMaxSuccesses=30;
}

CLocaLogicImpl::~CLocaLogicImpl()
{
	CALLSTACKITEM_N(_CL("CLocaLogicImpl"), _CL("~CLocaLogicImpl"));

	Settings().CancelNotifyOnChange(SETTING_LOCA_BLUEJACK_MAX_MESSAGES, this);
	Settings().CancelNotifyOnChange(SETTING_LOCA_BLUEJACK_MAX_RETRIES, this);

	delete iBBSubSession;
	delete iDevStats;
	delete iRemote;
}

void AppendTimeDifference(TDes8& aInto, TTime from, TTime now)
{
	CALLSTACKITEM_N(_CL("CLocaLogicImpl"), _CL("~CLocaLogicImpl"));


	if (from > now) {
		TTime s=from;
		from=now;
		now=s;
	}
	for(;;) {
		TTimeIntervalMinutes mins;
		if (now.MinutesFrom(from, mins)!=KErrNone) {
			mins=61;
		} 
		TTimeIntervalHours hours;
		if (now.HoursFrom(from, hours)!=KErrNone) {
			hours=25;
		} 
		TTimeIntervalDays days;
		days=now.DaysFrom(from);
		if (mins < TTimeIntervalMinutes(60)) {
			aInto.AppendNum(mins.Int());
			aInto.Append(_L8(" minutes "));
			break;
		} else if ( hours < TTimeIntervalDays(24)) {
			aInto.AppendNum(hours.Int());
			aInto.Append(_L8(" hours "));
			from+=TTimeIntervalHours(hours);
		} else {
			aInto.AppendNum(days.Int());
			aInto.Append(_L8(" days "));
			from+=TTimeIntervalDays(days);
		}
	}
}

void CLocaLogicImpl::GetMessage(const TBBBtDeviceInfo& aDevice,
	TInt	aEnvironmentCount,
	const TTime& aAtTime,
	TBool& doSend, 
	TInt& aMessageCode, TDes& aWithName,
	TDes& aWithTitle, HBufC8*& aBody) 
{
	CALLSTACKITEM_N(_CL("CLocaLogicImpl"), _CL("GetMessage"));

	TDevStats s;
	iDevStats->GetStatsL(aDevice, s);
	if (aAtTime > s.iLastSeen ) {
		s.iLastSeen=aAtTime;
		if (s.iLastSeenContinuously + TTimeIntervalMinutes(3) > aAtTime
			|| s.iLastSeenContinuously==TTime(0) ) {
			s.iLastSeenContinuously=aAtTime;
		} else {
			s.iPreviousSeen=s.iLastSeenContinuously;
			s.iLastSeenContinuouslyBegin=aAtTime;
			s.iLastSeenContinuously=aAtTime;
		}
		iDevStats->SetStatsL(aDevice, s);
	}

	TTime now; now=GetTime();
#ifndef __WINS__ // always 'send' on WINS for testing
	
	if (aAtTime+TTimeIntervalMinutes(1) < now) {
		// old data
		doSend=EFalse;
		return;
	}
	if (s.iFailureCount >= iMaxFailures || s.iSuccessCount >= iMaxSuccesses) {
		doSend=EFalse;
		return;
	}

	// phone or PDA
	if (aDevice.iMajorClass()==0x02 || (
		aDevice.iMajorClass()==0x01 && 
		aDevice.iMinorClass()>0x03 )) {
	} else {
		doSend=EFalse;
		return;
	}

	if (s.iSuccessCount==0 ||  s.iLocalFailureCount>0 ||
		(s.iPreviousSeen+TTimeIntervalMinutes(iMessageIntervals[s.iSuccessCount]) 
			< now) ) {
	} else {
		doSend=EFalse;
		return;
	}
	
#endif
	if (s.iSuccessCount >= iMaxSuccesses) s.iSuccessCount=iMaxSuccesses-1;

	doSend=ETrue;
	aMessageCode=s.iSuccessCount+1;

	if (!aBody) aBody=HBufC8::NewL(2048);
	else aBody->Des().Zero();

	if (s.iSuccessCount==0) {
		aWithName=_L("LOCA-IN-RCA");
		aBody->Des().Append(_L("We are experiencing trouble tracking you. Please wave your "));
		if (aDevice.iMajorClass()==0x02) {
			aWithTitle=_L("PUT YOUR PHONE IN THE AIR");
			aBody->Des().Append(_L("phone "));
		} else {
			aWithTitle=_L("PUT YOUR PDA IN THE AIR");
			aBody->Des().Append(_L("PDA "));
		}
		aBody->Des().Append(_L("in the air."));
	} else if (s.iSuccessCount==1) {
		aWithName=_L("LOCA-NEAR-YOU");
		aWithTitle=_L("STAND STILL");
		aBody->Des().Append(_L("We are experiencing trouble tracking you. Please stand absolutely still for 30 seconds."));
	} else {
		aWithName=_L("LOCA-SEES-YOU");
		aWithTitle=_L("YOU HAVE BEEN TRACKED");
		aBody->Des().Append(_L("Congratulations! You've been tracked at least 3 times. To see "
			L"what we've found out about you, go to "
			L"Theo Humphries at interaction design.\n"));
	}


	aBody->Des().Append(_L("\nLoca is an exercise in pervasive grassroots surveillance."));
	//aBody->Des()=_L8("test body");
	//aBody->Des().AppendNum(aMessageCode);


	TBuf<50> nodename;
	Settings().GetSettingL(SETTING_PUBLISH_AUTHOR, nodename);
	TBool seen_long=EFalse;
	if (s.iLastSeenContinuouslyBegin + TTimeIntervalMinutes(5) < aAtTime) {
		aBody->Des().Append(_L("\nHey, you've been here at "));
		aBody->Des().Append(nodename);
		aBody->Des().Append(_L(" for "));
		TPtr8 p(aBody->Des());
		AppendTimeDifference(p, s.iLastSeenContinuouslyBegin, now);
		aBody->Des().Append(_L("."));
	}

	if ( s.iPreviousSeen != TTime(0) ) {
		aBody->Des().Append(_L("\nWe last saw you here "));
		if (!seen_long) {
			aBody->Des().Append(nodename);
			aBody->Des().Append(_L(" "));
		}
		TPtr8 p(aBody->Des());
		AppendTimeDifference(p, s.iPreviousSeen, now);
		aBody->Des().Append(_L8("ago."));
	} else if (!seen_long) {
		aBody->Des().Append(_L("\nWe've not seen you here at "));
		aBody->Des().Append(nodename);
		aBody->Des().Append(_L(" before!"));
	}

	if (aEnvironmentCount > 1) {
		aBody->Des().Append(_L("\n"));
		aBody->Des().AppendNum(aEnvironmentCount -1 );
		aBody->Des().Append(_L(" other Bluetooth devices know you are here."));
	}

	if (s.iSuccessCount < 3) {
	aBody->Des().Append(_L("\nIf you want to know what this is all about, see "
		L"Theo Humphries at interaction design or http://www.loca-lab.org/ .\n"));
	} else {
	}

	/*
	// Context 6630 00:12:62:e3:53:b1
	_LIT8(mac, "\x00\x12\x62\xe3\x53\xb1");
	if (aDevice.iMAC().Compare(mac)==0) {
		doSend=ETrue;
		aMessageCode=1;
		aWithName=_L("testname");
		aWithTitle=_L("testtitle");
		_LIT8(KBody8, "this is the test body");
		if (aBody) {
			if (aBody->Des().MaxLength() < KBody8().Length()) {
				delete aBody; aBody=0;
			} else {
				aBody->Des().Zero();
			}
		}
		if (!aBody) {
			aBody=KBody8().AllocL();
		} else {
			aBody->Des()=KBody8;
		}
	}*/

	if (doSend) {
		Reporting().DebugLog(_L("localogic: asking to send message"));
	}
}

void CLocaLogicImpl::Failed(const TBBBtDeviceInfo& aDevice,
	TInt aMessageCode,
	const TTime& aAtTime,
	TSendFailure aErrorCode,
	TBool	aLocal) 
{
	CALLSTACKITEM_N(_CL("CLocaLogicImpl"), _CL("Failed"));

	TDevStats s;
	iDevStats->GetStatsL(aDevice, s);
	s.iFailureCount++;
	if (aLocal) {
		s.iPreviousLocalFailure=aAtTime;
		s.iLocalFailureCount++;;
	} else {
		s.iPreviousRemoteFailure=aAtTime;
	}

/*<<<<<<< loca_logic.cpp
	TTime lastseen, LastSeenContinuously; TInt failures, successes;
	iDevStats->GetStatsL(aDevice, lastseen, LastSeenContinuously, failures, successes);
	iDevStats->SetStatsL(aDevice, lastseen, aAtTime, failures+1, successes);

	TBuf<100> msg;
	msg=_L("localogic: failed to send message ");
	msg.AppendNum(aErrorCode);
	DebugLog(msg);

	iMsgStatus.iMessageId()=aMessageCode;
	iMsgStatus.iRecipientAddress()=aDevice.iMAC();
	iMsgStatus.iSucceeded()=EFalse;
	iMsgStatus.iAtTime()=aAtTime;
	iEvent.iStamp()=aAtTime;
	iBBSubSession->PutRequestL(KLocaMessageStatusTuple, KNullDesC,
		&iEvent, KNoComponent);
=======*/
	iDevStats->SetStatsL(aDevice, s);

#ifndef __WINS__
	if (aLocal) {
		TBuf<100> msg;
		msg=_L("localogic: failed to send message ");
		msg.AppendNum(aErrorCode);
		Reporting().DebugLog(msg);

		iMsgStatus.iMessageId()=aMessageCode;
		iMsgStatus.iRecipientAddress()=aDevice.iMAC();
		iMsgStatus.iSucceeded()=EFalse;
		iMsgStatus.iAtTime()=aAtTime;
		iEvent.iStamp()=aAtTime;
		iBBSubSession->PutRequestL(KLocaMessageStatusTuple, KNullDesC,
			&iEvent, KNoComponent);
	}
#endif
//>>>>>>> 1.2
}

void CLocaLogicImpl::Success(const TBBBtDeviceInfo& aDevice,
	TInt aMessageCode,
	const TTime& aAtTime,
	TBool aLocal)
{
	CALLSTACKITEM_N(_CL("CLocaLogicImpl"), _CL("Success"));

/*<<<<<<< loca_logic.cpp
	TTime lastseen, LastSeenContinuously; TInt failures, successes;
	iDevStats->GetStatsL(aDevice, lastseen, LastSeenContinuously, failures, successes);
	iDevStats->SetStatsL(aDevice, lastseen, aAtTime, 0, successes+1);

	DebugLog(_L("localogic: sent message"));

	iMsgStatus.iMessageId()=aMessageCode;
	iMsgStatus.iRecipientAddress()=aDevice.iMAC();
	iMsgStatus.iSucceeded()=ETrue;
	iMsgStatus.iAtTime()=aAtTime;
	iEvent.iStamp()=aAtTime;
	iBBSubSession->PutRequestL(KLocaMessageStatusTuple, KNullDesC,
		&iEvent, KNoComponent);
=======*/
	TDevStats s;
	iDevStats->GetStatsL(aDevice, s);
	s.iSuccessCount++;
	if (aLocal) {
		s.iPreviousSeen=aAtTime;
		s.iPreviousLocalSuccess=aAtTime;
		s.iLocalFailureCount=0;
	} else {
		s.iPreviousRemoteSuccess=aAtTime;
	}
	s.iFailureCount=0;

	iDevStats->SetStatsL(aDevice, s);

#ifndef __WINS__
	if (aLocal) {
		Reporting().DebugLog(_L("localogic: sent message"));

		iMsgStatus.iMessageId()=aMessageCode;
		iMsgStatus.iRecipientAddress()=aDevice.iMAC();
		iMsgStatus.iSucceeded()=ETrue;
		iMsgStatus.iAtTime()=aAtTime;
		iEvent.iStamp()=aAtTime;
		iBBSubSession->PutRequestL(KLocaMessageStatusTuple, KNullDesC,
			&iEvent, KNoComponent);
	}
#endif
//>>>>>>> 1.2
}


CDevStats* CDevStats::NewL(MApp_context& aContext, RDbDatabase& aDb)
{
	CALLSTACKITEM2_N(_CL("CDevStats"), _CL("NewL"), &aContext);

	auto_ptr<CDevStats> ret(new (ELeave) CDevStats(aContext, aDb));
	ret->ConstructL();
	return ret.release();
}

TTime CDevStats::GetColTime(TColumns aColumn)
{
	if (iTable.IsColNull(aColumn) ) {
		return TTime(0);
	} else {
		return iTable.ColTime(aColumn);
	}
}

TInt CDevStats::GetColInt(TColumns aColumn)
{
	if (iTable.IsColNull(aColumn) ) {
		return TInt(0);
	} else {
		return iTable.ColInt(aColumn);
	}
}

void CDevStats::GetStatsL(const TBBBtDeviceInfo& aDevice,
		TDevStats& aStats)
{
	CALLSTACKITEM_N(_CL("CDevStats"), _CL("GetStatsL"));


	if (SeekToDevL(aDevice)) {
		iTable.GetL();

		aStats.iLastSeen=GetColTime(ELastSeen);
		aStats.iLastSeenContinuously=GetColTime(ELastSeenContinuously);
		aStats.iLastSeenContinuouslyBegin=GetColTime(ELastSeenContinuouslyBegin);
		aStats.iPreviousSeen=GetColTime(EPreviousSeen);
		aStats.iFailureCount=iTable.ColInt(EFailureCount);
		aStats.iSuccessCount=iTable.ColInt(ESuccessCount);
		aStats.iPreviousLocalSuccess=GetColTime(EPreviousLocalSuccess);
		aStats.iPreviousRemoteSuccess=GetColTime(EPreviousRemoteSuccess);
		aStats.iPreviousLocalFailure=GetColTime(EPreviousLocalFailure);
		aStats.iPreviousRemoteFailure=GetColTime(EPreviousRemoteFailure);
		aStats.iLocalFailureCount=GetColInt(ELocalFailureCount);


	} else {
		Mem::FillZ( &aStats, sizeof(TDevStats) );
	}
}

void CDevStats::SetStatsL(const TBBBtDeviceInfo& aDevice,
		const TDevStats& aStats)
{
	CALLSTACKITEM_N(_CL("CDevStats"), _CL("SetStatsL"));


	if (SeekToDevL(aDevice)) {
		iTable.UpdateL();
	} else {
		iTable.InsertL();
		iTable.SetColL(EDevAddr, aDevice.iMAC());
	}

	iTable.SetColL(ELastSeen, aStats.iLastSeen);
	iTable.SetColL(ELastSeenContinuously, aStats.iLastSeenContinuously);
	iTable.SetColL(ELastSeenContinuouslyBegin, aStats.iLastSeenContinuouslyBegin);
	iTable.SetColL(EPreviousSeen, aStats.iPreviousSeen);
	iTable.SetColL(EFailureCount, aStats.iFailureCount);
	iTable.SetColL(ESuccessCount, aStats.iSuccessCount);
	iTable.SetColL(EPreviousLocalSuccess, aStats.iPreviousLocalSuccess);
	iTable.SetColL(EPreviousRemoteSuccess, aStats.iPreviousRemoteSuccess);
	iTable.SetColL(EPreviousLocalFailure, aStats.iPreviousLocalFailure);
	iTable.SetColL(EPreviousRemoteFailure, aStats.iPreviousRemoteFailure);
	iTable.SetColL(ELocalFailureCount, aStats.iLocalFailureCount);

	PutL();
}

TBool	CDevStats::SeekToDevL(const TBBBtDeviceInfo& aDevice)
{
	CALLSTACKITEM_N(_CL("CDevStats"), _CL("SeekToDevL"));


	TDbSeekKey k(aDevice.iMAC());
	return iTable.SeekL(k);
}

CDevStats::CDevStats(MApp_context& aContext, RDbDatabase& aDb) : MContextBase(aContext),
	MDBStore(aDb) { }

void CDevStats::ConstructL()
{
	CALLSTACKITEM_N(_CL("CDevStats"), _CL("ConstructL"));


	TInt cols[]= { EDbColText8, EDbColDateTime, EDbColInt32, EDbColInt32, 
		EDbColDateTime, EDbColDateTime, EDbColDateTime, EDbColDateTime,
		EDbColDateTime, EDbColDateTime, EDbColDateTime, EDbColInt32,
		-1 };
	TInt idxs[]= { EDevAddr, -1 };
	MDBStore::SetTextLen(6);
	MDBStore::ConstructL(cols, idxs, true, _L("DEVSTATS"), ETrue);
}
