#include "loca_sender.h"

#include "i_logger.h"
#include "bluejack.h"
#include "symbian_auto_ptr.h"
#include "loca_logic.h"
#include "csd_bluetooth.h"
#include "db.h"
#include "timeout.h"
#include <e32math.h>
#include "cl_settings.h"
#include "reporting.h"

class CLocaSenderImpl : public CLocaSender, public MContextBase,
	public Mlogger, public MObexNotifier, public MTimeOut,
	public MSettingListener {
private:
	CLocaSenderImpl(MApp_context& Context);
	~CLocaSenderImpl();
	void ConstructL();

	// Mlogger
	virtual void NewSensorEventL(const TTupleName& aName, 
		const TDesC& aSubName, const CBBSensorEvent& aEvent);

	// MObexNotifier
	virtual void	Error(TInt aError, MObexNotifier::TState  aAtState);
	virtual void	Success();
	virtual void	Cancelled(MObexNotifier::TState aStateBeforeCancel);

	// MTimeOut
	virtual void expired(CBase* Source);

	// MSettingListener
	virtual void SettingChanged(TInt Setting);

	// own
	void CancelSend();

	enum TState {
		EIdle,
		ESending,
		ECanceling
	};
	TState		iCurrentState;
	TBBBtDeviceInfo iSendingTo; TBTDevAddr iSendingToAddr;
	TInt		iSendingMessageId;
	TBuf<240>	iSendingWithName, iSendingWithTitle;
	HBufC8*		iSendingBody;

	CDb*		iDb;
	CBlueJack*	iBlueJack;
	CLocaLogic*	iLocaLogic;
	CTimeOut*	iTimeOut;

	TInt		iMessageTimeOut;
	TInt64		iRandomSeed;
	TBool		iEnabled;

	friend CLocaSender;
	friend auto_ptr<CLocaSenderImpl>;
};

CLocaSender* CLocaSender::NewL(MApp_context& Context)
{
	CALLSTACKITEM_N(_CL("CLocaSender"), _CL("NewL"));

	auto_ptr<CLocaSenderImpl> ret(new (ELeave) CLocaSenderImpl(Context));
	ret->ConstructL();
	return ret.release();
}


CLocaSenderImpl::CLocaSenderImpl(MApp_context& Context) : MContextBase(Context) { }

CLocaSenderImpl::~CLocaSenderImpl()
{
	CALLSTACKITEM_N(_CL("CLocaSenderImpl"), _CL("~CLocaSenderImpl"));

	Settings().CancelNotifyOnChange(SETTING_ENABLE_LOCA_BLUEJACK, this);
	Settings().CancelNotifyOnChange(SETTING_LOCA_BLUEJACK_MESSAGE_TIMEOUT, this);

	delete iBlueJack;
	delete iLocaLogic;
	delete iTimeOut;
	delete iDb;
	delete iSendingBody;
}

void CLocaSenderImpl::ConstructL()
{
	CALLSTACKITEM_N(_CL("CLocaSenderImpl"), _CL("ConstructL"));

	Mlogger::ConstructL(AppContextAccess());

	iDb=CDb::NewL(AppContext(), _L("LOCALOGIC"), EFileWrite);

	iLocaLogic=CLocaLogic::NewL(AppContext(), iDb->Db());
	iBlueJack=CBlueJack::NewL(AppContext(), *this);
	iTimeOut=CTimeOut::NewL(*this);

	SubscribeL(KBluetoothTuple);

	// FIXME: make a setting
	
	Settings().GetSettingL(SETTING_LOCA_BLUEJACK_MESSAGE_TIMEOUT, iMessageTimeOut);
	Settings().NotifyOnChange(SETTING_LOCA_BLUEJACK_MESSAGE_TIMEOUT, this);

	iRandomSeed=GetTime().Int64();
	iEnabled=EFalse;
	Settings().GetSettingL(SETTING_ENABLE_LOCA_BLUEJACK, iEnabled);
	Settings().NotifyOnChange(SETTING_ENABLE_LOCA_BLUEJACK, this);
}

void CLocaSenderImpl::SettingChanged(TInt Setting)
{
	if (Setting==SETTING_ENABLE_LOCA_BLUEJACK)
		Settings().GetSettingL(SETTING_ENABLE_LOCA_BLUEJACK, iEnabled);
	else
		Settings().GetSettingL(SETTING_LOCA_BLUEJACK_MESSAGE_TIMEOUT, iMessageTimeOut);
}

void CLocaSenderImpl::Cancelled(MObexNotifier::TState aStateBeforeCancel)
{
	iLocaLogic->Failed(iSendingTo, iSendingMessageId,
		GetTime(), CLocaLogic::ETimeOut, ETrue);
	iCurrentState=EIdle;
}

void CLocaSenderImpl::CancelSend()
{
	CALLSTACKITEM_N(_CL("CLocaSenderImpl"), _CL("CancelSend"));

	iCurrentState=ECanceling;
	iBlueJack->CancelSend();
}

void CLocaSenderImpl::NewSensorEventL(const TTupleName& , 
	const TDesC& , const CBBSensorEvent& aEvent)
{
	CALLSTACKITEM_N(_CL("CLocaSenderImpl"), _CL("NewSensorEventL"));

	Reporting().DebugLog(_L("CLocaSenderImpl::NewSensorEventL"));
	if (!iEnabled) return;

	const CBBBtDeviceList* devices=bb_cast<CBBBtDeviceList>(aEvent.iData());
	if (!devices) return;

	if (iCurrentState==ESending) {
		// shouldn't happen
		CancelSend();
		return;
	}
	if (iCurrentState==ECanceling) {
		return;
	}

	if (devices->Count()==0) return;

	auto_ptr< CArrayPtrFlat<TBBBtDeviceInfo> > arr(
		new CArrayPtrFlat<TBBBtDeviceInfo>(devices->Count()));
	arr->SetReserveL(devices->Count());

	const TBBBtDeviceInfo* i=0;
	TBool doSend=EFalse;
	for (i=devices->First(); i; i=devices->Next()) {
		iLocaLogic->GetMessage(*i, 
			devices->Count(),
			aEvent.iStamp(), 
			doSend, iSendingMessageId,
			iSendingWithName, iSendingWithTitle,
			iSendingBody);
		if (doSend) {
			arr->AppendL( (TBBBtDeviceInfo*) i);
		}
	}

	if (arr->Count()==0) return;

	// pick one of candidates randomly
	TInt r=Math::Rand(iRandomSeed);
	TInt step=KMaxTInt/arr->Count();
	TInt selected=r/step;
	if (selected > arr->Count()-1) selected=arr->Count()-1;

	iSendingTo=*(arr->At(selected));

	iLocaLogic->GetMessage(iSendingTo, 
		devices->Count(),
		aEvent.iStamp(), 
		doSend, iSendingMessageId,
		iSendingWithName, iSendingWithTitle,
		iSendingBody);

	iCurrentState=ESending;
	TBTDevAddr a(iSendingTo.iMAC());
	iSendingToAddr=a;
	TInt connectcount=10;
	Settings().GetSettingL(SETTING_LOCA_BLUEJACK_CONNECT_COUNT, connectcount);
	Reporting().DebugLog(_L("SendMessage"));
	iBlueJack->SendMessageL(iSendingToAddr,
		iSendingWithName, iSendingWithTitle,
		*iSendingBody, connectcount);
	iTimeOut->Wait(iMessageTimeOut);
}

void CLocaSenderImpl::Error(TInt aError, MObexNotifier::TState aAtState)
{
	CALLSTACKITEM_N(_CL("CLocaSenderImpl"), _CL("Error"));

	iTimeOut->Reset();
	TBuf<100> msg;
	msg=_L("locasender: error in send ");
	msg.AppendNum(aError);
	msg.Append(_L(" at state "));
	msg.AppendNum(aAtState);
	Reporting().DebugLog(msg);

	// FIXME: how to get error type?
	CLocaLogic::TSendFailure fail=CLocaLogic::EUnknown;
	if (aError==KErrCouldNotConnect && aAtState==MObexNotifier::ESending) {
		fail=CLocaLogic::ERefused;
	}
	iLocaLogic->Failed(iSendingTo, iSendingMessageId,
		GetTime(), fail, ETrue);
	iCurrentState=EIdle;
}

void CLocaSenderImpl::Success()
{
	CALLSTACKITEM_N(_CL("CLocaSenderImpl"), _CL("Success"));
	Reporting().DebugLog(_L("CLocaSenderImpl::Success"));

	iTimeOut->Reset();
	iLocaLogic->Success(iSendingTo, iSendingMessageId, GetTime(), ETrue);
	iCurrentState=EIdle;
}

void CLocaSenderImpl::expired(CBase* Source)
{
	CALLSTACKITEM_N(_CL("CLocaSenderImpl"), _CL("expired"));

	CancelSend();
}