/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "presence_publisher.h"
#include "symbian_auto_ptr.h"
#include "cl_settings.h"
#include "i_logger.h"
#include "cellmap.h"
#include "eikenv.h"
#include "status_notif.h"
#include <context_log.mbg>

#include "csd_cell.h"
#include "csd_profile.h"
#include "csd_bluetooth.h"
#include "csd_base.h"
#include "csd_idle.h"
#include "csd_gps.h"
#include "app_context_impl.h"
#include "callstack.h"

#ifndef __WINS__
_LIT(KIconFile, "c:\\system\\data\\context_log.mbm");
#else
_LIT(KIconFile, "z:\\system\\data\\context_log.mbm");
#endif

#pragma warning(disable: 4706) // assignment withing conditional expression

_LIT(KPresencePublisher, "PresencePublisher");

CPresencePublisher* CPresencePublisher::NewL(MApp_context& Context, i_status_notif* CallBack, 
					     MPresencePublisherListener& aListener, CBTDeviceList* aBuddyList,
					     CBTDeviceList *aLaptopBTs, CBTDeviceList *aDesktopBTs, 
					     CBTDeviceList *aPDABTs)
{
	CALLSTACKITEMSTATIC_N(_CL("CPresencePublisher"), _CL("NewL"));

	auto_ptr<CPresencePublisher> ret( new (ELeave) CPresencePublisher(Context, 
		CallBack, aListener, aBuddyList, aLaptopBTs, aDesktopBTs, aPDABTs) );
	ret->ConstructL();
	return ret.release();
}

CPresencePublisher::~CPresencePublisher()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("~CPresencePublisher"));

	delete ctx;
	Settings().CancelNotifyOnChange(SETTING_PRESENCE_ENABLE, this);
	Settings().CancelNotifyOnChange(SETTING_IP_AP, this);
	Cancel();
	delete iWait;
	iSession.Close();
	delete iSendPresence;
	delete iPresence;
	delete iPresenceRunning;
	delete iData;
}

void CPresencePublisher::SuspendConnection()
{
	iCallBack->status_change(_L("Suspend Presence"));

	if (iNextOp==EDisable || iNextOp==ESuspend) return;
	iWait->Cancel();

	switch (iCurrentState) {
	case EDisabled:
	case EDisabling:
		break;
	case ESuspended:
	case ESuspending:
		break;
	case EReconnecting:
		iCurrentState=ESuspended;
		break;
	default:
		if (IsActive()) {
			iNextOp=ESuspend;
		} else {
			iSession.MsgSuspendConnection(iStatus);
			SetActive();
			iCurrentState=ESuspending;
		}
		break;
	}
}

void CPresencePublisher::QueueUpdate()
{
	iNewValue=true;
	if (IsActive()) return;
	iCurrentState=EQueuingUpdate;
	TRequestStatus *s=&iStatus;
	User::RequestComplete(s, KErrNone);
	SetActive();
}
	
bool CPresencePublisher::Disabled()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("Disabled"));

	if (iCurrentState==EDisabling || iCurrentState==EDisabled || iNextOp==EDisable) {
		return true;
	}
	return false;
}

void CPresencePublisher::ResumeConnection()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("ResumeConnection"));
	iCallBack->status_change(_L("Resume Presence"));

	if (IsActive() && (iNextOp==EResume || iNextOp==EEnable)) return;
	iWait->Cancel();

	switch (iCurrentState) {
	case EReconnecting:
	default:
		if (IsActive()) {
			iNextOp=EResume;
		} else {
			Restart();
		}
		break;
	}
}

bool CPresencePublisher::ConnectionSuspended()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("ConnectionSuspended"));

	if (iCurrentState==ESuspending || iCurrentState==ESuspended || iNextOp==ESuspend ||
		iCurrentState==EDisabling || iCurrentState==EDisabled || iNextOp==EDisable) return true;
	return false;
}

CPresencePublisher::CPresencePublisher(MApp_context& Context, i_status_notif* CallBack , 
				       MPresencePublisherListener& aListener, CBTDeviceList* aBuddyList,
				       CBTDeviceList *aLaptopBTs, CBTDeviceList *aDesktopBTs, CBTDeviceList *aPDABTs): 
CCheckedActive(EPriorityIdle, _L("CPresencePublisher")), MContextBase(Context), iCallBack(CallBack), iListener(aListener),
iBuddyList(aBuddyList), iLaptopBTs(aLaptopBTs), iDesktopBTs(aDesktopBTs), iPDABTs(aPDABTs) { }

void CPresencePublisher::ConstructL()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("ConstructL"));

	Mlogger::ConstructL(AppContext());
	SubscribeL(KAnySensorEvent);

	iData=CBBPresence::NewL();

	{
		TRAPD(err, iPresenceRunning=CNotifyState::NewL(AppContext(), KIconFile));
		if (iPresenceRunning) iPresenceRunning->SetCurrentState(EMbmContext_logP_not, EMbmContext_logP_not);
	}

	{
		ctx=CApp_context::NewL(true, _L("ContextServer"));
	}

	TInt enabled=1;
	{
		iWait=CTimeOut::NewL(*this);
		iPresence=CXmlBufExternalizer::NewL(1024);

		iCurrentState=EDisabled;
		Settings().NotifyOnChange(SETTING_PRESENCE_ENABLE, this);
		Settings().NotifyOnChange(SETTING_IP_AP, this);

		if (! Settings().GetSettingL(SETTING_PRESENCE_ENABLE, enabled) ) {
			enabled=1;
		}
	}

	{
		CActiveScheduler::Add(this);
		Settings().GetSettingL(SETTING_IP_AP, iAP);
		if (enabled && iAP>0) {
			iWaitTime=1;
			Restart();
		} else {
			iCurrentState=EDisabled;
		}
	}
}

void CPresencePublisher::SettingChanged(TInt /*Setting*/)
{
#ifdef __WINS__
	RDebug::Print(_L("CPresencePublisher::SettingChanged"));
#endif

	//iCallBack->status_change(_L("settingchanged"));

	TInt enabled=1;
	if (! Settings().GetSettingL(SETTING_PRESENCE_ENABLE, enabled) ) {
		enabled=1;
	}
	Settings().GetSettingL(SETTING_IP_AP, iAP);
	if (enabled && iAP>0) {
		iWaitTime=1;
		Restart();
	} else {
		if (iCurrentState==EDisabled || iCurrentState==EDisabling) return;
		iWait->Cancel();
		if (iCurrentState==EReconnecting || iCurrentState==ESuspended) {
			Cancel();
			iCurrentState=EDisabled;
			return;
		}
		if (iCurrentState!=ESuspending) {
			Cancel();
			iSession.MsgSuspendConnection(iStatus);
			iCurrentState=ESuspending;
			SetActive();
		}
		iNextOp=EDisable;
	}
}

// Mlogger

void CPresencePublisher::NewSensorEventL(const TTupleName& aName, 
					 const TDesC& , const CBBSensorEvent& aEvent)
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("NewSensorEventL"));

	if (aEvent.iPriority()!=CBBSensorEvent::VALUE || !aEvent.iData()) return;

	const TTupleName& tuplename=aName;
	TTime time=aEvent.iStamp();

	if (tuplename==KCellIdTuple) {
		const TBBCellId* cell=bb_cast<TBBCellId>(aEvent.iData());
		if (!cell) return;
		iData->iCellId=*cell;
		return;
	} else if (tuplename==KUserGivenContextTuple) {
		const TBBUserGiven * user_given=bb_cast<TBBUserGiven>(aEvent.iData());
		if (!user_given) return;
		iData->iUserGiven=*user_given;
		Settings().WriteSettingL(SETTING_OWN_DESCRIPTION, user_given->iDescription());
		Settings().WriteSettingL(SETTING_OWN_DESCRIPTION_TIME, user_given->iSince());

	} else if (tuplename==KProfileTuple) {
		const TBBProfile* profile=bb_cast<TBBProfile>(aEvent.iData());
		if (!profile) return;
		if (iData->iProfile.Equals(profile)) return;

		iData->iProfile=*profile;
	} else if (tuplename==KIdleTuple ) {
		const TBBUserActive* act=bb_cast<TBBUserActive>(aEvent.iData());
		if (!act) return;
		if (iData->iUserActive.Equals(act)) return;
		iData->iUserActive=*act;
	} else if (tuplename==KGpsTuple ) {
		const TGpsLine* gps=bb_cast<TGpsLine>(aEvent.iData());
		if (!gps) return;
		iData->iGps=*gps;
		return;
	} else if (tuplename==KBaseTuple) {
		const TBBBaseInfo* base=bb_cast<TBBBaseInfo>(aEvent.iData());
		if (!base) return;
		if (iData->iBaseInfo.Equals(base)) return;
		iData->iBaseInfo=*base;

	} else if (tuplename==KBluetoothTuple) {
		const CBBBtDeviceList* devs=bb_cast<CBBBtDeviceList>(aEvent.iData());
		if (!devs) return;

		iData->SetDevices(devs);

		TUint prev_buddies=iData->iNeighbourhoodInfo.iBuddies();
		TUint prev_others=iData->iNeighbourhoodInfo.iOtherPhones();
		TUint prev_laptops=iData->iNeighbourhoodInfo.iLaptops();
		TUint prev_desktops=iData->iNeighbourhoodInfo.iDesktops();
		TUint prev_pdas=iData->iNeighbourhoodInfo.iPDAs();
		iData->iNeighbourhoodInfo.iBuddies()=0;
		iData->iNeighbourhoodInfo.iOtherPhones()=0;
		iData->iNeighbourhoodInfo.iLaptops()=0;
		iData->iNeighbourhoodInfo.iDesktops()=0;
		iData->iNeighbourhoodInfo.iPDAs()=0;

		for (const TBBBtDeviceInfo *node=devs->First(); node; node=devs->Next()) {
			if ( iBuddyList->ContainsDevice(node->iMAC()) ) {
				++iData->iNeighbourhoodInfo.iBuddies();
			} else if ( iLaptopBTs->ContainsDevice(node->iMAC()) ) {
				++iData->iNeighbourhoodInfo.iLaptops();
			} else if ( iDesktopBTs->ContainsDevice(node->iMAC()) ) {
				++iData->iNeighbourhoodInfo.iDesktops();
			} else if ( iPDABTs->ContainsDevice(node->iMAC()) ) {
				++iData->iNeighbourhoodInfo.iPDAs();
			} else if ( node->iMajorClass() == 2 ) {
				++iData->iNeighbourhoodInfo.iOtherPhones();
			} 
		}
#ifdef __WINS__
		iData->iNeighbourhoodInfo.iLaptops()=1;
		iData->iNeighbourhoodInfo.iDesktops()=1;
		iData->iNeighbourhoodInfo.iPDAs()=1;
#endif
		if (iData->iNeighbourhoodInfo.iBuddies()==prev_buddies &&
			iData->iNeighbourhoodInfo.iOtherPhones()==prev_others &&
			iData->iNeighbourhoodInfo.iLaptops()==prev_laptops &&
			iData->iNeighbourhoodInfo.iDesktops()==prev_desktops &&
			iData->iNeighbourhoodInfo.iPDAs()==prev_pdas) return;

	}

	MakePresence();
	if (iCurrentState==EConnected) {
		// wait one turn of the active scheduler
		QueueUpdate();
	} else {
		iNewValue=true;
	}
}

void CPresencePublisher::MakePresence()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("MakePresence"));

	iPresence->Zero();
	iData->IntoXmlL(iPresence);
}

void CPresencePublisher::SendUpdate()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("SendUpdate"));

	delete iSendPresence; iSendPresence=0;
	iSendPresence=iPresence->Buf().AllocL();

	iSession.MsgUpdateUserPresence(*iSendPresence, iStatus);

	SetActive();
	iCurrentState=ESendingUpdate;
	iNewValue=false;

	iSentTimeStamp=GetTime();

#ifdef __WINS__
	RDebug::Print(_L("CPresencePublisher::SendUpdate()"));
#endif

	//User::Leave(1003);

}

void CPresencePublisher::Restart()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("Restart"));

	//iCallBack->status_change(_L("restart"));

	Cancel();
	iSession.Close();
	iCurrentState=EReconnecting;
	iNextOp=ENone;
	iWait->Wait(iWaitTime);
#ifdef __WINS__
	RDebug::Print(_L("CPresencePublisher::Restart()"));
#endif
}

void CPresencePublisher::CheckedRunL()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("CheckedRunL"));

#ifdef __WINS__
	TBuf<150> msg;
	msg.Format(_L("CPresencePublisher::CheckedRunL %d, state"), iStatus.Int(), iCurrentState);
	RDebug::Print(msg);
	msg.Zero();
#else
	TBuf<50> msg;
#endif

	if (iStatus<0) {
		iWaitTime=(int)(iWaitTime*1.5);

		if(iPresenceRunning) iPresenceRunning->SetCurrentState(EMbmContext_logP_not, EMbmContext_logP_not);
		// depending on the error, we should not always restart
		//Restart();
		if (iStatus == -1 /*EIdentificationError*/) 
		{
			TBuf<100> error;
			error.Append(_L("Presence ID Error: "));
			error.Append(iUser);
			
			iCallBack->status_change(_L(""));
			iCallBack->error(error);
			iCurrentState = ESuspended;
			CEikonEnv::Static()->AlertWin(error,iPass);

		}
		else 
		{
			msg.Format(_L("Presence error %d, restarting"), iStatus.Int());
			iCallBack->status_change(msg);
			if (iStatus.Int() == KErrServerTerminated) {
				// ContextServer crashed
				auto_ptr<HBufC> stack(0);
				
				TRAPD(err, stack.reset(ctx->CallStackMgr().GetFormattedCallStack(_L("Presence"))));
				if (err==KErrNone && stack.get()!=0) {
					iCallBack->status_change(*stack);
				}
			}
			
			
			Restart();
		}
		return;
	}

	iWaitTime=10;

	switch (iCurrentState) {
	case EConnecting:
	case EResuming:
		msg=_L("Presence Connected");
		if(iPresenceRunning) iPresenceRunning->SetCurrentState(EMbmContext_logP, EMbmContext_logP);
	case ESendingUpdate:
		iCurrentState=EConnected;
		iData->iSentTimeStamp=GetTime();
		iData->iSent=ETrue;
		iListener.NotifyNewPresence(iData);
		break;
	case ESuspending:
		msg=_L("Presence Suspended");
		if(iPresenceRunning) iPresenceRunning->SetCurrentState(EMbmContext_logP_not, EMbmContext_logP_not);
		iCurrentState=ESuspended;
		iData->iSent=EFalse;
		iListener.NotifyNewPresence(iData);
		break;
	case EQueuingUpdate:
		iCurrentState=EConnected;
		break;
	default:
		msg=_L("Presence Restarting");
		Restart();
		break;
	}

	if (msg.Length()>0) iCallBack->status_change(msg);

	switch (iNextOp) {
	case ESuspend:
		iSession.MsgSuspendConnection(iStatus);
		SetActive();
		iCurrentState=ESuspending;
		break;
	case EResume:
		iSession.MsgResumeConnection(iStatus);
		SetActive();
		iCurrentState=EResuming;
		break;
	case EDisable:
		iSession.Close();
		iCurrentState=EDisabled;
		break;
	default:
		break;
	};
	iNextOp=ENone;

	if (iNewValue && iCurrentState==EConnected) {
		SendUpdate();
	}
}
		
void CPresencePublisher::DoCancel()
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("DoCancel"));

	iSession.Cancel();
}

TInt CPresencePublisher::CheckedRunError(TInt /*aError*/)
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("CheckedRunError"));

	Restart();
	
	return KErrNone;
}

void CPresencePublisher::expired(CBase*)
{
	CALLSTACKITEM_N(_CL("CPresencePublisher"), _CL("expired"));

	//iCallBack->status_change(_L("expired"));
	TBuf<50> tmp;

	TInt enabled=1;
	if (! Settings().GetSettingL(SETTING_PRESENCE_ENABLE, enabled) ) {
		enabled=1;
	}
	if (!enabled) {
		iCurrentState=EDisabled;
		return;
	}

	if (Settings().GetSettingL(SETTING_JABBER_NICK, tmp) && Settings().GetSettingL(SETTING_JABBER_PASS, iPass)) {
		TInt sep=tmp.Locate('@');
		if (sep!=KErrNotFound) {
			TInt ret=iSession.ConnectToContextServer();
			if (ret!=KErrNone) {
				TBuf<30> msg;
				msg.Format(_L("ConnectToContextServer %d"), ret);
				iCallBack->error(msg);
				Restart();
				return;
			}
			iUser=tmp.Mid(0, sep);
			iServer=tmp.Mid(sep+1);

			iCallBack->status_change(_L("Presence Connecting"));
			iSession.MsgConnectToPresenceServer(iUser, iPass,
				iServer, iAP, iStatus);
			iCurrentState=EConnecting;
			SetActive();
		}
	}
}

const CBBPresence* CPresencePublisher::Data() const
{
	return iData;
}
