/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "app_context.h"
#include <coemain.h>
#include <flogger.h>
#include <bautils.h>


EXPORT_C void print_stacktrace(RFs& fs, MApp_context* context, const TDesC& cat, TInt error)
{
	if (!context) return;
	RFile f;
	if (f.Open(fs, _L("C:\\crash.txt"), 
		EFileShareAny | EFileStreamText | EFileWrite)!=KErrNone) {
		if(f.Create(fs, _L("C:\\crash.txt"), 
			EFileShareAny | EFileStreamText | EFileWrite)!=KErrNone) {
			return;
		}
	}
	CleanupClosePushL(f);

	TInt pos=0;
	f.Seek(ESeekEnd, pos);

	TBuf<80> msg;
	msg.Format(_L("context_log crashed with error %d\ncategory "), error);
	TPtrC8 msg2( (TText8*)msg.Ptr(), msg.Length()*2);
	f.Write(msg2);
	TPtrC8 msg3( (TText8*)cat.Ptr(), cat.Length()*2);
	f.Write(msg3);
	f.Write(_L8("\ntrace:\n"));

	CCallStack::Node* n;
	n=context->CallStack().iFirst;
	while (n) {
		TPtrC8 msg2( (TText8*)n->Item.Ptr(), n->Item.Length()*2);
		f.Write(msg2);
		n=n->Next;
	}
	CleanupStack::PopAndDestroy();
}

EXPORT_C void exceptionhandler(TExcType t)
{
	MApp_context* c;
	c=MApp_context::Static();
	if (c) {
		print_stacktrace(c->Fs(), c, _L("Exception"), t);
	}
	User::Panic(_L("EXCEPTION"), t);
}

EXPORT_C int MApp_context::AddRef()
{
	return ++refcount;
}

EXPORT_C int MApp_context::Release()
{
	--refcount;
	if (!refcount) {
		delete this;
		return 0;
	}
	return refcount;
}

MApp_context::~MApp_context()
{
}

CApp_context::CApp_context() : CCheckedActive(EPriorityNormal, _L("CApp_context"))
{
}

void CApp_context::ConstructL()
{
	iDiskSpaceLimit=500*1024; // 500k
	iDrive=2; // C:
	iFileLog=0;

	iSpaceLeft=true;

	User::LeaveIfError(iFs.Connect());

	TVolumeInfo volinfo;
	if (iFs.Volume(volinfo, iDrive)==KErrNone) {
		if (volinfo.iFree<=iDiskSpaceLimit) {
			iSpaceLeft=false;
		}
	}
	globalNote=CAknGlobalNote::NewL();

	User::LeaveIfError(iSysAgent.Connect());

#ifndef __WINS__
	User::LeaveIfError(iTelServer.Connect());
	_LIT(KGsmModuleName, "phonetsy.tsy");
	User::LeaveIfError( iTelServer.LoadPhoneModule( KGsmModuleName ) );
	RTelServer::TPhoneInfo info;
	User::LeaveIfError( iTelServer.GetPhoneInfo( 0, info ) );
	User::LeaveIfError( iPhone.Open( iTelServer, info.iName ) );
#endif

	cc=CCnvCharacterSetConverter::NewL();
	cc->PrepareToConvertToOrFromL(KCharacterSetIdentifierIso88591, iFs);

	CActiveScheduler::Add(this);
	iFs.NotifyDiskSpace(iDiskSpaceLimit, iDrive, iStatus);
	SetActive();

	iCallStack=CCallStack::NewL();

	Dll::SetTls(this);

	RFileLogger iLog;
	iLog.Connect();
	iLog.CreateLog(_L("ContextCommon"),_L("CApp_context"),EFileLoggingModeAppend);
	iLog.Write(_L("constructed"));

	iLog.CloseLog();
	iLog.Close();

}

CApp_context::~CApp_context()
{
	Dll::SetTls(0);

	Cancel();
	delete iSettings;
	delete globalNote;
#ifndef __WINS__
	iPhone.Close();
	_LIT(KGsmModuleName, "phonetsy.tsy");
	iTelServer.UnloadPhoneModule( KGsmModuleName );
	iTelServer.Close();
#endif
	delete cc;
	iSysAgent.Close();
	iFs.Close();

	delete iCallStack;

	RFileLogger iLog;
	iLog.Connect();
	iLog.CreateLog(_L("ContextCommon"),_L("CApp_context"),EFileLoggingModeAppend);
	iLog.Write(_L("deleted"));

	iLog.CloseLog();
	iLog.Close();
}

EXPORT_C CApp_context* CApp_context::NewL()
{
	CApp_context* ret;
	ret=new (ELeave) CApp_context();
	CleanupStack::PushL(ret);
	ret->ConstructL();
	CleanupStack::Pop();
	return ret;
}

RFs& CApp_context::Fs()
{
	return iFs;
}

RSystemAgent& CApp_context::SysAgent()
{
	return iSysAgent;
}

RTelServer& CApp_context::TelServer()
{
	return iTelServer;
}

RAdvGsmPhone& CApp_context::Phone()
{
	return iPhone;
}

CCnvCharacterSetConverter* CApp_context::CC()
{
	return cc;
}

void CApp_context::SetFileLog(MPausable* log)
{
	iFileLog=log;
	if (iFileLog && !iSpaceLeft) iFileLog->pause();
}

void CApp_context::CheckedRunL()
{
	if (iStatus==KErrCancel) return;

	if (iStatus==KErrNone) {
		TVolumeInfo info;
		TBuf<100> msg;
		TAknGlobalNoteType notetype;
		if (iFs.Volume(info, iDrive)==KErrNone) {
			if (info.iFree<=iDiskSpaceLimit) {
				if (iFileLog) iFileLog->pause();
				iSpaceLeft=false;
				msg.Append(_L("No space left!. Logging of data will be stopped."));
				notetype=EAknGlobalErrorNote;
			} else {
				iSpaceLeft=true;
				if (iFileLog) iFileLog->unpause();
				msg.Append(_L("Logging continues."));	
				notetype=EAknGlobalInformationNote;
			}
			TRAPD(err, globalNote->ShowNoteL(notetype, msg));
		}
	}
	iFs.NotifyDiskSpace(iDiskSpaceLimit, iDrive, iStatus);
	SetActive();
}

void CApp_context::DoCancel()
{
	iFs.NotifyDiskSpaceCancel();
}

bool CApp_context::NoSpaceLeft()
{
	return !iSpaceLeft;
}

const TDesC& CApp_context::DataDir()
{
	return iDataDir;
}

void CApp_context::SetDataDir(const TDesC& Dir)
{
	BaflUtils::EnsurePathExistsL(iFs, Dir);
	TDriveInfo i;
	if (iFs.Drive(i, EDriveE)==KErrNone) {
		if (i.iType!=EMediaNotPresent &&
			i.iType!=EMediaUnknown &&
			i.iType!=EMediaCdRom &&
			i.iType!=EMediaRom) {

			// memory card

			TFileName f=Dir;
			f.Replace(0, 1, _L("E"));
			TRAPD(err, BaflUtils::EnsurePathExistsL(iFs, f));
		}
	}
	iDataDir=Dir;
}

const TDesC& CApp_context::AppDir()
{
	return iAppDir;
}

void CApp_context::SetAppDir(const TDesC& Dir)
{
	iAppDir=Dir;
}

void CApp_context::SetSettings(MSettings* Settings)
{
	iSettings=Settings;
}

MSettings& CApp_context::Settings()
{
	return *iSettings;
}

RWsSession& CApp_context::Ws()
{
	return CCoeEnv::Static()->WsSession();
}

CCallStack& CApp_context::CallStack()
{
	return *iCallStack;
}

void CApp_context::SetActiveErrorReporter(MActiveErrorReporter* Reporter)
{
	iReporter=Reporter;
}

void CApp_context::ReportActiveError(const TDesC& Source,
	const TDesC& Reason, TInt Code)
{
	if (iReporter) {
		iReporter->ReportError(Source, Reason, Code);
	} else if (globalNote) {
		TBuf<256> msg;
		msg.Format(_L("Unhandled error %d %S %S"), Code, &(Source.Left(100)), &(Reason.Left(120)));
		globalNote->ShowNoteL(EAknGlobalErrorNote, msg);
	}
}

EXPORT_C MApp_context* MApp_context::Static()
{
	return (MApp_context*)Dll::Tls();
}

EXPORT_C const TDesC& MContextBase::DataDir()
{
	return iContext.DataDir();
}

EXPORT_C const TDesC& MContextBase::AppDir()
{
	return iContext.AppDir();
}

EXPORT_C MSettings& MContextBase::Settings()
{
	return iContext.Settings();
}


EXPORT_C bool MContextBase::NoSpaceLeft()
{
	return iContext.NoSpaceLeft();
}

EXPORT_C MContextBase::MContextBase() : iContext(*(MApp_context::Static()))
{
}

EXPORT_C MContextBase::MContextBase(MApp_context& Context, bool OwnedByContext) : iContext(Context)
{
	iOwnedByContext=OwnedByContext;
	if (!iOwnedByContext) iContext.AddRef();
}

EXPORT_C void MContextBase::SetOwnedByContext()
{
	if (!iOwnedByContext)
		iContext.Release();
	iOwnedByContext=true;
}

EXPORT_C bool MContextBase::OwnedByContext()
{
	return iOwnedByContext;
}

EXPORT_C MContextBase::~MContextBase()
{
	if (!iOwnedByContext)
		iContext.Release();
}

EXPORT_C RFs& MContextBase::Fs()
{
	return iContext.Fs();
}

EXPORT_C RSystemAgent& MContextBase::SysAgent()
{
	return iContext.SysAgent();
}

EXPORT_C RTelServer& MContextBase::TelServer()
{
	return iContext.TelServer();
}

EXPORT_C RAdvGsmPhone& MContextBase::Phone()
{
	return iContext.Phone();
}

EXPORT_C MApp_context& MContextBase::AppContext()
{
	return iContext;
}

EXPORT_C CCnvCharacterSetConverter* MContextBase::CC()
{
	return iContext.CC();
}

EXPORT_C RWsSession& MContextBase::Ws()
{
	return iContext.Ws();
}

EXPORT_C void MContextBase::ReportActiveError(const TDesC& Source,
	const TDesC& Reason, TInt Code)
{
	iContext.ReportActiveError(Source, Reason, Code);
}

EXPORT_C CCallStack& MContextBase::CallStack()
{
	return iContext.CallStack();
}

EXPORT_C TCallStackItem::TCallStackItem(const TDesC& Name)
{
#ifdef __WINS2__
	TBuf<128> msg;
	msg.Format(_L("--> %S"), &Name);
	RDebug::Print(msg);
#endif
	iOnStack=false;

	MApp_context* AppContext=MApp_context::Static();
	if (!AppContext) return;
	TRAPD(err, AppContext->CallStack().Push(Name.Left(100)));
	if (err==KErrNone) {
		iOnStack=true;
	}
}

EXPORT_C TCallStackItem::~TCallStackItem()
{
#ifdef __WINS2__
	TBuf<128> msg;
	msg.Format(_L("<--"));
	RDebug::Print(msg);
#endif
	if (!iOnStack) return;
	MApp_context* AppContext=MApp_context::Static();
	if (!AppContext) return;
	AppContext->CallStack().Pop();
}

EXPORT_C void NoAppContext()
{
	Dll::SetTls(0);
}

EXPORT_C void MoveIntoDataDirL(MApp_context_access& Context, const TDesC& FileName)
{
	TFileName appfile, datafile;

	appfile.Format(_L("%S%S"), &Context.AppDir(), &FileName);
	datafile.Format(_L("%S%S"), &Context.DataDir(), &FileName);
	if (BaflUtils::FileExists(Context.Fs(), appfile)) {
		User::LeaveIfError(BaflUtils::CopyFile(Context.Fs(), appfile, datafile));
		User::LeaveIfError(BaflUtils::DeleteFile(Context.Fs(), appfile));
	}
}