/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#pragma warning(disable: 4706)

#include "ver.h"

#include "connectioninit.h"
#include <cdbcols.h>
#include <commdb.h>
#include "pointer.h"


const TInt CConnectionOpener::MAX_RETRIES=10;

TInt KeepAlive(TAny* Ptr);

#ifndef __S60V2__
EXPORT_C CConnectionOpener* CConnectionOpener::NewL(MSocketObserver& Observer, RSocketServ& Serv)
#else
EXPORT_C CConnectionOpener* CConnectionOpener::NewL(MSocketObserver& Observer, RSocketServ& Serv, RConnection& Connection)
#endif
{
#ifndef __S60V2__
	auto_ptr<CConnectionOpener> ret(new (ELeave) CConnectionOpener(Observer, Serv));
#else
	auto_ptr<CConnectionOpener> ret(new (ELeave) CConnectionOpener(Observer, Serv, Connection));
#endif
	ret->ConstructL();
	return ret.release();
}

void CConnectionOpener::ConstructL()
{
	iWait=CTimeOut::NewL(*this);
	CActiveScheduler::Add(this);
}

#ifndef __S60V2__
CConnectionOpener::CConnectionOpener(MSocketObserver& Observer, RSocketServ &Serv) : 
CCheckedActive(EPriorityIdle, _L("ContextCommon::CConnectionOpener")), iObserver(Observer), iServ(Serv)
#else
CConnectionOpener::CConnectionOpener(MSocketObserver& Observer, RSocketServ &Serv, RConnection& Connection) : 
CCheckedActive(EPriorityIdle, _L("ContextCommon::CConnectionOpener")), iObserver(Observer), iServ(Serv), iConnection(Connection)
#endif
{
}

EXPORT_C void CConnectionOpener::MakeConnectionL(TUint32 IapID)
{
	TBuf<40> msg;
	iMadeConnection=false;
	msg.Format(_L("MakeConnectionL, iap %d"), IapID);
	iObserver.info(this, msg);
	
	iIapId=IapID;
	
#ifndef __S60V2__
	Cancel();

	if (!iInitiator) iInitiator=CIntConnectionInitiator::NewL();
	TUint32 activeiap;
	if (iInitiator->GetActiveIap(activeiap)==KErrNone && activeiap==IapID) {
		iObserver.info(this, _L("connection exists"));
		iObserver.success(this);
		return;
	}
	
	iConnPref.iRanking = 1; 
	iConnPref.iDirection = ECommDbConnectionDirectionOutgoing; 
	iConnPref.iDialogPref = ECommDbDialogPrefDoNotPrompt; 
	CCommsDbConnectionPrefTableView::TCommDbIapBearer bearer; 
	bearer.iBearerSet = KMaxTUint32;
	bearer.iIapId = IapID; 
	iConnPref.iBearer = bearer;
	
	// we cannot really know if the initiator is usable
	// at this point (if retrying/reusing). Let's just
	// recreate it
	delete iInitiator; iInitiator=0;
	iInitiator=CIntConnectionInitiator::NewL();
	
	TInt ret=iInitiator->TerminateActiveConnection(iStatus);
	if (ret==KErrNone) {
		current_state=CLOSING;
		SetActive();
	} else {
		current_state=CONNECTING;
		iInitiator->ConnectL(iConnPref, iStatus);
		SetActive();
#  if defined(__WINS__)
		// we don't seem to get the right return on the
		// emulator :-(
		iStatus=KErrNone;
		return;
#  endif
	}
#else
	/*
	auto_ptr<CCommsDatabase> db(CCommsDatabase::NewL(EDatabaseTypeIAP));
	db->SetGlobalSettingL(TPtrC(ASK_USER_BEFORE_DIAL),(TInt)false);
	*/

	iConnPref.SetIapId(IapID);
	iConnPref.SetDirection(ECommDbConnectionDirectionOutgoing);
	iConnPref.SetDialogPreference(ECommDbDialogPrefDoNotPrompt);
	iConnPref.SetBearerSet(ECommDbBearerUnknown);
	
	TBool connected = EFalse;
	
	TUint connectionCount;
	//Enumerate currently active connections across all socket servers
	User::LeaveIfError(iConnection.EnumerateConnections(connectionCount));
	
	if (connectionCount)
	{
		TPckgBuf<TConnectionInfo> connectionInfo;
		for (TUint i = 1; i <= connectionCount; ++i)
		{
			iConnection.GetConnectionInfo(i, connectionInfo);
			msg.Format(_L("existing conn, iap %d"), connectionInfo().iIapId);
			iObserver.info(this, msg);
			
			if (connectionInfo().iIapId == IapID)
			{
				RConnection tmp;
				
				TInt ret;
				User::LeaveIfError(tmp.Open(iServ));
				CleanupClosePushL(tmp);
				ret=tmp.Attach(connectionInfo, RConnection::EAttachTypeNormal);
				if (ret==KErrNone) {
					iConnection.Close();
					User::LeaveIfError(iConnection.Open(iServ));
					ret=iConnection.Attach(connectionInfo, RConnection::EAttachTypeNormal);
				}
				CleanupStack::PopAndDestroy(); //tmp
				if (ret==KErrNone) {
					connected = ETrue;
				} else {
					msg.Format(_L("Error in Attach %d"), ret);
					iObserver.info(this, msg);
				}
				break;
			}
			
		}
	}
	
	if (connected) {
		iMadeConnection=false;
		iWait->Reset();
		iObserver.info(this, _L("already connected"));
		iObserver.success(this);
	} else {
		Cancel();

		iStatus=KRequestPending;
		iConnection.Start(iConnPref, iStatus);
		SetActive();
		current_state=CONNECTING;
#  ifdef __WINS__
		iWait->Wait(10);
#  else
		iWait->Wait(30);
#  endif
	}	
#endif
}

void CConnectionOpener::expired(CBase*)
{
	++iRetryCount;
	if (iRetryCount>MAX_RETRIES) {
		iObserver.error(this, -1003, _L("Opener retries exceeded"));
	} else {
		iObserver.info(this, _L("Conn retry"));
		current_state=IDLE;
		TRAPD(err, MakeConnectionL(iIapId));
		if (err!=KErrNone) {
			iObserver.error(this, err, _L("Opener retry failed"));
		}
	}
}

void CConnectionOpener::CheckedRunL()
{
	TBuf<50> msg;
	iWait->Reset();
	
#ifndef __S60V2__
	if ( (iStatus==KConnectionTerminated || iStatus==0)
		&& current_state==CLOSING) {
		iObserver.info(this, _L("prev conn closed"));
		// The initiator seems to close its handles after a TerminateActiveConnection
		// so it has to be recreated if we want to call some other methods
		delete iInitiator; iInitiator=0;
		iInitiator=CIntConnectionInitiator::NewL();
		iInitiator->ConnectL(iConnPref, iStatus);
		current_state=CONNECTING;
		SetActive();
		return;
	}
	
	if (iStatus==CONNECTED) return;
	
	if (iStatus!=KErrNone && iStatus!=KConnectionPref1Exists &&
		iStatus!=KConnectionPref1Created) {
		if (iRetryCount>=MAX_RETRIES) {
			msg.Format(_L("Opener error %d"), iStatus.Int());
			iObserver.error(this, iStatus.Int(), msg);
			return;
		} else {
			msg.Format(_L("Opener error %d (Retry)"), iStatus.Int());
			iObserver.info(this, msg);
			if (current_state==CONNECTING) {
				current_state=RETRYING_CONNECT;
			} else {
				current_state=RETRYING_CLOSE;
			}
			
			// it seems that we sometimes get an error even
			// though the connection gets established. We wait
			// for 15 secs, which should be enough to detect
			// the new connection
			iWait->Wait(15);
			return;
		}
	}
	
	// If the connection doesn't exist the initiator sends *2* requestcompletes:
	// one with KConnectionPref1Exists and one with KConnectionPref1Created
	// if it does only KConnectionPref1Exists is send.
	
	if (current_state==CONNECTING) {
		iMadeConnection=true;
		iObserver.success(this);
		current_state=CONNECTED;
	}
	
	if (iStatus==KConnectionPref1Exists) {
		iStatus=KRequestPending;
		SetActive();
	}
#else
	msg.Format(_L("ConnectionOpener CheckedRunL state %d status %d"), current_state, iStatus.Int());
	iObserver.info(this, msg);
	
	if (iStatus==CONNECTED) {
		return;
	}

	if (iStatus==KErrNone || iStatus==KErrAlreadyExists) {
		if (current_state==CONNECTING) {
			iMadeConnection=true;
			iObserver.success(this);
			current_state=CONNECTED;
		}
	} else {
		msg.Format(_L("Opener error %d (Retry)"), iStatus.Int());
		iObserver.info(this, msg);
		if (current_state==CONNECTING) {
			current_state=RETRYING_CONNECT;
		} else {
			current_state=RETRYING_CLOSE;
		}
		
		// it seems that we sometimes get an error even
		// though the connection gets established. We wait
		// for 15 secs, which should be enough to detect
		// the new connection
		iWait->Wait(15);
		return;
	}
#endif
}

EXPORT_C bool CConnectionOpener::MadeConnection()
{
	return iMadeConnection;
}

void CConnectionOpener::DoCancel()
{
#ifndef __S60V2__
	iInitiator->Cancel();
#else
	iConnection.Close();
	TRequestStatus *s=&iStatus;
	User::RequestComplete(s, KErrCancel);
	iConnection.Open(iServ);
#endif
}

TInt CConnectionOpener::CheckedRunError(TInt aError)
{
	if (current_state==CONNECTING) {
		TBuf<50> msg;
		msg.Format(_L("CConnectionOpener::CheckedRunError %d"), aError);
		iObserver.error(this, aError, msg);
	}
	iWait->Reset();
	return KErrNone;
}

EXPORT_C void CConnectionOpener::CloseConnection()
{
	
	iMadeConnection=false;
	iWait->Reset();
	iRetryCount=0;
	Cancel();
#ifndef __S60V2__
	TRAPD(err,
		if (!iInitiator) iInitiator=CIntConnectionInitiator::NewL();
		iInitiator->TerminateActiveConnection();
		);
		delete iInitiator; iInitiator=0;
#else
		iConnection.Stop();
#endif
		current_state=IDLE;
}

EXPORT_C CConnectionOpener::~CConnectionOpener()
{
	Cancel();
	delete iWait;
#ifndef __S60V2__
	if (iInitiator && iMadeConnection) iInitiator->TerminateActiveConnection();
	delete iInitiator; iInitiator=0;
#else
	iConnection.Close();
#endif
}

class CKeepAlive : public CCheckedActive {
public:
	void Stop() {
		Cancel();
	}
	static CKeepAlive* NewL() {
		auto_ptr<CKeepAlive> ret(new (ELeave) CKeepAlive);
		ret->ConstructL();
		return ret.release();
	}
	~CKeepAlive() {
		Cancel();
		if (iSocketOpen) socket.Close();
		if (iServOpen) serv.Close();
	}
private:
	CKeepAlive() : CCheckedActive(EPriorityIdle, _L("CKeepAlive")), iShortWait(50*1000), iLongWait(10*1000*1000) { }
	void ConstructL() {
		timer.CreateLocal();
		CActiveScheduler::Add(this);
		iCurrentState=EServConnect;
		timer.After(iStatus, iShortWait);
		SetActive();
	}
	void DoCancel() {
		switch (iCurrentState) {
		case EServConnect:
		case ESocketOpen:
		case ESocketDoConnect:
		case ESocketWait:
			timer.Cancel();
			break;
		case ESocketConnect:
			socket.CancelConnect();
			break;
		case ESocketClose:
			break;
		}
	}
	
	void CheckedRunL() {
		TBuf<50> msg;
		msg.Format(_L("CKeepAlive CheckedRunL state %d status %d"), (TInt)iCurrentState, iStatus.Int() );
		RDebug::Print(msg);
		
		TWait wait=EShort;
		switch (iCurrentState) {
		case EServConnect:
			if (serv.Connect()== KErrNone) {
				iServOpen=true;
				iCurrentState=ESocketOpen;	
			} else {
				wait=ELong;
			}
			break;
		case ESocketOpen:
			if (socket.Open(serv, KAfInet, KSockStream, KProtocolInetTcp) == KErrNone) {
				iSocketOpen=true;
				iCurrentState=ESocketDoConnect;
			} else {
				wait=ELong;
			}
			break;
		case ESocketDoConnect:
			{
				TInetAddr www(INET_ADDR(128, 214, 48, 81), 80);
				socket.Connect(www, iStatus);
				iCurrentState=ESocketConnect;
				wait=ENone;
			}
			break;
		case ESocketConnect:
			if (iStatus.Int()==KErrNone) {
				socket.Shutdown(RSocket::ENormal, iStatus);
				wait=ENone;
			} else {
				socket.Close();
				iSocketOpen=false;
				serv.Close();
				iServOpen=false;
				iCurrentState=EServConnect;
				wait=ELong;
			}
			break;
		case ESocketClose:
			iCurrentState=ESocketWait;
			socket.Close(); iSocketOpen=false;
			wait=ELong;
			break;
		case ESocketWait:
			iCurrentState=ESocketOpen;
			wait=EShort;
			break;
		}
		switch(wait) {
		case EShort:
			timer.After(iStatus, iShortWait);
			break;
		case ELong:
			timer.After(iStatus, iLongWait);
			break;
		case ENone:
			break;
		}
		SetActive();
	}
	
	
	enum TState { EIdle, EServConnect, ESocketOpen, ESocketDoConnect, 
		ESocketConnect, ESocketClose, ESocketWait };
	enum TWait { ENone, EShort, ELong };
	TState		iCurrentState;
	
	TTimeIntervalMicroSeconds32 iShortWait, iLongWait;
	
	RSocketServ serv; bool iServOpen;
	RSocket socket; bool iSocketOpen;
	RTimer timer;
};

class CWorker : public CCheckedActive {
public:
	static CWorker* NewL(CKeepAlive* keep) {
		auto_ptr<CWorker> ret(new (ELeave) CWorker);
		ret->ConstructL(keep);
		return ret.release();
	}
	TRequestStatus* GetRequestStatus() {
		return &iStatus;
	}
	~CWorker() { }
private:
	void CheckedRunL() {
		RDebug::Print(_L("CWorker::stop"));
		iKeep->Stop();
		CActiveScheduler::Stop();
	}
	void DoCancel() { }
	
	CWorker() : CCheckedActive(EPriorityNormal, _L("CWorker")) { }
	void ConstructL(CKeepAlive* keep) {
		iKeep=keep;
		CActiveScheduler::Add(this);
		SetActive();
	}
	CKeepAlive* iKeep;
};

void DoKeepAlive(TAny* aPtr)
{
	RDebug::Print(_L("DoKeepAlive"));
	
	CActiveScheduler* c=new (ELeave) CActiveScheduler;
	CleanupStack::PushL(c);
	CActiveScheduler::Install(c);
	
	CKeepAlive* keep=CKeepAlive::NewL();
	CleanupStack::PushL(keep);
	
	worker_info* wi=(worker_info*)aPtr;
	CWorker* w=CWorker::NewL(keep);
	CleanupStack::PushL(w);
	wi->set_do_stop(w->GetRequestStatus());
	
	RDebug::Print(_L("DoKeepAlive:Start"));
	CActiveScheduler::Start();
	RDebug::Print(_L("DoKeepAlive:Stop"));
	
	CleanupStack::PopAndDestroy(3);
}

TInt KeepAlive(TAny* aPtr)
{
	RDebug::Print(_L("KeepAlive"));
	
        CTrapCleanup *cl;
        cl=CTrapCleanup::New();
	
        TInt err=0;
        TRAP(err,
                DoKeepAlive(aPtr));
	
	delete cl;
	
	TTimeIntervalMicroSeconds32 w(50*1000);
	User::After(w);
	worker_info* wi=(worker_info*)aPtr;
	wi->stopped();
        return err;
}
