/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "db.h"

#include <bautils.h>
#include <e32std.h>
#include "symbian_auto_ptr.h"

const int COMPACT_BETWEEN_UPDATES=20;
const int COMPACT_BETWEEN_DELETES=20;

EXPORT_C CDb* CDb::NewL(MApp_context& Context, const TDesC& dbname, TInt aFileMode, bool shared)
{
	auto_ptr<CDb> ret(new (ELeave) CDb(Context, shared));
	ret->ConstructL(dbname, aFileMode);
	return ret.release();
}

EXPORT_C CDb::~CDb()
{
	TInt err;
	if (iDbOpen) { 
		TRAP(err, iDb.Compact());
		TRAP(err, iDb.Close());
	}
	delete iStore;
}

EXPORT_C RDbDatabase& CDb::Db()
{
	return iDb;
}

CDb::CDb(MApp_context& Context, bool shared) : MContextBase(Context), iShared(shared)
{
}

void CDb::ConstructL(const TDesC& dbname, TInt aFileMode)
{
	TFileName nameonly;
	nameonly.Format(_L("%S.db"), &dbname);
	MoveIntoDataDirL(AppContext(), nameonly);

	TFileName database_file_install;
	TFileName database_file;

	database_file_install.Format(_L("%S%S_inst.db"), &AppDir(), &dbname);
	database_file.Format(_L("%S%S.db"), &DataDir(), &dbname);

	if (BaflUtils::FileExists(Fs(), database_file_install)) {
		User::LeaveIfError(BaflUtils::CopyFile(Fs(), database_file_install, database_file));
		User::LeaveIfError(BaflUtils::DeleteFile(Fs(), database_file_install));
	}

	TInt store_exists;
	TRAP(store_exists, iStore = CPermanentFileStore::OpenL(Fs(), database_file, aFileMode));
	
	//TRAP (store_exists, iStore = ::OpenL(Fs(), database_file, EFileRead|EFileWrite));

	if (store_exists==KErrNone) { 
		iDb.OpenL(iStore, iStore->Root());
		User::LeaveIfError(iDb.Recover());
		iDbOpen=true;
	} else {
		// construct database
		iStore = CPermanentFileStore::ReplaceL(Fs(), database_file,aFileMode);
		iStore->SetTypeL(iStore->Layout());
		TStreamId id=iDb.CreateL(iStore);
		iDbOpen=true;
		iStore->SetRootL(id);
		iStore->CommitL();
	}
	/*
	 * named databases not supported?
	 *
	if (BaflUtils::FileExists(Fs(), database_file)) {
		TInt err=iDb.Open(Fs(), database_file);
		User::LeaveIfError(err);
	} else {
		TInt err=iDb.Create(Fs(), database_file);
		User::LeaveIfError(err);
	}
	*/
}

EXPORT_C MDBStore::MDBStore(RDbDatabase& Db) : iDb(Db)
{
}

EXPORT_C MDBStore::~MDBStore()
{
	TInt err;
	if (iTableOpen) { TRAP(err, iTable.Close()); }
}

EXPORT_C void MDBStore::ConstructL(int* columns, int* idx_columns, bool unique_idx,
			  const TDesC& name)
{
	iCurrentIdx=-1;

	TInt table_exists=iTable.Open(iDb, name);

	if (table_exists!=KErrNone) {
		CDbColSet* cols=CDbColSet::NewLC();

		TBuf<10> colname; int colcount=1;
		for (int* col_i=columns; *col_i!=-1; col_i++) {
			colname.Format(_L("C%d"), colcount);
			colcount++;
			TDbCol n(colname, (TDbColType)*col_i);
			if (n.iType==EDbColText && iTextLen!=0) n.iMaxLength=iTextLen;
			cols->AddL(n);
		}

		User::LeaveIfError(iDb.CreateTable(name, *cols));
		CleanupStack::PopAndDestroy(); // cols

		CDbKey* idx_key=CDbKey::NewLC();
		TInt idx_count=0;

		for (int* idx_i=idx_columns; ; idx_i++) {
			if (*idx_i<0) {
				if (unique_idx && idx_count==0) 
					idx_key->MakeUnique();
				TBuf<30> idx_name;
				if (idx_count>0) idx_name.Format(_L("IDX%d"), idx_count);
				else idx_name=_L("IDX");
				User::LeaveIfError(iDb.CreateIndex(idx_name, name, *idx_key));
				idx_key->Clear();
				++idx_count;

			} else {
				colname.Format(_L("C%d"), *idx_i);
				idx_key->AddL(TDbKeyCol(colname));
			}
			if (*idx_i==-1) break;
		}

		CleanupStack::PopAndDestroy(); // idx_key

		User::LeaveIfError(iTable.Open(iDb, name));
		iTableOpen=true;
	} else {
		iTableOpen=true;
	}

	SwitchIndexL(0);
}

EXPORT_C void MDBStore::SwitchIndexL(TInt Idx)
{
	if (iCurrentIdx==Idx) return;

	TBuf<30> idx_name;
	if (Idx==-1) {
		User::LeaveIfError(iTable.SetNoIndex());
		iCurrentIdx=Idx;
		return;
	}
	if (Idx>0) idx_name.Format(_L("IDX%d"), Idx);
	else idx_name=_L("IDX");
	User::LeaveIfError(iTable.SetIndex(idx_name));
	iCurrentIdx=Idx;
}

EXPORT_C TInt MDBStore::GetCurrentIndex()
{
	return iCurrentIdx;
}


EXPORT_C void MDBStore::PutL()
{
	iTable.PutL();
	if (iUpdateCount==COMPACT_BETWEEN_UPDATES) {
		iUpdateCount=0;
		iDb.Compact();
	}
	++iUpdateCount;
}

EXPORT_C void MDBStore::SetTextLen(TInt Len)
{
	iTextLen=Len;
}

CSingleColDbBase::CSingleColDbBase(MApp_context& Context, RDbDatabase& Db) : MDBStore(Db), MContextBase(Context)
{
}

CSingleColDbBase::~CSingleColDbBase()
{
}

bool CSingleColDbBase::SeekL(TInt Idx, bool Updatable, bool Add)
{
	TDbSeekKey rk(Idx);
	if (iTable.SeekL(rk) ) {
		iTable.GetL();
		if (Updatable) iTable.UpdateL();
		return true;
	} else if (Add) {
		if (!Updatable) User::Leave(-1001);
		iTable.InsertL();
		iTable.SetColL(1, Idx);
		return true;
	}
	return false;
}

void CSingleColDbBase::ConstructL(const TDesC& TableName,
	int ColType)
{
	int columns[] = { EDbColUint32, 0, -1 };
	columns[1]=ColType;
	int id_cols[]= { 1, -1 };

	MDBStore::ConstructL(columns, id_cols, true, TableName);
}
