/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


//CContextServerSession.cpp

#include "ContextServerSession.h"
#include <ContextCommon.h>
#include <e32svr.h>
#include <flogger.h>
#include <symbian_auto_ptr.h>

bool ClientAlive(TInt ThreadId)
{
	if (ThreadId==0) return false;

	RThread c; bool ret = false;
	if (c.Open(ThreadId) != KErrNone) return false;
	if (c.ExitType() == EExitPending) ret=true;
	c.Close();
	return ret;
}

CContextServerSession::CContextServerSession(RThread& aClient, CContextServer& aServer) : CSession(aClient), iServer(aServer)
{
    
}

CContextServerSession::~CContextServerSession()
{
    	iServer.DecrementSessions();
}

void CContextServerSession::CompleteMessage(TInt Code)
{
	if (ClientAlive(iMessageThreadId)) {
		iMessage.Complete(Code);
	}
	SetMessage(RMessage());
}

void CContextServerSession::ServiceL(const RMessage& aMessage)
{
	if (aMessage.Function()!=ECancel) SetMessage(aMessage);

	switch (aMessage.Function())
	{
		case ETerminateContextServer:
			CompleteMessage(ERequestCompleted);
			iServer.TerminateContextServer();
			break;
			
		case EConnectToPresenceServer:
			ConnectToPresenceServer(aMessage);
			break;

		case ERequestPresenceInfo:
			HandleRequestPresenceInfo(aMessage);
			break;

		case ERequestPresenceNotification:
			HandleRequestPresenceNotification(aMessage);
			break;
		case ERequestMessageNotification:
			HandleRequestMessageNotification(aMessage);
			break;

		case EUpdateUserPresence:
			UpdateUserPresence(aMessage);
			break;

		case ESuspendConnection:
			SuspendConnection(aMessage);
			break;

		case EResumeConnection:
			ResumeConnection(aMessage);
			break;
		
		case ECancel:
			{
			RMessage prev=iMessage;
			CompleteMessage(KErrCancel);
			iServer.CancelRequest(prev);
			aMessage.Complete(0);
			break;
			}

		default :
            	PanicClient(EBadRequest);
	}

}

void CContextServerSession::PanicClient(TInt aPanic) const
{
	Panic(KContextServer, aPanic) ; // Note: this panics the client thread, not server
}

CContextServerSession* CContextServerSession::NewL(RThread& aClient, CContextServer& aServer)
{
	CContextServerSession* self = CContextServerSession::NewLC(aClient, aServer) ;
    	CleanupStack::Pop(self) ;
    	return self ;
}

CContextServerSession* CContextServerSession::NewLC(RThread& aClient, CContextServer& aServer)
{
	CContextServerSession* self = new (ELeave) CContextServerSession(aClient, aServer);
    	CleanupStack::PushL(self) ;
    	self->ConstructL() ;
    	return self ;
}

void CContextServerSession::ConstructL()
{
	iServer.IncrementSessions();
}

//------------------------------------------------------------------------

void CContextServerSession::TerminateContextServer(const RMessage& aMessage)
{
	CompleteMessage(ERequestCompleted);
	iServer.TerminateContextServer();
}

void CContextServerSession::ConnectToPresenceServer(const RMessage& aMessage)
{
	if (iServer.GetJabberClient()->Connected())
	{
		CompleteMessage(ERequestCompleted);
	}
	else
	{
		iServer.ConnectToPresenceServer(aMessage);
	}
}

void CContextServerSession::SendNewMessage(TMessage aJabberMessage)
{
	if (aJabberMessage.contact==0 || aJabberMessage.message==0 || aJabberMessage.subject==0) return;

	TInt len0 = iMessage.Client().GetDesMaxLength(iMessage.Ptr0());
	TInt len1 = iMessage.Client().GetDesMaxLength(iMessage.Ptr1());
	TInt len2 = iMessage.Client().GetDesMaxLength(iMessage.Ptr2());

	if (len0 < 0 || len1 < 0 || len2 < 0) {
		CompleteMessage(KErrBadHandle);
		return;
	}

	if ( len0 < aJabberMessage.contact->Des().Length() || len1 < aJabberMessage.subject->Des().Length() 
		|| len2 < aJabberMessage.message->Des().Length())
	{		
		TPckgC<TInt> len0Package(aJabberMessage.contact->Des().Length());
		iMessage.WriteL(iMessage.Ptr0(),len0Package,0);
		
		TPckgC<TInt> len1Package(aJabberMessage.subject->Des().Length());
		iMessage.WriteL(iMessage.Ptr1(),len1Package,0);

		TPckgC<TInt> len2Package(aJabberMessage.message->Des().Length());
		iMessage.WriteL(iMessage.Ptr1(),len2Package,0);

		CompleteMessage(EBufferTooSmall);

		iServer.GetMessageHolder()->AppendMessageL(aJabberMessage);
	}
	else
	{
		iMessage.WriteL(iMessage.Ptr0(),*aJabberMessage.contact,0);
		iMessage.WriteL(iMessage.Ptr1(),*aJabberMessage.subject,0);
		iMessage.WriteL(iMessage.Ptr2(),*aJabberMessage.message,0);
		
		CompleteMessage(ERequestCompleted);
	}	
}

void CContextServerSession::SendPresenceInfo(const TPresenceInfo& info)
{
	if (!ClientAlive(iMessageThreadId)) return;

	TInt len0 = iMessage.Client().GetDesMaxLength(iMessage.Ptr0());
	TInt len1 = iMessage.Client().GetDesMaxLength(iMessage.Ptr1());

	if (len0 < 0 || len1 < 0 ) {
		CompleteMessage(KErrBadHandle);
		return;
	}

	if ( len0 < info.contact.Length() || len1 < info.info.Length() )
	{
		iServer.GetPresenceInfo()->SetAsNew(info.contact);
		
		TPckgC<TInt> len0Package(info.contact.Length());
		iMessage.WriteL(iMessage.Ptr0(),len0Package,0);
		
		TPckgC<TInt> len1Package(info.info.Length());
		iMessage.WriteL(iMessage.Ptr1(),len1Package,0);
		CompleteMessage(EBufferTooSmall);
	}
	else
	{
		iMessage.WriteL(iMessage.Ptr0(),info.contact,0);
		iMessage.WriteL(iMessage.Ptr1(),info.info,0);
		
		TPckgC<TTime> package(info.send_timestamp);
		iMessage.WriteL(iMessage.Ptr2(),package,0);

		CompleteMessage(ERequestCompleted);
	}	
}

void CContextServerSession::HandleRequestPresenceInfo(const RMessage& aMessage)
{
	if ( !(iServer.GetJabberClient()->Connected()) )
	{
		CompleteMessage(EServerUnreachable);
		return;
	}
		
	iServer.GetPresenceInfo()->SetAllAsNew();

	HandleRequestPresenceNotification(aMessage);
}

void CContextServerSession::HandleRequestMessageNotification(const RMessage& aMessage)
{
	TMessage m(iServer.GetMessageHolder()->GetMessage());
	if (m.subject && m.message && m.contact) {
		SendNewMessage(m);
	}
}


void CContextServerSession::SetMessage(const RMessage& aMessage) 
{
	if (aMessage==RMessage()) {
		iMessageThreadId=0;
	} else {
		iMessageThreadId=aMessage.Client().Id();
	}
	iMessage=aMessage;
}

void CContextServerSession::HandleRequestPresenceNotification(const RMessage& aMessage)
{
	TPresenceInfo info = iServer.GetPresenceInfo()->GetNewPresenceInfo();

	if (info.newInfo == 1)
	{
		SendPresenceInfo(info);
	}
}

void CContextServerSession::NotifyEvent(CContextServer::TEvent aEvent)
{
	if (!ClientAlive(iMessageThreadId)) return;

	if (aEvent==CContextServer::EDisconnected) {
		CompleteMessage(EServerUnreachable);
	} else if (aEvent==CContextServer::ETerminated) {
		CompleteMessage(EContextServerTerminated);
	} else if (aEvent==CContextServer::EConnected && iMessage.Function()==EConnectToPresenceServer) {
		CompleteMessage(ERequestCompleted);
	} else if (aEvent==CContextServer::ENewPresenceInfo && (
		iMessage.Function()==ERequestPresenceNotification ||
		iMessage.Function()==ERequestPresenceInfo) ) {
		SendPresenceInfo(iServer.GetPresenceInfo()->GetNewPresenceInfo());
	} else if (aEvent==CContextServer::ENewMessage && iMessage.Function()==ERequestMessageNotification) {
		SendNewMessage(iServer.GetMessageHolder()->GetMessage());
	}
}

void CContextServerSession::DisconnectFromPresenceServer(const RMessage& aMessage)
{
	CompleteMessage(ERequestCompleted);
	iServer.SuspendConnection();
}

void CContextServerSession::UpdateUserPresence(const RMessage & aMessage)
{
	TInt len=Client().GetDesLength(aMessage.Ptr0());
	auto_ptr<HBufC> tempPresenceInfo(HBufC::NewL(len));
	TPtr16 bufd=tempPresenceInfo->Des();
	
	ReadL(aMessage.Ptr0(), bufd,0);

	CompleteMessage(ERequestCompleted);
	iServer.GetJabberClient()->SendPresenceInfoL(bufd);
}

void CContextServerSession::SuspendConnection(const RMessage & aMessage)
{
	CompleteMessage(ERequestCompleted);
	iServer.SuspendConnection();
}

void CContextServerSession::ResumeConnection(const RMessage & aMessage)
{
	iServer.ResumeConnection();
}

void CContextServerSession::ReportError(TContextServRqstComplete aErrorType, 
					TDesC & /*aErrorCode*/, TDesC & /*aErrorValue*/)
{
	CompleteMessage(aErrorType);
}