/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "MessageHolder.h"
#include <symbian_auto_ptr.h>
#include <list.h>

TMessage_ref::TMessage_ref(HBufC* i_contact, HBufC* i_subject, HBufC* i_message) :
	contact(i_contact), subject(i_subject), message(i_message)
{
}

TMessage::operator TMessage_ref()
{
	HBufC *c=contact, *s=subject, *m=message;
	contact=subject=message=0;
	TMessage_ref r(c, s, m);
	return r;
}

TMessage::TMessage() : contact(0), subject(0), message(0) { }
TMessage::~TMessage() 
{ 
	delete contact; 
	delete subject;
	delete message; 
}

TMessage& TMessage::operator=(TMessage& rhs)
{
	if (&rhs != this) {
		delete contact; contact=0;
		delete message; message=0;
		delete subject; subject=0;
		contact=rhs.contact; rhs.contact=0;
		message=rhs.message; rhs.message=0;
		subject=rhs.subject; rhs.subject=0;
	}
	return *this;
}

TMessage::TMessage(TMessage& f)
{
	contact=f.contact; f.contact=0;
	message=f.message; f.message=0;
	subject=f.subject; f.subject=0;
}

TMessage& TMessage::operator=(const TMessage_ref& rhs)
{
	delete contact; contact=0;
	delete message; message=0;
	delete subject; subject=0;
	contact=rhs.contact;
	message=rhs.message;
	subject=rhs.subject;
	return *this;
}

TMessage::TMessage(const TMessage_ref& f)
{
	contact=f.contact;
	message=f.message;
	subject=f.subject;
}

class CMessageHolderImpl : public CMessageHolder
{
public: 
	virtual ~CMessageHolderImpl();
private:
	void ConstructL();

	virtual TMessage GetMessage(); // caller gets ownership
	virtual void AppendMessageL(const TDesC& contact, const TDesC& subject, const TDesC& message);
	virtual void AppendMessageL(TMessage msg); // takes ownership

	CList<TMessage>* iList;

	friend class CMessageHolder;
};

CMessageHolder * CMessageHolder::NewL()
{
	auto_ptr<CMessageHolderImpl> ret(new (ELeave) CMessageHolderImpl);
	ret->ConstructL();
	return ret.release();
}

CMessageHolderImpl::~CMessageHolderImpl()
{
	delete iList;
}

void CMessageHolderImpl::ConstructL()
{
	iList=CList<TMessage>::NewL();
}

TMessage CMessageHolderImpl::GetMessage()
{
	TMessage m(iList->Pop());
	return m;
}

void CMessageHolderImpl::AppendMessageL(const TDesC& contact, const TDesC& subject, const TDesC& message)
{
	auto_ptr<HBufC> c(contact.AllocL());
	auto_ptr<HBufC> msg(message.AllocL());
	auto_ptr<HBufC> s(subject.AllocL());

	TMessage m;
	m.contact=c.get();
	m.message=msg.get();
	m.subject=s.get();

	iList->AppendL(m);

	c.release(); msg.release(); s.release();
}

void CMessageHolderImpl::AppendMessageL(TMessage msg)
{
	iList->AppendL(msg);
}
