/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


/* Copyright (c) 2001, Nokia. All rights reserved */

#include "Sockets.pan"
#include "SocketsReader.h"
#include "EngineNotifier.h"
#include <flogger.h>

void Log(const TDesC& msg);
void Log(const TDesC8& msg);
void Log(const TDesC& msg, TInt);

const TInt CSocketsReader::KReadTimeOut = 600;


CSocketsReader* CSocketsReader::NewL(MEngineNotifier& aEngineNotifier, RSocket& aSocket)
{
	CSocketsReader* self = CSocketsReader::NewLC(aEngineNotifier, aSocket);
	CleanupStack::Pop(self);
	return self;
}


CSocketsReader* CSocketsReader::NewLC(MEngineNotifier& aEngineNotifier, RSocket& aSocket)
{
	CSocketsReader* self = new (ELeave) CSocketsReader(aEngineNotifier, aSocket);
	CleanupStack::PushL(self);
	self->ConstructL();
	return self;
}


CSocketsReader::CSocketsReader(MEngineNotifier& aEngineNotifier, RSocket& aSocket)
: CActive(EPriorityStandard),
iSocket(aSocket),
iEngineNotifier(aEngineNotifier),
issueRead(ETrue)
{
}


CSocketsReader::~CSocketsReader()
    {
    Cancel();
	delete iReadTimer;
    }


void CSocketsReader::SetIssueRead(TBool issue)
{
	issueRead = issue;
}


void CSocketsReader::ConstructL()
{
	iReadTimer = CTimeOut::NewL(*this);
	CActiveScheduler::Add(this);
}

void CSocketsReader::DoCancel()
{
	// Cancel asychronous read request
	iReadTimer->Reset();
	iSocket.CancelRead();
}

TInt CSocketsReader::CheckedRunError(TInt aError)
{
	Log(_L("Error in CSocketsReader::RunL %d"), aError);
	return aError;
}

void CSocketsReader::expired(CBase*)
{
	Cancel();
	iEngineNotifier.ReportError(MEngineNotifier::ETimeOutOnRead, 0);
}

void CSocketsReader::RunL()
{
	iReadTimer->Reset();
	
	// Active object request complete handler
	switch (iStatus.Int())
        {
        case KErrNone:
		// Character has been read from socket
		iEngineNotifier.ResponseReceived(iBuffer);
		if (issueRead)
		{ 
			// Immediately start another read
			//Log(_L("IssueRead()"));
			IssueRead();
		}
		break;
		
        case KErrDisconnected:
		iEngineNotifier.ReportError(MEngineNotifier::EDisconnected,iStatus.Int());
		break;
		
        default:
		iEngineNotifier.ReportError(MEngineNotifier::EGeneralReadError,iStatus.Int());
		break;
        }	
}

void CSocketsReader::IssueRead()
{
	// Initiate a new read from socket into iBuffer
	__ASSERT_ALWAYS(!IsActive(), User::Panic(KPanicSocketsEngineRead, ESocketsBadState));
	iReadTimer->Wait(KReadTimeOut);
	iSocket.RecvOneOrMore(iBuffer, 0, iStatus, iDummyLength);
	SetActive();
}

void CSocketsReader::Start()
{
	// Initiate a new read from socket into iBuffer
	if (!IsActive())
        {
		IssueRead();
        }
}
