#include "camera.h"

#include <cameraserv.h>
#include <list.h>
#include <symbian_auto_ptr.h>
#include <bautils.h>
#include <MdaImageConverter.h>

class CSnapShotImpl : public CSnapShot, public CCheckedActive, public MContextBase, public MMdaImageUtilObserver {
private:
	CSnapShotImpl(MApp_context& Context);
	void ConstructL();
	static CSnapShot* NewL(MApp_context& Context);
	virtual void TakeSnapShot(const TDesC& aDir, MSnapShotNofifier* aNotifier);
	virtual void CancelSnapShot(MSnapShotNofifier* aNotifier);

	virtual void CheckedRunL();
	virtual TInt CheckedRunError(TInt aCode);
	virtual void DoCancel();

	void Async(TInt aError=KErrNone);
	void CreateSaveImage();
	void CloseCamera();
	void Convert();

	virtual void MiuoConvertComplete(TInt aError);
	virtual void MiuoCreateComplete(TInt aError);
	virtual void MiuoOpenComplete(TInt aError);
public:
	~CSnapShotImpl();

private:
	struct TSnapShotRequest {
		TFileName	iDir;
		MSnapShotNofifier* iNotifier;

		TSnapShotRequest(const TDesC& aDir, MSnapShotNofifier* aNotifier) :
		iDir(aDir), iNotifier(aNotifier) { }
		TSnapShotRequest() : iNotifier(0) { }
	};

	CList<TSnapShotRequest>* iRequests;
	
	enum TState { EIdle, EStarting, ETurningOn, EGettingImage, ECreatingFile, EConverting };
	TState iState;
	RCameraServ iCamera; bool iCameraIsOpen; bool iCameraIsOn;
	CFbsBitmap* iBitMap;
	MSnapShotNofifier* iNotifier;
	CMdaImageBitmapToFileUtility* iFileSaver;
	TFileName	iFileName;
	TMdaJfifClipFormat jfifFormat;
	
	friend class CSnapShot;
};

CSnapShot* CSnapShot::NewL(MApp_context& Context)
{
	auto_ptr<CSnapShotImpl> ret(new (ELeave) CSnapShotImpl(Context));
	ret->ConstructL();
	return ret.release();
}

CSnapShotImpl::CSnapShotImpl(MApp_context& Context) : MContextBase(Context), CCheckedActive(EPriorityNormal, _L("CSnapShotImpl"))
{
}

void CSnapShotImpl::ConstructL()
{
	iRequests=CList<TSnapShotRequest>::NewL();
	CActiveScheduler::Add(this);
}

void CSnapShotImpl::TakeSnapShot(const TDesC& aDir, MSnapShotNofifier* aNotifier)
{
	iRequests->AppendL(TSnapShotRequest(aDir, aNotifier));

	if (iState==EIdle) {
		iState=EStarting;
		Async();
	}
	return;
}

CSnapShotImpl::~CSnapShotImpl()
{
	if (iRequests) {
		TSnapShotRequest r;
		for (r=iRequests->Pop(); r.iNotifier; r=iRequests->Pop()) {
			r.iNotifier->Error(KErrCancel, _L("Cancelled"));
		}
	}
	delete iRequests;

	Cancel();

	CloseCamera();

	delete iBitMap;
	delete iFileSaver;
}

void CSnapShotImpl::Async(TInt aError)
{
	TRequestStatus* s=&iStatus;
	User::RequestComplete(s, aError);
	SetActive();
}

void CSnapShotImpl::CloseCamera()
{
	if (iCameraIsOn) iCamera.TurnCameraOff();
	if (iCameraIsOpen) iCamera.Close();
	iCameraIsOn=iCameraIsOpen=false;
}

void CSnapShotImpl::DoCancel()
{
	CloseCamera();
	TRequestStatus* s=&iStatus;
	User::RequestComplete(s, KErrCancel);
}

void CSnapShotImpl::CheckedRunL()
{
	if (iStatus.Int()!=KErrNone) {
		CheckedRunError(iStatus.Int());
		return;
	}

	switch(iState) {
	case EStarting:
		iNotifier=iRequests->iFirst->Item.iNotifier;
		User::LeaveIfError(iCamera.Connect());
		iCameraIsOpen=true;
		iCamera.TurnCameraOn(iStatus);
		iState=ETurningOn;
		iNotifier->Info(_L("Turning Camer On"));
		SetActive();
		break;
	case ETurningOn:
		iCameraIsOn=true;
		delete iBitMap; iBitMap=0;
		iBitMap=new (ELeave) CFbsBitmap;
		iState=EGettingImage;
		iCamera.GetImage(iStatus, *iBitMap);
		iNotifier->Info(_L("Getting Image"));
		SetActive();
		break;
	case EGettingImage:
		iState=ECreatingFile;
		iNotifier->Info(_L("Creating File"));
		CreateSaveImage();
		break;
	case ECreatingFile:
		iState=EConverting;
		iNotifier->Info(_L("Saving File"));
		Convert();
		break;
	case EConverting:
		iNotifier->Taken(iFileName);
		iRequests->Pop();
		delete iBitMap; iBitMap=0;
		delete iFileSaver; iFileSaver=0;
		if (iRequests->iFirst) {
			iState=ETurningOn;
			Async();
		} else {
			iState=EIdle;
			CloseCamera();
		}
		break;
	}
}

void CSnapShotImpl::CreateSaveImage()
{
	TFileName base=iRequests->iFirst->Item.iDir;
	if (base.Right(1).Compare(_L("\\"))) {
		base.Append(_L("\\"));
	}
	base.Append(_L("SnapShot"));
	iFileName=base;
	TInt i=1; iFileName.AppendNum(i); iFileName.Append(_L(".jpg"));
	while (BaflUtils::FileExists(Fs(), iFileName)) {
		iFileName=base;
		++i; iFileName.AppendNum(i); iFileName.Append(_L(".jpg"));
	}

	delete iFileSaver; iFileSaver=0;
	iFileSaver=CMdaImageBitmapToFileUtility::NewL(*this);

	jfifFormat.iSettings.iSampleScheme  = TMdaJpgSettings::EColor420;
	jfifFormat.iSettings.iQualityFactor = 50;

	iFileSaver->CreateL(iFileName, &jfifFormat, NULL, NULL);

}

TInt CSnapShotImpl::CheckedRunError(TInt aCode)
{
	delete iBitMap; iBitMap=0;
	delete iFileSaver; iFileSaver=0;
	TBuf<50> msg=_L("Error taking picture while ");
	switch(iState) {
		case EStarting:
			msg.Append(_L("initializing camera"));
			break;
		case ETurningOn:
			msg.Append(_L("turning camera on"));
			break;
		case EGettingImage:
			msg.Append(_L("getting image"));
			break;
		case ECreatingFile:
			msg.Append(_L("creating file"));
			break;
		case EConverting:
			msg.Append(_L("saving file"));
			break;
		default:
			msg.Append(_L("[unknown]"));
			break;
	};

	iNotifier->Error(aCode, msg);
	CloseCamera();
	iRequests->Pop();
	iState=EIdle;
	return KErrNone;
}

void CSnapShotImpl::CancelSnapShot(MSnapShotNofifier* aNotifier)
{
	CList<TSnapShotRequest>::Node *n=0, *temp=0;
	for (n=iRequests->iFirst; n; ) {
		temp=n->Next;
		if (n->Item.iNotifier==aNotifier) {
			iRequests->DeleteNode(n, true);
		}
		n=temp;
	}
}

void CSnapShotImpl::Convert()
{
	iFileSaver->ConvertL(*iBitMap);
}

void CSnapShotImpl::MiuoConvertComplete(TInt aError)
{
	Async(aError);
}

void CSnapShotImpl::MiuoCreateComplete(TInt aError)
{
	Async(aError);
}

void CSnapShotImpl::MiuoOpenComplete(TInt aError)
{
	Async(aError);
}
