/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "cellmap.h"
#include "symbian_auto_ptr.h"
#include "cl_settings.h"

CCellMap* CCellMap::NewL(MApp_context& Context, RDbDatabase& Db)
{
	CALLSTACKITEM(_L("CCellMap::NewL"));

	auto_ptr<CCellMap> ret(new (ELeave) CCellMap(Context, Db));
	ret->ConstructL();
	return ret.release();
}

void CCellMap::Parse(const TDesC& Cell, TUint& cell, TUint& area, TDes& nw)
{
	CALLSTACKITEM(_L("CCellMap::Parse"));

	TLex lex(Cell);

	// ignore errors, sometimes we have misformed
	// cellids
	lex.Val(area);
	lex.SkipCharacters(); lex.SkipSpace();
	lex.Val(cell);
	lex.SkipCharacters(); lex.SkipSpace();
	nw=lex.Remainder().Left(20);
	if (nw.Length()==0) nw=_L("UNKNOWN");
}

TInt CCellMap::GetId(const TDesC& Cell)
{
	CALLSTACKITEM(_L("CCellMap::GetId"));

	TDbSeekMultiKey<3> rk;
	TUint area=0, cell=0;
	TBuf<20> nw;
	Parse(Cell, cell, area, nw);
	rk.Add(cell); rk.Add(area); rk.Add(nw);

	TInt id;
	if (iTable.SeekL(rk)) {
		iTable.GetL();
		id=iTable.ColInt32(4);
		return id;
	} else {
		iTable.InsertL();
		iTable.SetColL(1, cell);
		iTable.SetColL(2, area);
		iTable.SetColL(3, nw);
		id=iNextId++;
		iTable.SetColL(4, id);
		PutL();
		return id;
	}

}

void CCellMap::SetId(const TDesC& Cell, TInt id)
{
	CALLSTACKITEM(_L("CCellMap::SetId"));

	TDbSeekMultiKey<3> rk;
	TUint area=0, cell=0;
	TBuf<20> nw;
	Parse(Cell, cell, area, nw);
	rk.Add(cell); rk.Add(area); rk.Add(nw);

	if (iTable.SeekL(rk)) {
		iTable.UpdateL();
		iTable.SetColL(4, id);
	} else {
		iTable.InsertL();
		iTable.SetColL(1, cell);
		iTable.SetColL(2, area);
		iTable.SetColL(3, nw);
		iTable.SetColL(4, id);
		if (id>=iNextId) iNextId=id+1;
	}
	PutL();
}

CCellMap::~CCellMap()
{
	CALLSTACKITEM(_L("CCellMap::~CCellMap"));

	TRAPD(err,
	TDbSeekMultiKey<3> rk;
	rk.Add(0); rk.Add(0); rk.Add(_L(""));
	if (iTable.SeekL(rk)) {
		iTable.UpdateL();
		iTable.SetColL(4, iNextId);
		PutL();
		Settings().WriteSettingL(SETTING_CELLMAP_CLOSED, ETrue);
	}
	);
}

CCellMap::CCellMap(MApp_context& Context, RDbDatabase& Db) : MDBStore(Db), MContextBase(Context)
{
	CALLSTACKITEM(_L("CCellMap::CCellMap"));

}

void CCellMap::ConstructL()
{
	CALLSTACKITEM(_L("CCellMap::ConstructL"));

	TInt cols[]={ EDbColUint32, EDbColUint32, EDbColText, EDbColInt32, -1 };
	TInt idx[]={ 1, 2, 3, -1 };
	SetTextLen(20);
	MDBStore::ConstructL(cols, idx, true, _L("CELLMAP"));

	TDbSeekMultiKey<3> rk;
	rk.Add(0); rk.Add(0); rk.Add(_L(""));
	if (iTable.SeekL(rk)) {
		TBool is_closed;
		if (!Settings().GetSettingL(SETTING_CELLMAP_CLOSED, is_closed)) {
			is_closed=EFalse;
		}
		if (is_closed) {
			iTable.GetL();
			iNextId=iTable.ColInt32(4);
		} else {
			SwitchIndexL(-1);
			TInt idx;
			TBool rows=iTable.FirstL();
			while (rows) {
				iTable.GetL();
				idx=iTable.ColInt32(4);
				if (idx>=iNextId) iNextId=idx+1;
				rows=iTable.NextL();
			}
			SwitchIndexL(0);
		}
	} else {
		iNextId=1;
		iTable.InsertL();
		iTable.SetColL(1, (TUint32)0);
		iTable.SetColL(2, (TUint32)0);
		iTable.SetColL(3, _L(""));
		iTable.SetColL(4, iNextId);
		PutL();
	}
	Settings().WriteSettingL(SETTING_CELLMAP_CLOSED, EFalse);
}

void CCellMap::PrintMapping(Mfile_output_base& To)
{
	CALLSTACKITEM(_L("CCellMap::PrintMapping"));

	SwitchIndexL(-1);
	TInt idx, cell, area;
	TBool rows=iTable.FirstL();
	TBuf<20> nw;
	TBuf<60> line;
	while (rows) {
		iTable.GetL();
		cell=iTable.ColUint32(1);
		area=iTable.ColUint32(2);
		nw=iTable.ColDes(3);
		idx=iTable.ColInt32(4);
		line.Format(_L("%d\t%d, %d, %S\n"), idx, area, cell, &nw);
		To.write_to_output(line);
		rows=iTable.NextL();
	}
	SwitchIndexL(0);
}
