/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "discover.h"
#include "cl_settings.h"

#include <e32base.h>
#include <f32file.h>
#include <e32std.h>

#include "log_base_impl.h"
#include <timeout.h>

#include <es_sock.h>
#include <bt_sock.h>

#include <bteng.h>

class CDiscoverImpl: public CDiscover, public CCheckedActive, public MTimeOut
{
public:
	~CDiscoverImpl();
	
	const TDesC& get_value();
	
	static CDiscoverImpl* NewL(MApp_context& Context);
private:
	void ConstructL();
	CDiscoverImpl(MApp_context& Context);
	virtual void DoCancel();
	virtual void CheckedRunL();
private:
	enum comm_state { IDLE, FIRST_SEARCH, NEXT_SEARCH };
	comm_state current_state;
	TBuf<1024> names;
	TBuf<1024> prev_env;

	RHostResolver hr; bool hr_is_open;
	RSocket socket; bool socket_is_open;
	RSocketServ socket_server; bool socket_server_is_open;
	TInquirySockAddr addr;
	TNameEntry name;
	TProtocolDesc pInfo;
	TInt	iInterval;

	RTimer timer; bool timer_is_open;
	CTimeOut*	iTimeOut;

	RBTDeviceHandler iBtHandler;

	void release();
	void finished();
	void Restart();
	void expired(CBase* source);
	
	friend class CDiscover;
};

CDiscover* CDiscover::NewL(MApp_context& Context)
{
	CALLSTACKITEM(_L("CDiscoverImpl::NewL"));

	CDiscoverImpl* ret=new (ELeave) CDiscoverImpl(Context);
	CleanupStack::PushL(ret);
	ret->ConstructL();
	CleanupStack::Pop();
	return ret;
}

CDiscover::~CDiscover()
{
}

CDiscover::CDiscover(MApp_context& Context) : Mlog_base_impl(Context)
{
}

CDiscoverImpl::CDiscoverImpl(MApp_context& Context) : CDiscover(Context),
CCheckedActive(CCheckedActive::EPriorityIdle, _L("CDiscoverImpl")), 
current_state(IDLE)
{
	CALLSTACKITEM(_L("CDiscoverImpl::CDiscoverImpl"));

}

void CDiscoverImpl::ConstructL() 
{
	CALLSTACKITEM(_L("CDiscoverImpl::ConstructL"));

	Mlog_base_impl::ConstructL(KLog_bt, _L(""));
	iTimeOut=CTimeOut::NewL(*this);

	CActiveScheduler::Add(this); // add to scheduler

	Restart();
}

void CDiscoverImpl::Restart()
{
	Cancel();
	release();
	iTimeOut->Reset();

	User::LeaveIfError(timer.CreateLocal()); timer_is_open=true;
#ifndef __WINS__
	User::LeaveIfError(socket_server.Connect()); socket_server_is_open=true;
	TProtocolName aName=_L("BTLinkManager");
	User::LeaveIfError(socket_server.FindProtocol(aName, pInfo));
	User::LeaveIfError(hr.Open(socket_server,pInfo.iAddrFamily,pInfo.iProtocol)); hr_is_open=true;

	CBTMCMSettings* BtSettings=CBTMCMSettings::NewLC(0);
	TBool power;
	BtSettings->GetPowerStateL(power);
	if (!power) {
		post_new_value(_L("Powering up Bt"), Mlogger::INFO);
		BtSettings->SetPowerStateL(ETrue);
	}
	CleanupStack::PopAndDestroy();

#endif

	iInterval=5*60;
	Settings().GetSettingL(SETTING_BT_SCAN_INTERVAL, iInterval);
	TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
	timer.After(iStatus, w);
	SetActive();
	current_state=IDLE;
}

CDiscoverImpl* CDiscoverImpl::NewL(MApp_context& Context)
{
	CALLSTACKITEM(_L("CDiscoverImpl::NewL"));

	CDiscoverImpl* ret=new (ELeave) CDiscoverImpl(Context);
	CleanupStack::PushL(ret);
	ret->ConstructL();
	CleanupStack::Pop();
	return ret;
}

void CDiscoverImpl::release()
{	
	CALLSTACKITEM(_L("CDiscoverImpl::release"));

	if (timer_is_open) timer.Close();
	timer_is_open=false;
	if (hr_is_open) hr.Close();
	hr_is_open=false;
	if (socket_server_is_open) socket_server.Close();
	socket_server_is_open=false;
}

CDiscoverImpl::~CDiscoverImpl()
{
	CALLSTACKITEM(_L("CDiscoverImpl::~CDiscoverImpl"));

	delete iTimeOut;
	Cancel();
	release();
}	

void CDiscoverImpl::DoCancel()
{
	CALLSTACKITEM(_L("CDiscoverImpl::DoCancel"));

	switch(current_state) {
	case IDLE:
		timer.Cancel();
		break;
	default:
		hr.Cancel();
		break;
	}
}

const TDesC& CDiscoverImpl::get_value()
{
	CALLSTACKITEM(_L("CDiscoverImpl::get_value"));

	return names;
}


void CDiscoverImpl::CheckedRunL()
{
	CALLSTACKITEM(_L("CDiscoverImpl::CheckedRunL"));

	iTimeOut->Reset();
	TBuf<30> msg;
	if (iStatus.Int()!=KErrNone && iStatus.Int()!=KErrHostResNoMoreResults ) {
		_LIT(stat_err, "error: %d");
		msg.Format(stat_err, iStatus.Int());
		post_new_value(msg, Mlogger::ERR);
		Restart();
		return;
	}
#ifndef __WINS__
	switch(current_state) {
	case FIRST_SEARCH:
		if (iStatus.Int()!=KErrHostResNoMoreResults) {
			names.Zero();
			TBuf<30> addr;
			TBTSockAddr btaddr(name().iAddr);
			btaddr.BTAddr().GetReadable(addr);
			names.Append(addr);
			names.Append(_L(" ["));
			names.Append(name().iName);
			names.Append(_L("] "));
			hr.Next(name, iStatus);
			iTimeOut->Wait(3*60);
			SetActive();
			current_state=NEXT_SEARCH;
		} else {
			if (prev_env.Length()>0) {
				post_new_value(_L(""));
				prev_env.Zero();
			}
			current_state=IDLE;
			TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
			timer.After(iStatus, w);
			current_state=IDLE;
			SetActive();
		}

		break;
	case NEXT_SEARCH:
		if (iStatus.Int()!=KErrHostResNoMoreResults) {
			if (! (names.Length()+32+name().iName.Length() > names.MaxLength()) ) {
				TBuf<30> addr;
				TBTSockAddr btaddr(name().iAddr);
				btaddr.BTAddr().GetReadable(addr);
				names.Append(addr);
				names.Append(_L(" ["));
				names.Append(name().iName);
				names.Append(_L("] "));
			}
			hr.Next(name, iStatus);
			iTimeOut->Wait(3*60);
			SetActive();
		} else {
			if (prev_env.Compare(names)) {
				post_new_value(names);
				prev_env=names;
			}
			TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
			timer.After(iStatus, w);
			current_state=IDLE;
			SetActive();
		}
		break;
	case IDLE:
		{
		TBool connections;
		iBtHandler.AreExistingConnectionsL(connections);
		if (connections) {
			TTimeIntervalMicroSeconds32 w(iInterval*1000*1000);
			timer.After(iStatus, w);
			current_state=IDLE;
			names=_L("BT connections in use");
			post_new_value(names);
			SetActive();
		} else {
			addr.SetIAC(KGIAC);
			addr.SetAction(KHostResInquiry|KHostResName|KHostResIgnoreCache);
			hr.GetByAddress(addr, name, iStatus);

			SetActive();
			
			current_state=FIRST_SEARCH;
		}
		break;
		}
	}
#else
	post_new_value(_L("0020e04c71b8 [GLOMGOLD-25] 0002ee51c437 [Reno7650]"));

#endif

}

void CDiscoverImpl::expired(CBase* /*source*/)
{
	post_new_value(_L("timeout in Bt discover"), Mlogger::ERR);
	Restart();
}

