/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "file_logger.h"
#include "cl_settings.h"

#include "e32std.h"

void Cfile_logger::register_source(const TDesC& name, const TDesC& initial_value, const TTime& time)
{
	CALLSTACKITEM(_L("Cfile_logger::register_source"));

	write_time(time);
	write_to_output(name);
	_LIT(start, " STARTING:");
	write_to_output(start);
	write_to_output(initial_value);
	write_nl();
}

void Cfile_logger::new_value(log_priority priority, const TDesC& name, const TDesC& value, const TTime& time)
{
	CALLSTACKITEM(_L("Cfile_logger::new_value"));

	if (priority<priority_limit) return;

	write_time(time);
	write_to_output(name);
	_LIT(sep, ": ");
	write_to_output(sep);
	write_to_output(value);
	write_nl();
}

void Cfile_logger::unregister_source(const TDesC& name, const TTime& time)
{
	CALLSTACKITEM(_L("Cfile_logger::unregister_source"));

	write_time(time);
	write_to_output(name);
	_LIT(stop, " STOPPING");
	write_to_output(stop);
	write_nl();
}

void Cfile_logger::ConstructL(const TDesC& prefix)
{
	CALLSTACKITEM(_L("Cfile_logger::ConstructL"));

	TBool logging=ETrue;
	Settings().GetSettingL(SETTING_LOGGING_ENABLE, logging);
	Settings().NotifyOnChange(SETTING_LOGGING_ENABLE, this);
	enabled=logging;

	Mfile_output_base::ConstructL(prefix, logging);
	if (!logging) paused=true;
}

void Cfile_logger::write_to_output(const TDesC& str)
{
	CALLSTACKITEM(_L("Cfile_logger::write_to_output"));

	if (paused) return;

	Mfile_output_base::write_to_output(str);
}

void Cfile_logger::SettingChanged(TInt /*Setting*/)
{
	TBool logging;
	if (Settings().GetSettingL(SETTING_LOGGING_ENABLE, logging) && ! logging) {
		enabled=false;
		pause();
	} else {
		enabled=true;
		unpause();
	}
}

void Cfile_logger::pause()
{
	CALLSTACKITEM(_L("Cfile_logger::pause"));
	if (paused) return;
	paused=true;

	write_time();
	_LIT(pausedtxt, "PAUSED");
	write_to_output(pausedtxt);
	write_nl();

}

void Cfile_logger::unpause()
{
	CALLSTACKITEM(_L("Cfile_logger::unpause"));

	if (!paused || !enabled) return;
	switch_file();

	write_time();
	_LIT(unpaused, "CONTINUING");
	write_to_output(unpaused);
	write_nl();
	paused=false;
}

bool Cfile_logger::is_paused()
{
	CALLSTACKITEM(_L("Cfile_logger::is_paused"));

	return paused;
}

Cfile_logger::Cfile_logger(MApp_context& Context, log_priority limit): Mfile_output_base(Context),
	priority_limit(limit) { }
Cfile_logger::~Cfile_logger() 
{
	CALLSTACKITEM(_L("Cfile_logger::~Cfile_logger"));

	Settings().CancelNotifyOnChange(SETTING_LOGGING_ENABLE, this);

	TInt err;
	_LIT(stopped, "STOPPED");
	TRAP(err, 
		write_time();
		write_to_output(stopped);
		write_nl();
	)
}

Cfile_logger* Cfile_logger::NewL(MApp_context& Context,
				 const TDesC& prefix, log_priority limit)
{
	CALLSTACKITEM(_L("Cfile_logger::NewL"));

	Cfile_logger* ret=new (ELeave) Cfile_logger(Context, limit);
	CleanupStack::PushL(ret);
	ret->ConstructL(prefix);
	CleanupStack::Pop();
	return ret;
}

_LIT(CLASS_NAME, "Cfile_logger");

const TDesC& Cfile_logger::name() const
{
	CALLSTACKITEM(_L("Cfile_logger::name"));

	return CLASS_NAME;
}
