/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "picture_publisher.h"
#include "transfer.h"
#include "cl_settings.h"

_LIT(KClassName, "CPicturePublisherImpl");

class CPicturePublisherImpl : public CPicturePublisher, public MContextBase,
	public CCheckedActive, public i_status_notif, public MSettingListener  {
public:
	~CPicturePublisherImpl();
private:
	CPicturePublisherImpl(MApp_context& Context, i_status_notif& notif, MUploadPrompt& Prompt);
	void ConstructL(const TDesC& Path, const TDesC& Files, TInt Setting,
		const TDesC& AdditionalFiles=KNullDesC);

	void CheckedRunL();
	TInt CheckedRunError(TInt aError);
	void DoCancel();

	virtual void register_source(const TDesC&, const TDesC& initial_value, const TTime&);
	virtual void new_value(log_priority priority, const TDesC&, const TDesC& value, const TTime&);
	virtual void unregister_source(const TDesC&, const TTime&);
	virtual const TDesC& name() const;

	virtual void finished();
	virtual void error(const TDesC& descr);
	virtual void status_change(const TDesC& status);

	virtual void SettingChanged(TInt Setting);

	void Transfer();
	bool IsEnabled();
	void StartL();
	void Stop();

	i_status_notif& iNotif;
	CFtpTransfer*	iTransferer;
	TBuf<200>	iLoc;
	bool		iQueued;
	bool		iTransfering;
	bool		iNoPics;
	TFileName	iPath;
	TBuf<30>	iFiles, iAddFiles;
	TInt		iSetting;

#ifdef __WINS__test
	RTimer		iTimer;
#endif
	MUploadPrompt&	iPrompt;

	friend class CPicturePublisher;
};

CPicturePublisher* CPicturePublisher::NewL(MApp_context& Context, i_status_notif& notif,
					const TDesC& Path, const TDesC& Files, TInt Setting, 
					MUploadPrompt& Prompt, const TDesC& AdditionalFiles)
{
	CALLSTACKITEM(_L("CPicturePublisher::NewL"));

	auto_ptr<CPicturePublisherImpl> ret(new (ELeave) 
		CPicturePublisherImpl(Context, notif, Prompt));
	ret->ConstructL(Path, Files, Setting, AdditionalFiles);
	return ret.release();
}

CPicturePublisherImpl::~CPicturePublisherImpl()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::~CPicturePublisherImpl"));

	Settings().CancelNotifyOnChange(iSetting, this);
	Stop();
#ifdef __WINS__test
	iTimer.Close();
#endif
}

CPicturePublisherImpl::CPicturePublisherImpl(MApp_context& Context, i_status_notif& notif, MUploadPrompt& Prompt) :
	MContextBase(Context), CCheckedActive(EPriorityNormal, KClassName), iNotif(notif), iPrompt(Prompt)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::CPicturePublisherImpl"));

}

void CPicturePublisherImpl::ConstructL(const TDesC& Path, const TDesC& Files, TInt Setting,
				const TDesC& AdditionalFiles)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::ConstructL"));

	iPath=Path; iFiles=Files; iSetting=Setting; iAddFiles=AdditionalFiles;
	Settings().NotifyOnChange(iSetting, this);
#ifdef __WINS__test
	iTimer.CreateLocal();
#endif
	CActiveScheduler::Add(this);

	if (IsEnabled()) {
		StartL();
	}
}

void CPicturePublisherImpl::Stop()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::Stop"));

	Cancel();
	delete iTransferer; iTransferer=0;
}

void CPicturePublisherImpl::StartL()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::StartL"));

	Stop();

	if (iPath.Left(1).Compare(_L("?"))) {
		iTransferer=CFtpTransfer::NewL(AppContext(), this,
			iPath,
			_L(""), CFtpTransfer::HTTP, iPrompt);
	} else {
		TFileName iPhysPath1;
		iPhysPath1=iPath;
		iPhysPath1.Replace(0, 1, _L("c"));
		iTransferer=CFtpTransfer::NewL(AppContext(), this,
			iPhysPath1, _L(""),
			CFtpTransfer::HTTP, iPrompt);
	}
	iTransferer->add_filesL(iFiles, false);
	if (iAddFiles.Length()>0) iTransferer->add_filesL(iAddFiles, false);
#ifndef __WINS__test
	Fs().NotifyChange(ENotifyEntry, iStatus, iPath);
#else
	iTimer.After( iStatus, 30*1000*1000 );
#endif
	SetActive();
}

bool CPicturePublisherImpl::IsEnabled()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::IsEnabled"));

	TBool enabled;
	if ( Settings().GetSettingL(iSetting, enabled) && enabled) return true;
	return false;
}

void CPicturePublisherImpl::SettingChanged(TInt)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::SettingChanged"));

	if (IsEnabled()) {
		TRAPD(err, StartL());
		if (err!=KErrNone) {
			TBuf<100> msg;
			msg.Format(_L("Error starting PicturePublisher: %d"), err);
			iNotif.error(msg);
		}
	} else {
		Stop();
	}
}
			

void CPicturePublisherImpl::CheckedRunL()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::CheckedRunL"));

	Transfer();
#ifndef __WINS__test
	Fs().NotifyChange(ENotifyEntry, iStatus, iPath);
#else
	iTimer.After( iStatus, 10*1000*1000 );
#endif
	SetActive();
}

void CPicturePublisherImpl::Transfer()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::Transfer"));

	iNotif.status_change(_L("new pictures noticed"));
	iTransfering=true;
	iNoPics=true;
	if (iLoc.Length()==0) iLoc=_L("Unknown");
	TTime now; now.HomeTime(); 

#ifndef __WINS__
	now-=TTimeIntervalMinutes(2);
#else
	now=0;
#endif
	iTransferer->TransferFiles(iLoc, now);
	iNoPics=false;
}

TInt CPicturePublisherImpl::CheckedRunError(TInt aError)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::CheckedRunError"));

	return aError;
}

void CPicturePublisherImpl::DoCancel()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::DoCancel"));

#ifndef __WINS__test
	Fs().NotifyChangeCancel();
#else
	iTimer.Cancel();
#endif
}

void CPicturePublisherImpl::register_source(const TDesC&, const TDesC& initial_value, const TTime&)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::register_source"));

	if (initial_value.Length()>0) iLoc=initial_value;
}

void CPicturePublisherImpl::new_value(log_priority priority, const TDesC&, const TDesC& value, const TTime&)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::new_value"));

	if (priority!=Mlogger::VALUE) iLoc=_L("");
	else iLoc=value;
}

void CPicturePublisherImpl::unregister_source(const TDesC&, const TTime&)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::unregister_source"));

}

const TDesC& CPicturePublisherImpl::name() const
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::name"));

	return KClassName;
}

void CPicturePublisherImpl::finished()
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::finished"));

	if (! iNoPics ) {
		iNotif.status_change(_L("uploaded pics"));
	}
	iTransfering=false;
	if (iQueued) Transfer();
}

void CPicturePublisherImpl::error(const TDesC& descr)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::error"));

	iNotif.error(descr);
	iTransfering=false;
	if (iQueued) Transfer();
}

void CPicturePublisherImpl::status_change(const TDesC& status)
{
	CALLSTACKITEM(_L("CPicturePublisherImpl::status_change"));

	iNotif.status_change(status);
}
