#include "uploadview.h"

#include "context_log.rsg"
#include <aknviewappui.h>
#include <bautils.h>
#include <eikedwin.h>
#include <eiklabel.h>
#include "symbian_auto_ptr.h"
#include "context_log.hrh"
#include "cl_settings.h"
#include "MdaImageConverter.h"

enum KEYCODES {
	JOY_LEFT = 0xF807,
	JOY_RIGHT = 0xF808,
	JOY_UP = 0xF809,
	JOY_DOWN = 0xF80A,
	JOY_CLICK = 0xF845,
	KEY_CALL = 0xF862,
	KEY_CANCEL = 0xF863
};

#define IMG_HEIGHT	40

class CUploadContainer : public CCoeControl, public MEikEdwinObserver, public MContextBase,
	public MMdaImageUtilObserver {
public:
	CUploadContainer(MApp_context& Context, const TDesC& FileName);
	~CUploadContainer();
	void ConstructL(const TRect& aRect);

	void GetFieldsLC(HBufC*& Tag, HBufC*& Description);
	void CloseFile();
private:
        void SizeChanged();
        TInt CountComponentControls() const;
        CCoeControl* ComponentControl(TInt aIndex) const;
        void Draw(const TRect& aRect) const;
	TKeyResponse OfferKeyEventL(const TKeyEvent &aKeyEvent, TEventCode aType);
	void HandleEdwinEventL(CEikEdwin* aEdwin,TEdwinEvent aEventType);

	virtual void MiuoConvertComplete(TInt aError);
	virtual void MiuoCreateComplete(TInt aError);
	virtual void MiuoOpenComplete(TInt aError);

	void SetEditSelected(CEikEdwin *Edit, bool IsFocused);
	void SetEditActive(CEikEdwin *Edit, bool IsActive);
	void ShowImageError(const TDesC& Descr, TInt Code);

	CArrayPtrFlat< CCoeControl > *iControls;
	CEikEdwin *iTagEdit, *iDescriptionEdit;
	CEikLabel *iTagLabel, *iDescriptionLabel;

	CEikLabel *iImgPlaceHolder;

	CEikEdwin *iSelected, *iActive;
	const TDesC&	iFileName;
	CMdaServer*	iMdaServer;
	CMdaImageFileToBitmapUtility*	iFileUtil;
	CMdaBitmapScaler*		iScaler;

	CFbsBitmap	*iOrigBitmap, *iScaledBitmap; bool scaled;
};

class CUploadViewImpl : public CUploadView, public MContextBase {
private:

	struct TCallBackItem {
		TFileName		iFileName;
		MUploadCallBack*	iCallBack;
		TCallBackItem() : iCallBack(0) { }
		TCallBackItem(const TDesC& aFileName, MUploadCallBack* aCallBack) :
			iFileName(aFileName), iCallBack(aCallBack) { }
	};

	CUploadViewImpl(MApp_context& Context, MPresenceData* Presence, 
		TVwsViewId* NextViewId);
	void ConstructL();

        TUid Id() const;
	
        void HandleCommandL(TInt aCommand);
        void DoActivateL(const TVwsViewId& aPrevViewId,
		TUid aCustomMessageId,
		const TDesC8& aCustomMessage);
        void DoDeactivate();

	friend class CUploadView;
	CUploadContainer* iContainer;
	TVwsViewId	iPrevView;
	virtual void Prompt(const TDesC& FileName, MUploadCallBack* CallBack);
	CXmlBuf* MakePacketLC();

	TFileName	iFileName;
	MUploadCallBack* iCallBack;

	CList<TCallBackItem> *iCallBacks;

	MPresenceData	*iPresence;
	TVwsViewId*	iNextViewId;
	bool		iNext;
public:
	virtual ~CUploadViewImpl();
};

EXPORT_C CUploadView* CUploadView::NewL(MApp_context& Context, MPresenceData* Presence, 
		TVwsViewId* NextViewId)
{
	CALLSTACKITEM(_L("CUploadView::NewL"));

	auto_ptr<CUploadViewImpl> ret(new (ELeave) CUploadViewImpl(Context, Presence, 
		NextViewId));
	ret->ConstructL();
	return ret.release();
}

CUploadContainer::CUploadContainer(MApp_context& Context, const TDesC& FileName) : 
	MContextBase(Context), iFileName(FileName)
{
	CALLSTACKITEM(_L("CUploadContainer::CUploadContainer"));

}

void CUploadContainer::GetFieldsLC(HBufC*& Tag, HBufC*& Description)
{
	CALLSTACKITEM(_L("CUploadContainer::GetFieldsLC"));

	const TDesC& tag=iTagEdit->Text()->Read(0);
	delete Tag; Tag=0;
	if (tag.Length()>0) Tag=tag.Left(tag.Length()-1).AllocLC();
	else Tag=tag.AllocLC();

	delete Description; Description=0;
	if (iDescriptionEdit) {
		const TDesC& desc=iDescriptionEdit->Text()->Read(0);
	
		if (desc.Length()>0) Description=desc.Left(desc.Length()-1).AllocLC();
		else Description=desc.AllocLC();
	} else {
		Description=HBufC::NewLC(0);
	}
}

CUploadContainer::~CUploadContainer()
{
	CALLSTACKITEM(_L("CUploadContainer::~CUploadContainer"));

	if (iTagEdit) {
		const TDesC& tag=iTagEdit->Text()->Read(0);
		if (tag.Length()>0) {
			Settings().WriteSettingL(SETTING_UPLOAD_TAG, tag.Left(tag.Length()-1));
		} else {
			Settings().WriteSettingL(SETTING_UPLOAD_TAG, _L(""));
		}
	}
	if (iFileUtil)
		iFileUtil->Close();
	delete iFileUtil;
	if (iControls) iControls->ResetAndDestroy();
	delete iControls;

	delete iScaler;
	delete iOrigBitmap; 
	delete iMdaServer;
}

void CUploadContainer::HandleEdwinEventL(CEikEdwin* /*aEdwin*/, TEdwinEvent /*aEventType*/)
{
	CALLSTACKITEM(_L("CUploadContainer::HandleEdwinEventL"));

}

void CUploadContainer::ShowImageError(const TDesC& Descr, TInt Code)
{
	CALLSTACKITEM(_L("CUploadContainer::ShowImageError"));

	if (!iImgPlaceHolder) return;

	iImgPlaceHolder->MakeVisible(ETrue);

	TBuf<30> msg;
	msg.Append(Descr); msg.Append(_L(": "));
	msg.AppendNum(Code);
	iImgPlaceHolder->SetTextL(msg);
}

void CUploadContainer::MiuoConvertComplete(TInt aError)
{
	CALLSTACKITEM(_L("CUploadContainer::MiuoConvertComplete"));

	if (!iScaledBitmap) {
		iScaler=CMdaBitmapScaler::NewL();
		iScaledBitmap=iOrigBitmap;
		iScaler->ScaleL(*this, *iScaledBitmap,
			TSize(100, IMG_HEIGHT));
	} else {
		scaled=true;
		iImgPlaceHolder->MakeVisible(EFalse);
		DrawNow();
	}
}

void CUploadContainer::MiuoCreateComplete(TInt aError)
{
	CALLSTACKITEM(_L("CUploadContainer::MiuoCreateComplete"));

}

void CUploadContainer::MiuoOpenComplete(TInt aError)
{
	CALLSTACKITEM(_L("CUploadContainer::MiuoOpenComplete"));

	if (aError!=KErrNone) {
		TBuf<40> msg;
		msg.Format(_L("error opening pic %d"), aError);
		iImgPlaceHolder->SetTextL(msg);
	} else {
		TFrameInfo frameInfo;
		iFileUtil->FrameInfo(0, frameInfo);
		iOrigBitmap=new (ELeave) CFbsBitmap;
		iOrigBitmap->Create(frameInfo.iOverallSizeInPixels, EColor4K);
		iFileUtil->ConvertL(*iOrigBitmap);
	}
}

TKeyResponse CUploadContainer::OfferKeyEventL(const TKeyEvent &aKeyEvent, TEventCode aType)
{
	CALLSTACKITEM(_L("CUploadContainer::OfferKeyEventL"));

	if (! iDescriptionEdit) {
		if (iActive) {
			return iActive->OfferKeyEventL(aKeyEvent, aType);
		} else {
			return EKeyWasNotConsumed;
		}
	}

	if (aKeyEvent.iCode==JOY_CLICK) {
		SetEditActive(iSelected, iActive==0);
		return EKeyWasConsumed;
	}
	else if(iActive) 
	{
		TKeyResponse ret;
		ret=iActive->OfferKeyEventL(aKeyEvent, aType);
		return ret;
	} else if (aKeyEvent.iCode==JOY_UP || aKeyEvent.iCode==JOY_DOWN ||
		aKeyEvent.iCode==JOY_LEFT || aKeyEvent.iCode==JOY_RIGHT  ) {
		CEikEdwin *prev=iSelected;
		SetEditSelected(iSelected, false);
		if (prev==iTagEdit) {
			SetEditSelected(iDescriptionEdit, true);
		} else {
			SetEditSelected(iTagEdit, true);
		}
		return EKeyWasConsumed;
	} else {
		return EKeyWasNotConsumed;
	}
}

void CUploadContainer::ConstructL(const TRect& aRect)
{
	CALLSTACKITEM(_L("CUploadContainer::ConstructL"));

	iControls=new (ELeave) CArrayPtrFlat< CCoeControl >(10);
	CreateWindowL();

	TRect r(TPoint(5, 5), TSize(aRect.Width()-10, 1));

	if (iFileName.Right(3).CompareF(_L("jpg"))==0) {
		iMdaServer=CMdaServer::NewL();
		iFileUtil=CMdaImageFileToBitmapUtility::NewL(*this, iMdaServer);
		iFileUtil->OpenL(iFileName);
		scaled=false;
		iImgPlaceHolder=new (ELeave) CEikLabel;
		iControls->AppendL(iImgPlaceHolder);
		iImgPlaceHolder->SetContainerWindowL( *this );
		iImgPlaceHolder->SetFont(iEikonEnv->DenseFont());
		iImgPlaceHolder->SetTextL( _L("loading image...") ); 
		r.SetHeight(IMG_HEIGHT);
		iImgPlaceHolder->SetRect(r);
		r.Move(0, r.Height()+4);
	} else {
		r.SetHeight(IMG_HEIGHT);
		r.Move(0, r.Height()+4);
	}

	iTagLabel=new (ELeave) CEikLabel;
	iControls->AppendL(iTagLabel);
	iTagLabel->SetContainerWindowL( *this );
	iTagLabel->SetTextL( _L("Tag:") );
	r.SetHeight(12);
	iTagLabel->SetRect(r);
	TInt tagw=iTagLabel->MinimumSize().iWidth+4;
	r.Move( tagw , 0); r.SetWidth(r.Width()-tagw);

	TBuf<100> tag;
	Settings().GetSettingL(SETTING_UPLOAD_TAG, tag);

	iTagEdit=new (ELeave) CEikEdwin;
	iControls->AppendL(iTagEdit);
	iTagEdit->SetContainerWindowL( *this );
	iTagEdit->ConstructL();
	iTagEdit->SetTextL(&tag);
	iTagEdit->AddEdwinObserverL(this);
	r.SetHeight(16);
	iTagEdit->SetRect(r);
	iTagEdit->ActivateL();
	r.Move(-tagw, r.Height()+4); r.SetWidth(r.Width()+tagw);


	TBool no_descr;
	if (Settings().GetSettingL(SETTING_TAG_ONLY, no_descr) && no_descr) 
		no_descr=true;
	else
		no_descr=false;
	
	if (!no_descr) {
		iDescriptionLabel=new (ELeave) CEikLabel;
		iControls->AppendL(iDescriptionLabel);
		iDescriptionLabel->SetContainerWindowL( *this );
		iDescriptionLabel->SetTextL( _L("Description:") );
		r.SetHeight(12);
		iDescriptionLabel->SetRect(r);
		r.Move(0, r.Height()+4);

		iDescriptionEdit=new (ELeave) CEikEdwin;
		iControls->AppendL(iDescriptionEdit);
		iDescriptionEdit->SetContainerWindowL( *this );
		iDescriptionEdit->ConstructL();
		iDescriptionEdit->AddEdwinObserverL(this);
		r.SetHeight(40);
		iDescriptionEdit->SetRect(r);
		r.Move(0, r.Height()+4);

		SetEditSelected(iDescriptionEdit, true); SetEditActive(iDescriptionEdit, true);
	} else {
		SetEditSelected(iTagEdit, true); SetEditActive(iTagEdit, true);
	}

	SetRect(aRect);
	ActivateL();	
}

void CUploadContainer::SetEditSelected(CEikEdwin *Edit, bool IsSelected)
{
	CALLSTACKITEM(_L("CUploadContainer::SetEditSelected"));

	if (IsSelected) 
		iSelected=Edit;
	else
		iSelected=0;
	DrawNow();
}

void CUploadContainer::SetEditActive(CEikEdwin *Edit, bool IsActive)
{
	CALLSTACKITEM(_L("CUploadContainer::SetEditActive"));

	if (IsActive) {
		iActive=Edit;
		iActive->SetFocus(ETrue);
	} else {
		if (iActive) iActive->SetFocus(EFalse);
		iActive=0;
	}
	DrawNow();
}

void CUploadContainer::SizeChanged()
{
	CALLSTACKITEM(_L("CUploadContainer::SizeChanged"));

}

TInt CUploadContainer::CountComponentControls() const
{
	CALLSTACKITEM(_L("CUploadContainer::CountComponentControls"));

	return iControls->Count();
}

CCoeControl* CUploadContainer::ComponentControl(TInt aIndex) const
{
	CALLSTACKITEM(_L("CUploadContainer::ComponentControl"));

	return iControls->At(aIndex);
}

void CUploadContainer::Draw(const TRect& aRect) const
{
	CALLSTACKITEM(_L("CUploadContainer::Draw"));

	CWindowGc& gc = SystemGc();
	// TODO: Add your drawing code here
	// example code...
	gc.SetPenStyle(CGraphicsContext::ENullPen);
	gc.SetBrushColor(KRgbWhite);
	gc.SetBrushStyle(CGraphicsContext::ESolidBrush);
	gc.DrawRect(aRect);

	gc.SetPenStyle(CGraphicsContext::ESolidPen);
	gc.SetBrushStyle(CGraphicsContext::ENullBrush);

	TGulBorder border(TGulBorder::ESingleGray);
	TRect edit_rect=iTagEdit->Rect();
	edit_rect.Resize(4, 4);
	edit_rect.Move(-2, -2);
	border.Draw(gc, edit_rect);

	if (iDescriptionEdit) {
		edit_rect=iDescriptionEdit->Rect();
		edit_rect.Resize(4, 4);
		edit_rect.Move(-2, -2);
		border.Draw(gc, edit_rect);
	}

	if (iSelected) {
		TGulBorder border(TGulBorder::ESingleBlack);
		//gc.SetPenColor(KRgbBlack);
		TRect edit_rect=iSelected->Rect();
		edit_rect.Resize(4, 4);
		edit_rect.Move(-2, -2);
		border.Draw(gc, edit_rect);
	}
	if (iActive) {
		TGulBorder border(TGulBorder::EFocusedSunkenControl);
		//gc.SetPenColor(KRgbBlack);
		TRect edit_rect=iActive->Rect();
		edit_rect.Resize(4, 4);
		edit_rect.Move(-2, -2);
		border.Draw(gc, edit_rect);
	}

	if (scaled) {
		TSize s=iScaledBitmap->SizeInPixels();
		TPoint lt=TPoint( (Rect().Width()-s.iWidth)/2, 4);
		TRect r( lt, s);
		gc.DrawBitmap(r, iScaledBitmap);
	}
}

CUploadViewImpl::CUploadViewImpl(MApp_context& Context, MPresenceData* Presence, 
		TVwsViewId* NextViewId) : MContextBase(Context),
		iPresence(Presence), iNextViewId(NextViewId)
{
	CALLSTACKITEM(_L("CUploadViewImpl::CUploadViewImpl"));

}

void CUploadViewImpl::ConstructL()
{
	CALLSTACKITEM(_L("CUploadViewImpl::ConstructL"));

	iCallBacks=CList<TCallBackItem>::NewL();

	BaseConstructL( R_UPLOAD_VIEW );
}

TUid CUploadViewImpl::Id() const
{
	CALLSTACKITEM(_L("CUploadViewImpl::Id"));

	return KUploadView;
}

CXmlBuf* CUploadViewImpl::MakePacketLC()
{
	CALLSTACKITEM(_L("CUploadViewImpl::MakePacketLC"));

	CXmlBuf* buf=CXmlBuf::NewL(1024);
	CleanupStack::PushL(buf);

	HBufC *tag=0, *desc=0;
	iContainer->GetFieldsLC(tag, desc);

	buf->BeginElement(_L("packet"));
	buf->Leaf(_L("tag"), *tag);
	buf->Leaf(_L("description"), *desc);
	buf->BeginElement(_L("presence"));

	CleanupStack::PopAndDestroy(2);

	const MXmlInfo* events[] = { &(iPresence->CellInfo()), &(iPresence->LocInfo()), 
		&(iPresence->ProfileInfo()), &(iPresence->BluetoothInfo()), 0 };

	const MXmlInfo* event;
	TBuf<20> buffer;
	for (int i=0; event=events[i]; i++) {

		buf->BeginElement(_L("event"));
		_LIT(KFormatTxt,"%04d%02d%02dT%02d%02d%02d");
		TDateTime dt=event->iUpdated.DateTime();

		buffer.Format(KFormatTxt, dt.Year(), (TInt)dt.Month()+1, (TInt)dt.Day()+1,
			dt.Hour(), dt.Minute(), dt.Second());
		buf->Leaf(_L("datetime"), buffer);
		event->Persist(buf);
		buf->EndElement(_L("event"));
	}
	buf->EndElement(_L("presence"));

	buf->EndElement(_L("packet"));

	return buf;
}

void CUploadContainer::CloseFile()
{
	if (iFileUtil) {
		iFileUtil->Close();
	}
	delete iFileUtil; iFileUtil=0;
}

void CUploadViewImpl::HandleCommandL(TInt aCommand)
{
	CALLSTACKITEM(_L("CUploadViewImpl::HandleCommandL"));

	iNext=false; 
	MUploadCallBack* cb=iCallBack; iCallBack=0;
	if (cb) {
		switch(aCommand) {
		case Econtext_logCmdSoftkeyUpload:
			{
			TBool del=ETrue;
			Settings().GetSettingL(SETTING_DELETE_UPLOADED, del);
			CXmlBuf* buf=MakePacketLC();
			iContainer->CloseFile();
			cb->Back(true, del, buf);
			CleanupStack::PopAndDestroy();
			}
			break;
		case Econtext_logCmdSoftkeyCancel:
			iContainer->CloseFile();
			cb->Back(false, false, 0);
			break;
		default:
			return;
			break;
		}
	}
	if (!iNext && iCallBacks->iCount > 0) {
		iCallBack=0;
		TCallBackItem i=iCallBacks->Pop();
		Prompt(i.iFileName, i.iCallBack);
	}
	if (!iNext) {
		iCallBack=0;
		TUid statusv={1};
		AppUi()->ActivateLocalViewL(statusv);
		//CSwitchBack::NewL(iPrevView);
		*iNextViewId=iPrevView;
	} else {
		MUploadCallBack* cb=iCallBack; iCallBack=0;
		DoDeactivate();
		TUid dummy={0};
		iCallBack=cb;
		DoActivateL(iPrevView, dummy, _L8(""));
	}
}

void CUploadViewImpl::Prompt(const TDesC& FileName, MUploadCallBack* CallBack)
{
	CALLSTACKITEM(_L("CUploadViewImpl::Prompt"));
	*iNextViewId=TVwsViewId();

	if (!iCallBack) {
		iFileName=FileName;
		iCallBack=CallBack;
		if (iContainer) iNext=true;
		else AppUi()->ActivateLocalViewL(KUploadView);
	} else {
		iCallBacks->AppendL(TCallBackItem(FileName, CallBack));
	}
}

void CUploadViewImpl::DoActivateL(const TVwsViewId& aPrevViewId,
	TUid /*aCustomMessageId*/,
	const TDesC8& /*aCustomMessage*/)
{
	CALLSTACKITEM(_L("CUploadViewImpl::DoActivateL"));

	iPrevView=aPrevViewId;
	if (!iContainer) {
		iContainer=new (ELeave) CUploadContainer(AppContext(), iFileName);
		iContainer->SetMopParent(this);
		iContainer->ConstructL(ClientRect());
		AppUi()->AddToStackL( *this, iContainer );
        } 
}

void CUploadViewImpl::DoDeactivate()
{
	CALLSTACKITEM(_L("CUploadViewImpl::DoDeactivate"));

	if ( iContainer )
        {
		AppUi()->RemoveFromViewStack( *this, iContainer );
        }
	
	delete iContainer;
	iContainer = 0;
	if (iCallBack) iCallBack->Back(false, false, 0);
	iCallBack=0;
}

CUploadViewImpl::~CUploadViewImpl()
{
	CALLSTACKITEM(_L("CUploadViewImpl::~CUploadViewImpl"));

	delete iCallBacks;
	delete iContainer;
}