/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/
#include "user_notifier.h"
#include "timeout.h"
#include "Contextclientsession.h"
#include "symbian_auto_ptr.h"
#include "contextcommon.h"
#include "eikenv.h"
#include "aknquerydialog.h"
#include "aknmessagequerydialog.h"
#include <context_log.rsg>
#include "browser_interface.h"
#include <e32std.h>
#include <CFLDRingingTonePlayer.h>
#include <viewcli.h> //CVwsSessionWrapper
#include <vwsdef.h>

#include "expat.h"


_LIT(KClassName, "CUserNotifierImpl");


class CUserNotifierImpl : public CUserNotifier, public CCheckedActive, public MContextBase, public MTimeOut {
public:
	~CUserNotifierImpl();
private:
	CUserNotifierImpl(MApp_context& Context, CFLDRingingTonePlayer * aRingp, Clog_profile * aPl);
	void ConstructL();

	void CheckedRunL();
	TInt CheckedRunError(TInt aError);
	void DoCancel();

	void expired(CBase*);

	void Start();
	void Restart();

	void RequestMessageNotification();
	void CancelRequest();

	void NotifyUser(/*TDesC& contact, TDesC& subject, TDesC& message*/);

	RContextClientSession iSession;
	bool iSessionOpen;
	CTimeOut* iWait;

	static void startElement(void *userData, const XML_Char *el, const XML_Char **atts);
	void startElement(const XML_Char *el, const XML_Char **atts);
	
	static void endElement(void *userData, const XML_Char *name);
	void endElement(const XML_Char *name);

	static void charData(void *userData, const XML_Char *s, int len);
	void charData(const XML_Char *s, int len);

	enum TState
	{
      		EIdle,
        	EWaitingForMessage		
	};
	TState current_state;

	enum TParseState
	{
		EStackUndefined,
		EStackIgnoreElement,
		EStackUrl,
		EStackAlert,
		EStackDescription,
		EStackError
	};

	TPtr iC;
	TPtr iS;
	TPtr iM;

	HBufC * iContact;
	HBufC * iSubject;
	HBufC * iMessage;

	CFLDRingingTonePlayer * ringp;
	Clog_profile * pl;

	XML_Parser		iParser;
	CList<TInt>	* iStack;

	HBufC * iAlertDesc;
	HBufC * iAlertUrl;

	const static TInt KDefaultBufferSize;

	friend class CUserNotifier;
};

const TInt CUserNotifierImpl::KDefaultBufferSize=512;

CUserNotifier* CUserNotifier::NewL(MApp_context& Context, CFLDRingingTonePlayer * aRingp, Clog_profile * aPl)
{
	CUserNotifierImpl * self = new (ELeave) CUserNotifierImpl(Context, aRingp, aPl);
    	CleanupStack::PushL(self);
    	self->ConstructL();
	CleanupStack::Pop();

    	return self;
}

CUserNotifierImpl::CUserNotifierImpl(MApp_context& Context, CFLDRingingTonePlayer * aRingp, Clog_profile * aPl) : CCheckedActive(EPriorityNormal, KClassName), MContextBase(Context), iC(0,0), iS(0,0), iM(0,0), ringp(aRingp), pl(aPl)
{
    	CActiveScheduler::Add(this);
}

void CUserNotifierImpl::ConstructL()
{
	iWait=CTimeOut::NewL(*this);
	current_state = EIdle;

	iStack = CList<TInt>::NewL();

	iParser = XML_ParserCreate(NULL);

	XML_ParserReset(iParser, NULL);
	XML_SetUserData(iParser,this);
	XML_SetElementHandler(iParser, CUserNotifierImpl::startElement, CUserNotifierImpl::endElement);
	XML_SetCharacterDataHandler(iParser, CUserNotifierImpl::charData);


	iContact = HBufC::NewL(KDefaultBufferSize/4);
	iSubject = HBufC::NewL(KDefaultBufferSize/4);
	iMessage = HBufC::NewL(KDefaultBufferSize);

	iAlertDesc=HBufC::NewL(256);
	iAlertUrl=HBufC::NewL(128);

	Start();
}

void CUserNotifierImpl::startElement(void *userData, const XML_Char *el, const XML_Char **atts)
{
	CUserNotifierImpl* not=(CUserNotifierImpl*)userData;
	not->startElement(el, atts);
}

void CUserNotifierImpl::startElement(const XML_Char *el, const XML_Char **atts)
{
	TInt currentState = EStackUndefined;
	
	if ( iStack->iCurrent != NULL )
	{
		currentState = iStack->iCurrent->Item;
	}
	
	// if we're already ignoring parent tag
	if ( currentState == EStackIgnoreElement)
	{
		// then let's ignore all children
		iStack->AppendL(EStackIgnoreElement);
		return;
	}
	
	//--------------- XML TAGS AND ATTR _LIT-------------
	
	_LIT(KAlert, "alert");
	_LIT(KDescription, "description");
	_LIT(KUrl, "url");

	TPtrC element = TPtrC( (TUint16*)el );
	
	if (element.Compare(KAlert) == 0) {
		// Case: <alert>
		if (currentState != EStackUndefined)
		{
			iStack->AppendL(EStackError);
		}else{
			iStack->AppendL(EStackAlert);
		}
	} else if ( element.Compare(KDescription) == 0) {
		//case: <description>
		if (currentState != EStackAlert)
		{
			iStack->AppendL(EStackError);
		}else{
			iStack->AppendL(EStackDescription);
		}
	} else if ( element.Compare(KUrl) == 0) {
		//Case: <url>
		if (currentState != EStackAlert)
		{
			iStack->AppendL(EStackError);
		}else{
			iStack->AppendL(EStackUrl);
		}
		
	}
	else {
		// Case: unhandled tag
		iStack->AppendL(EStackIgnoreElement);
	}


}

void CUserNotifierImpl::endElement(void *userData, const XML_Char *name)
{
	CUserNotifierImpl* not=(CUserNotifierImpl*)userData;
	not->endElement(name);
}

void CUserNotifierImpl::endElement(const XML_Char *name)
{
	switch (iStack->iCurrent->Item)
	{
		
	case EStackIgnoreElement:
		// just pop the last item
		iStack->DeleteLast();
		break;
		
	case EStackUrl:
		iStack->DeleteLast();
		break;

	case EStackDescription:
		iStack->DeleteLast();
		break;

	case EStackAlert:
		iStack->DeleteLast();
		NotifyUser();

		break;

	case EStackError:
		XML_ParserReset(iParser, NULL);
		XML_SetUserData(iParser,this);
		XML_SetElementHandler(iParser, CUserNotifierImpl::startElement, CUserNotifierImpl::endElement);
		XML_SetCharacterDataHandler(iParser, CUserNotifierImpl::charData);
		iStack->reset();
		break;

	}

}

void CUserNotifierImpl::charData(void *userData, const XML_Char *s, int len)
{
	CUserNotifierImpl* not=(CUserNotifierImpl*)userData;
	not->charData(s,len);
}

void CUserNotifierImpl::charData(const XML_Char *s, int len)
{
	if (!iStack->iCurrent) return;
	
	if (iStack->iCurrent->Item == EStackDescription) {
		TPtrC t = TPtrC((TUint16*)s,len);
		while ( iAlertDesc->Length()+len > iAlertDesc->Des().MaxLength()) {
			iAlertDesc=iAlertDesc->ReAllocL(iAlertDesc->Des().MaxLength()*2);
		}
		iAlertDesc->Des().Append(t);
	} else if (iStack->iCurrent->Item == EStackUrl)	{
		TPtrC t = TPtrC((TUint16*)s,len);
		while ( iAlertUrl->Length()+len > iAlertUrl->Des().MaxLength()) {
			iAlertUrl=iAlertUrl->ReAllocL(iAlertUrl->Des().MaxLength()*2);
		}
		iAlertUrl->Des().Append(t);
	}
}



//------------------------------------------------------------------------------

void CUserNotifierImpl::Start()
{
	if (iSession.ConnectToContextServer() == KErrNone)
	{
		iSessionOpen=true;
		RequestMessageNotification();
	} else {
		Restart();
	}
}

void CUserNotifierImpl::Restart()
{
	Cancel();
	iWait->Cancel();
	if (iSessionOpen) {
		iSession.Close();
	}
	iSessionOpen=false;
	//RDebug::Print(_L("restarting"));
	iWait->Wait(20);
}


void CUserNotifierImpl::expired(CBase*)
{
	Start();
}

CUserNotifierImpl::~CUserNotifierImpl()
{
	Cancel();
	if (iSessionOpen) iSession.Close();
	delete iWait;

	delete iContact;
	delete iSubject;
	delete iMessage;
	XML_ParserFree(iParser);
	delete iStack;
	delete iAlertUrl;
	delete iAlertDesc;
}

void CUserNotifierImpl::RequestMessageNotification()
{
	if(!IsActive())
	{
		current_state = EWaitingForMessage;

		iC.Set(iContact->Des());
		iS.Set(iSubject->Des());
		iM.Set(iMessage->Des());

		iSession.MsgRequestMessageNotification(iC, iS, iM, iStatus);
		SetActive();
	}		
}

void CUserNotifierImpl::CancelRequest()
{
    	Cancel() ;
}

void CUserNotifierImpl::DoCancel()
{
	iSession.Cancel();
	current_state = EIdle;
}

void CUserNotifierImpl::CheckedRunL()
{
	if (iStatus == ERequestCompleted)
	{
		switch (current_state)
		{
			case EWaitingForMessage:
			{
//				RDebug::Print(iC);
//				RDebug::Print(iS);
//				RDebug::Print(iM);

				//NotifyUser(iC,iS,iM);
				XML_ParserReset(iParser, NULL);
				XML_SetUserData(iParser,this);
				XML_SetElementHandler(iParser, CUserNotifierImpl::startElement, CUserNotifierImpl::endElement);
				XML_SetCharacterDataHandler(iParser, CUserNotifierImpl::charData);
				
				if (!XML_Parse(iParser,(char*)(iM.Ptr()), iM.Size(), 1)) {
					XML_Error code=XML_GetErrorCode(iParser);
					//TPtrC descr((TUint16*)XML_ErrorString(code));
					//Log(_L("XML parse error"));
					//Log(descr);
					//ReportError(MEngineNotifier::EXmlParseError, code);
				}

				RequestMessageNotification();
				break;
			}
			default:
				Restart();
				// ASSERT(0); //Unexpected error
				break;
		}
	}
	else if (iStatus == EBufferTooSmall)
	{
		iContact = iContact->ReAllocL(iC.MaxLength() *2);
		iC.Set(iContact->Des());

		iSubject = iSubject->ReAllocL(iS.MaxLength() *2);
		iS.Set(iSubject->Des());

		iMessage = iMessage->ReAllocL(iM.MaxLength() *2);
		iM.Set(iMessage->Des());

		RequestMessageNotification();
	}
	else 
	{
		Restart();
	}
}

TInt CUserNotifierImpl::CheckedRunError(TInt aError)
{
	return aError;
}

void CUserNotifierImpl::NotifyUser(/*TDesC& contact, TDesC& subject, TDesC& message*/)
{
	//unhide
	RWsSession& wsSession=CEikonEnv::Static()->WsSession();
	TApaTask task(wsSession);
	task.SetWgId(CEikonEnv::Static()->RootWin().Identifier());
	task.BringToForeground();
	((MCoeForegroundObserver*)ringp)->HandleGainingForeground();

	//-----------------------------
	
	int profile = pl->get_current_profile();
	// cannot play tone if we want to play a ringtone
	// CAknQueryDialog::TTone tone=CAknQueryDialog::EWarningTone;
	CAknQueryDialog::TTone tone=CAknQueryDialog::ENoTone; 
	ringp->SetVibra(ETrue);
	ringp->SetRingingType(2); // ring once
	switch (profile) {
	case 2:
		// meeting
		((MFLDFileProcessor*)ringp)->ProcessFileL(_L("c:\\system\\apps\\context_log\\silent.rng"));
		break;
	case 1:
		// silent
		break;
	default:
		// general, outdoor
		((MFLDFileProcessor*)ringp)->ProcessFileL(_L("c:\\system\\apps\\context_log\\tone.rng"));
		break;
	}

	CAknMessageQueryDialog* dlg = CAknMessageQueryDialog::NewL(*iAlertDesc, tone);
		 
	CleanupStack::PushL(dlg);
	dlg->PrepareLC(R_MESSAGE_QUERY);
	dlg->QueryHeading()->SetTextL(_L("Notification"));
	CleanupStack::Pop();
	
	if ( dlg->RunLD() )
	{
		#ifdef __WINS2__
		CEikonEnv::Static()->BusyMsgL(*iAlertUrl, 2000);
		#endif

		#ifndef __WINS2__
		
		#  ifndef __S60V2__
		auto_ptr<CDorisBrowserInterface> ido(CDorisBrowserInterface::NewL());
		ido->AppendL(CDorisBrowserInterface::EOpenURL_STRING, *iAlertUrl);
		ido->ExecuteL();
		#  else
		HBufC8* addr8=HBufC8::NewLC(iAlertUrl->Length());
		TPtr8 addrp=addr8->Des();
		CC()->ConvertFromUnicode(addrp, *iAlertUrl);
		TUid KUidOperaBrowserUid = {0x101F4DED};
		TUid KUidOperaRenderViewUid = {0};
		TVwsViewId viewId(KUidOperaBrowserUid, KUidOperaRenderViewUid);
		CVwsSessionWrapper* vws;
		vws=CVwsSessionWrapper::NewLC();
		vws->ActivateView(viewId, TUid::Uid(0), *addr8);
		CleanupStack::PopAndDestroy(2);
		#  endif

		#endif
	}
	((MCoeForegroundObserver*)ringp)->HandleLosingForeground();
	((MFLDFileProcessor*)ringp)->Cancel();

	iAlertDesc->Des().Zero();
	iAlertUrl->Des().Zero();
	
}