/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/

#include "bt_dev_ap_view.h"

#include "symbian_auto_ptr.h"
#include "context_log.hrh"
#include <aknviewappui.h>
#include <context_log.rsg>
#include <eiktxlbx.h>
#include <eikfrlb.h>
#include "cl_settings.h"
#include <aknlists.h>

#include <btmanclient.h>
#include <btextnotifiers.h>
#include <btsdp.h>
#include <bt_sock.h>

class CBTDevApContainer;

class CBTDiscoverer : public CCheckedActive, public MContextBase{
public:
	static CBTDiscoverer * NewL(MApp_context& Context, MListObserver * aObserver, CBTDeviceList* aList);
	void ConstructL(MListObserver * aObserver);
	~CBTDiscoverer();
	void SelectDevice();
	void MoveUp(TInt idx);
	void DeleteDevice(TInt idx);
	void MoveDown(TInt idx);

	void DoCancel();
	void CheckedRunL();

	MDesC16Array * GetNameArray();

		
private:
	CBTDiscoverer(MApp_context& Context, MListObserver * aObserver, CBTDeviceList* aList);

	void AddToArrays(const TDesC8& bt_addr, const TDesC16& bt_name);

	RNotifier not; bool not_is_open;
	TBTDeviceSelectionParams selectionFilter;
	TUUID targetServiceClass;
	TBTDeviceSelectionParamsPckg pckg;
	TBTDeviceResponseParamsPckg result;

	enum comm_state { IDLE, SELECTING};
	comm_state current_state;

	CBTDeviceList	*iList;
	MListObserver * iObserver;

#ifdef __WINS__
	int	iDevCount;
#endif
};

class CBTDevApContainer : public CCoeControl, MCoeControlObserver, public MContextBase, public MListObserver {
public:
	CBTDevApContainer(MApp_context& Context, TBool aShowMove);
	~CBTDevApContainer();
	void ConstructL(const TRect& aRect, CBTDeviceList* aList);
	TInt GetCurrentIndex();
	void ContentsChanged();
private:
        void SizeChanged();
	TInt CountComponentControls() const;
        CCoeControl* ComponentControl(TInt aIndex) const;
        void Draw(const TRect& aRect) const;
        void HandleControlEventL(CCoeControl* aControl,TCoeEvent aEventType);
	TKeyResponse OfferKeyEventL(const TKeyEvent &aKeyEvent, TEventCode aType);
private:
	CEikTextListBox* iListBox;
public:
	CBTDiscoverer * iDiscoverer;
	TBool iShowMove;
};

class CBTDevApViewImpl : public CBTDevApView, public MContextBase {
private:

	CBTDevApViewImpl(MApp_context& Context, CBTDeviceList* aList, TBool aShowMove);
	void ConstructL();

        TUid Id() const;
	
        void HandleCommandL(TInt aCommand);
        void DoActivateL(const TVwsViewId& aPrevViewId,
		TUid aCustomMessageId,
		const TDesC8& aCustomMessage);
        void DoDeactivate();

	friend class CBTDevApView;
	CBTDevApContainer* iContainer;
	CBTDeviceList* iList;
public:
	virtual ~CBTDevApViewImpl();
	TBool iShowMove;
};

//---------------------------------------------------------------------------
CBTDiscoverer * CBTDiscoverer::NewL(MApp_context& Context, MListObserver * aObserver, CBTDeviceList* aList)
{
	CALLSTACKITEM2(_L("CBTDiscoverer::NewL"), &Context);

	auto_ptr<CBTDiscoverer> ret(new (ELeave) CBTDiscoverer(Context, aObserver, aList));
	ret->ConstructL(aObserver);
	return ret.release();
}

void CBTDiscoverer::ConstructL(MListObserver * aObserver)
{
	CALLSTACKITEM(_L("CBTDiscoverer::ConstructL"));

	iList->AddObserver(aObserver);

	current_state=IDLE;
	CActiveScheduler::Add(this);
}

MDesC16Array * CBTDiscoverer::GetNameArray()
{
	return iList->NameArray();
}

void CBTDiscoverer::DeleteDevice(TInt idx)
{
	iList->RemoveDeviceL(idx);
}

void CBTDiscoverer::MoveUp(TInt idx)
{
	iList->MoveUpL(idx);
}

void CBTDiscoverer::MoveDown(TInt idx)
{
	iList->MoveDownL(idx);
}

CBTDiscoverer::CBTDiscoverer(MApp_context& Context, MListObserver * aObserver, CBTDeviceList* aList) : 
		MContextBase(Context), CCheckedActive(EPriorityNormal, _L("CBTDiscoverer")), iObserver(aObserver),
			iList(aList)
{
	CALLSTACKITEM(_L("CBTDiscoverer::CBTDiscoverer"));
}

CBTDiscoverer::~CBTDiscoverer()
{
	CALLSTACKITEM(_L("CBTDiscoverer::~CBTDiscoverer"));

	Cancel();
	if (iList && iObserver) iList->RemoveObserver(iObserver);
}

void CBTDiscoverer::SelectDevice()
{
	CALLSTACKITEM(_L("CBTDiscoverer::SelectDevice"));

	#ifdef __WINS__

	TBuf8<10> addr=_L8("12345");
	if (iDevCount==0) {
		unsigned char a[]={ 0x00, 0x02, 0xee, 0x51, 0xc4, 0x37};
		TPtrC8 ap(&a[0], 6);
		addr=ap;
	} else {
		addr.AppendNum(iDevCount);
	}
	
	TBuf<20> name=_L("test name ");
	
	name.AppendNum(iDevCount);
	AddToArrays(addr, name);
	++iDevCount;
	return;
	#endif

	TInt ret;
	if ( (ret=not.Connect())!= KErrNone ) return;
	
	not_is_open=true;
	
	selectionFilter.SetUUID(targetServiceClass);
	result.Zero();
	pckg.Zero();
	not.StartNotifierAndGetResponse(iStatus, KDeviceSelectionNotifierUid, pckg, result);
	
	SetActive();	
	current_state=SELECTING;
}

void CBTDiscoverer::CheckedRunL()
{
	CALLSTACKITEM(_L("CLogGpsImpl::CheckedRunL"));

	if (iStatus.Int() != 0)
	{
		if (not_is_open) not.Close(); not_is_open=false;
		current_state=IDLE;
		return;
	}

	switch(current_state) {
	case SELECTING:
		if(!result().IsValidBDAddr()) {
			current_state=IDLE;
		} else {
			AddToArrays(result().BDAddr().Des(), result().DeviceName());
			current_state=IDLE;
		}
		if (not_is_open) not.Close(); not_is_open=false;
		break;
	case IDLE:
		break;
	}
}

void CBTDiscoverer::AddToArrays(const TDesC8& bt_addr, const TDesC16& bt_name)
{
	iList->AddDeviceL(bt_name, bt_addr);
}

void CBTDiscoverer::DoCancel()
{
	CALLSTACKITEM(_L("CBTDiscoverer::DoCancel"));

	switch(current_state) {
	case SELECTING:
		not.CancelNotifier(KDeviceSelectionNotifierUid);
		break;
	default:
		break;
	}
	current_state=IDLE;
}

CBTDevApView* CBTDevApView::NewL(MApp_context& Context, CBTDeviceList* aList, TBool aShowMove)
{
	CALLSTACKITEM(_L("CBTDevApView::NewL"));

	auto_ptr<CBTDevApViewImpl> ret(new (ELeave) CBTDevApViewImpl(Context, aList, aShowMove));
	ret->ConstructL();
	return ret.release();
}

CBTDevApContainer::CBTDevApContainer(MApp_context& Context, TBool aShowMove) : 
	MContextBase(Context), iShowMove(aShowMove)
{
	CALLSTACKITEM(_L("CBTDevApContainer::CBTDevApContainer"));

}

CBTDevApContainer::~CBTDevApContainer()
{
	CALLSTACKITEM(_L("CBTDevApContainer::~CBTDevApContainer"));

	delete iListBox;
	delete iDiscoverer;
}

void CBTDevApContainer::ContentsChanged()
{
	CALLSTACKITEM(_L("BTDevApContainer::ContentsChanged"));

	TRAPD(err,	
		if (iListBox) iListBox->HandleItemRemovalL();
		iListBox->SetCurrentItemIndexAndDraw(0);
		iListBox->DrawNow(););

}


TKeyResponse CBTDevApContainer::OfferKeyEventL(const TKeyEvent &aKeyEvent, TEventCode aType)
{
	CALLSTACKITEM(_L("CBTDevApContainer::OfferKeyEventL"));
	return iListBox->OfferKeyEventL(aKeyEvent, aType);
}

TInt CBTDevApContainer::GetCurrentIndex()
{
	CALLSTACKITEM(_L("CBTDevApContainer::GetCurrentIndex"));

	TInt idx = iListBox->View()->CurrentItemIndex();

	return idx;
}

void CBTDevApContainer::ConstructL(const TRect& aRect, CBTDeviceList* aList)
{
	CALLSTACKITEM(_L("CBTDevApContainer::ConstructL"));

	iDiscoverer = CBTDiscoverer::NewL(AppContext(), this, aList);

	CreateWindowL(); 

	iListBox = new (ELeave) CEikTextListBox; //CAknSingleStyleListBox; //CAknSingleNumberStyleListBox; 
	iListBox->SetMopParent(this);
	iListBox->ConstructL(this, EAknListBoxSelectionList);

	iListBox->SetItemHeightL(20);
	iListBox->View()->SetMatcherCursor(EFalse);
	
	iListBox->Model()->SetItemTextArray(iDiscoverer->GetNameArray());
	iListBox->Model()->SetOwnershipType(ELbmDoesNotOwnItemArray);
	iListBox->CreateScrollBarFrameL(ETrue);
	iListBox->ScrollBarFrame()->SetScrollBarVisibilityL( CEikScrollBarFrame::EOff, CEikScrollBarFrame::EAuto);

	iListBox->MakeVisible(ETrue);

	iListBox->ActivateL();
	iListBox->DrawNow();

	SetRect(aRect);
	ActivateL();
}

void CBTDevApContainer::SizeChanged()
{
	CALLSTACKITEM(_L("CBTDevApContainer::SizeChanged"));
	iListBox->SetRect(Rect());
}

TInt CBTDevApContainer::CountComponentControls() const
{
	CALLSTACKITEM(_L("CBTDevApContainer::CountComponentControls"));

	return 1;
}

void CBTDevApContainer::Draw(const TRect& aRect) const
{
	CALLSTACKITEM(_L("CBTDevApContainer::Draw"));

	CWindowGc& gc = SystemGc();
	gc.SetPenStyle(CGraphicsContext::ENullPen);
	gc.SetBrushColor(KRgbWhite);
	gc.SetBrushStyle(CGraphicsContext::ESolidBrush);
	gc.DrawRect(aRect);
}

CCoeControl* CBTDevApContainer::ComponentControl(TInt aIndex) const
 {
	CALLSTACKITEM(_L("CBTDevApContainer::ComponentControl"));
	if (aIndex==0) return iListBox;
	return 0;
 }
 
void CBTDevApContainer::HandleControlEventL(CCoeControl* aControl,TCoeEvent aEventType)
{
	CALLSTACKITEM(_L("CBTDevApContainer::HandleControlEventL"));
	// TODO: Add your control event handler code here
}


CBTDevApViewImpl::CBTDevApViewImpl(MApp_context& Context, CBTDeviceList* aList, TBool aShowMove) : MContextBase(Context), iList(aList), iShowMove(aShowMove)
{
	CALLSTACKITEM(_L("CBTDevApViewImpl::CBTDevApViewImpl"));

}

void CBTDevApViewImpl::ConstructL()
{
	CALLSTACKITEM(_L("CBTDevApViewImpl::ConstructL"));

	if (iShowMove) 
		BaseConstructL( R_BT_DEV_AP_VIEW );
	else
		BaseConstructL( R_BT_DEV_SET_VIEW );
}

TUid CBTDevApViewImpl::Id() const
{
	CALLSTACKITEM(_L("CBTDevApViewImpl::Id"));

	// FIXME: should use a parameter for id instead

	if (iShowMove)
		return KBtDevApViewId;
	else 
		return KBtDevSetViewId;
}

void CBTDevApViewImpl::HandleCommandL(TInt aCommand)
{
	CALLSTACKITEM(_L("CBTDevApViewImpl::HandleCommandL"));

	switch(aCommand) {
		case Econtext_logAddDevice:
			iContainer->iDiscoverer->SelectDevice();
			break;
		case Econtext_logDeleteDevice:
			iContainer->iDiscoverer->DeleteDevice(iContainer->GetCurrentIndex());
			break;
		case EAknSoftkeyBack:
			AppUi()->HandleCommandL(Econtext_logCmdlogviewClose);
			break;
		case Econtext_logMoveDeviceUp:
			iContainer->iDiscoverer->MoveUp(iContainer->GetCurrentIndex());
			break;
		case Econtext_logMoveDeviceDown:
			iContainer->iDiscoverer->MoveDown(iContainer->GetCurrentIndex());
			break;
		default:
			break;
	}
}


void CBTDevApViewImpl::DoActivateL(const TVwsViewId& aPrevViewId,
	TUid /*aCustomMessageId*/,
	const TDesC8& /*aCustomMessage*/)
{
	CALLSTACKITEM(_L("CBTDevApViewImpl::DoActivateL"));

	if (!iContainer) {
		iContainer=new (ELeave) CBTDevApContainer(AppContext(), iShowMove);
		iContainer->SetMopParent(this);
		iContainer->ConstructL(ClientRect(), iList);
		AppUi()->AddToStackL( *this, iContainer );
        } 
}

void CBTDevApViewImpl::DoDeactivate()
{
	CALLSTACKITEM(_L("CBTDevApViewImpl::DoDeactivate"));

	if ( iContainer )
        {
		AppUi()->RemoveFromViewStack( *this, iContainer );
        }
	
	delete iContainer;
	iContainer = 0;
}

CBTDevApViewImpl::~CBTDevApViewImpl()
{
	CALLSTACKITEM(_L("CBTDevApViewImpl::~CBTDevApViewImpl"));
	delete iContainer;
	
}