#include "camera.h"

#include <cameraserv.h>
#include <list.h>
#include <symbian_auto_ptr.h>
#include <bautils.h>
#include <MdaImageConverter.h>

class CSnapShotImpl : public CSnapShot, public CCheckedActive, public MContextBase, public MMdaImageUtilObserver {
private:
	CSnapShotImpl(MApp_context& Context);
	void ConstructL();
	static CSnapShot* NewL(MApp_context& Context);
	virtual void TakeSnapShot(const TDesC& aDir, MSnapShotNofifier* aNotifier);
	virtual void CancelSnapShot(MSnapShotNofifier* aNotifier);

	virtual void CheckedRunL();
	virtual TInt CheckedRunError(TInt aCode);
	virtual void DoCancel();

	void Next(bool ok);
	void Async(TInt aError=KErrNone);
	void CreateSaveImage();
	void CloseCamera();
	void Convert();
	void Retry();

	virtual void MiuoConvertComplete(TInt aError);
	virtual void MiuoCreateComplete(TInt aError);
	virtual void MiuoOpenComplete(TInt aError);
public:
	~CSnapShotImpl();

private:
	struct TSnapShotRequest {
		TFileName	iDir;
		MSnapShotNofifier* iNotifier;

		TSnapShotRequest(const TDesC& aDir, MSnapShotNofifier* aNotifier) :
		iDir(aDir), iNotifier(aNotifier) { }
		TSnapShotRequest() : iNotifier(0) { }
	};

	CList<TSnapShotRequest>* iRequests;
	
	enum TState { EIdle, EStarting, ETurningOn, EGettingImage, ECreatingFile, EConverting };
	TState iState;
	RCameraServ iCamera; bool iCameraIsOpen; bool iCameraIsOn;
	CFbsBitmap* iBitMap;
	MSnapShotNofifier* iNotifier;
	CMdaImageBitmapToFileUtility* iFileSaver;
	TFileName	iFileName;
	TMdaJfifClipFormat jfifFormat;
	TInt	retries;
	
	friend class CSnapShot;
};

CSnapShot* CSnapShot::NewL(MApp_context& Context)
{
	CALLSTACKITEM2(_L("CSnapShot::NewL"), &Context);

	auto_ptr<CSnapShotImpl> ret(new (ELeave) CSnapShotImpl(Context));
	ret->ConstructL();
	return ret.release();
}

CSnapShotImpl::CSnapShotImpl(MApp_context& Context) : MContextBase(Context), CCheckedActive(EPriorityNormal, _L("CSnapShotImpl"))
{
	CALLSTACKITEM(_L("CSnapShotImpl::CSnapShotImpl"));

}

void CSnapShotImpl::ConstructL()
{
	CALLSTACKITEM(_L("CSnapShotImpl::ConstructL"));

	iRequests=CList<TSnapShotRequest>::NewL();
	CActiveScheduler::Add(this);
}

void CSnapShotImpl::TakeSnapShot(const TDesC& aDir, MSnapShotNofifier* aNotifier)
{
	CALLSTACKITEM(_L("CSnapShotImpl::TakeSnapShot"));

	iRequests->AppendL(TSnapShotRequest(aDir, aNotifier));

	if (iState==EIdle) {
		iState=EStarting;
		Async();
	}
	return;
}

CSnapShotImpl::~CSnapShotImpl()
{
	CALLSTACKITEM(_L("CSnapShotImpl::~CSnapShotImpl"));

	if (iRequests) {
		TSnapShotRequest r;
		for (r=iRequests->Pop(); r.iNotifier; r=iRequests->Pop()) {
			r.iNotifier->Error(KErrCancel, _L("Cancelled"));
		}
	}
	delete iRequests;

	Cancel();

	CloseCamera();

	delete iBitMap;
	delete iFileSaver;
}

void CSnapShotImpl::Async(TInt aError)
{
	CALLSTACKITEM(_L("CSnapShotImpl::Async"));

	TRequestStatus* s=&iStatus;
	User::RequestComplete(s, aError);
	SetActive();
}

void CSnapShotImpl::CloseCamera()
{
	CALLSTACKITEM(_L("CSnapShotImpl::CloseCamera"));

	if (iCameraIsOn) iCamera.TurnCameraOff();
	if (iCameraIsOpen) iCamera.Close();
	iCameraIsOn=iCameraIsOpen=false;
}

void CSnapShotImpl::DoCancel()
{
	CALLSTACKITEM(_L("CSnapShotImpl::DoCancel"));

	CloseCamera();
	TRequestStatus* s=&iStatus;
	User::RequestComplete(s, KErrCancel);
}

void CSnapShotImpl::Retry()
{
	CALLSTACKITEM(_L("CSnapShotImpl::Retry"));

}

void CSnapShotImpl::CheckedRunL()
{
	CALLSTACKITEM(_L("CSnapShotImpl::CheckedRunL"));

	if (iStatus.Int()!=KErrNone) {
		CheckedRunError(iStatus.Int());
		return;
	}

	retries=0;

	switch(iState) {
	case EStarting:
		iNotifier=iRequests->iFirst->Item.iNotifier;
		User::LeaveIfError(iCamera.Connect());
		iCameraIsOpen=true;
		iCamera.TurnCameraOn(iStatus);
		SetActive();
		iState=ETurningOn;
		iNotifier->Info(_L("Turning Camer On"));
		break;
	case ETurningOn:
		iCameraIsOn=true;
		delete iBitMap; iBitMap=0;
		iBitMap=new (ELeave) CFbsBitmap;
		iState=EGettingImage;
		User::LeaveIfError(iCamera.SetImageQuality(RCameraServ::EQualityHigh));
		User::LeaveIfError(iCamera.SetLightingConditions(RCameraServ::ELightingNormal));
		iCamera.GetImage(iStatus, *iBitMap);
		SetActive();
		iNotifier->Info(_L("Getting Image"));
		break;
	case EGettingImage:
		iState=ECreatingFile;
		iNotifier->Info(_L("Creating File"));
		CreateSaveImage();
		break;
	case ECreatingFile:
		iState=EConverting;
		iNotifier->Info(_L("Saving File"));
		Convert();
		break;
	case EConverting:
		iRequests->Pop();
		delete iBitMap; iBitMap=0;
		delete iFileSaver; iFileSaver=0;
		iNotifier->Taken(iFileName);
		Next(true);
		break;
	}
}

void CSnapShotImpl::Next(bool ok)
{
	CALLSTACKITEM(_L("CSnapShotImpl::Next"));

	if (iState==EStarting || iState==ETurningOn) return;

	if (iRequests->iFirst) {
		if (ok) {
			iState=ETurningOn;
		} else {
			CloseCamera();
			iState=EStarting;
		}
		Async();
	} else {
		iState=EIdle;
		CloseCamera();
	}
}

void CSnapShotImpl::CreateSaveImage()
{
	CALLSTACKITEM(_L("CSnapShotImpl::CreateSaveImage"));

	TFileName base=iRequests->iFirst->Item.iDir;
	if (base.Right(1).Compare(_L("\\"))) {
		base.Append(_L("\\"));
	}
	base.Append(_L("SnapShot"));
	iFileName=base;
	TInt i=1; iFileName.AppendNum(i); iFileName.Append(_L(".jpg"));
	while (BaflUtils::FileExists(Fs(), iFileName)) {
		iFileName=base;
		++i; iFileName.AppendNum(i); iFileName.Append(_L(".jpg"));
	}

	delete iFileSaver; iFileSaver=0;
	iFileSaver=CMdaImageBitmapToFileUtility::NewL(*this);

	jfifFormat.iSettings.iSampleScheme  = TMdaJpgSettings::EColor420;
	jfifFormat.iSettings.iQualityFactor = 50;

	iFileSaver->CreateL(iFileName, &jfifFormat, NULL, NULL);

}

TInt CSnapShotImpl::CheckedRunError(TInt aCode)
{
	CALLSTACKITEM(_L("CSnapShotImpl::CheckedRunError"));

	retries++;
	if (0 && retries<5) {
		Retry();
		return KErrNone;
	}
	delete iBitMap; iBitMap=0;
	delete iFileSaver; iFileSaver=0;
	TBuf<50> msg=_L("Error taking picture while ");
	switch(iState) {
		case EStarting:
			msg.Append(_L("initializing camera"));
			break;
		case ETurningOn:
			msg.Append(_L("turning camera on"));
			break;
		case EGettingImage:
			msg.Append(_L("getting image"));
			break;
		case ECreatingFile:
			msg.Append(_L("creating file"));
			break;
		case EConverting:
			msg.Append(_L("saving file"));
			break;
		default:
			msg.Append(_L("[unknown]"));
			break;
	};

	CloseCamera();
	iRequests->Pop();
	iState=EIdle;
	iNotifier->Error(aCode, msg);
	Next(false);
	return KErrNone;
}

void CSnapShotImpl::CancelSnapShot(MSnapShotNofifier* aNotifier)
{
	CALLSTACKITEM(_L("CSnapShotImpl::CancelSnapShot"));

	CList<TSnapShotRequest>::Node *n=0, *temp=0;
	for (n=iRequests->iFirst; n; ) {
		temp=n->Next;
		if (n->Item.iNotifier==aNotifier) {
			iRequests->DeleteNode(n, true);
		}
		n=temp;
	}
}

void CSnapShotImpl::Convert()
{
	CALLSTACKITEM(_L("CSnapShotImpl::Convert"));

	iFileSaver->ConvertL(*iBitMap);
}

void CSnapShotImpl::MiuoConvertComplete(TInt aError)
{
	CALLSTACKITEM(_L("CSnapShotImpl::MiuoConvertComplete"));

	Async(aError);
}

void CSnapShotImpl::MiuoCreateComplete(TInt aError)
{
	CALLSTACKITEM(_L("CSnapShotImpl::MiuoCreateComplete"));

	Async(aError);
}

void CSnapShotImpl::MiuoOpenComplete(TInt aError)
{
	CALLSTACKITEM(_L("CSnapShotImpl::MiuoOpenComplete"));

	Async(aError);
}
