/* 
    Copyright (C) 2004  Mika Raento - Renaud Petit

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    email: mraento@cs.helsinki.fi - petit@cs.helsinki.fi 
*/


#include "ver.h"
#include "http.h"
#include "symbian_auto_ptr.h"

const TInt CHttp::MAX_RETRIES=5;

CHttp* CHttp::NewL(MApp_context& Context, MSocketObserver& obs)
{
	CALLSTACKITEMSTATIC(_L("CHttp::NewL"));

	auto_ptr<CHttp> ret(new (ELeave) CHttp(Context, obs));
	ret->ConstructL();
	return ret.release();
}

CHttp::CHttp(MApp_context& Context, MSocketObserver& obs) : MContextBase(Context),
	observer(obs), MAXSIZE(14*1024) { }


void CHttp::success(CBase* source)
{
	CALLSTACKITEM(_L("CHttp::success"));

	TBuf8<100> msg;
	msg.Format(_L8("CHttp success at state: %d\n"), (int)current_state);
	file.Write(msg);

	if (source==iInitiator) {
		// only from connection initiator

		observer.info(this, _L("Connected to net"));
		if (!iRetryCount) {
			observer.success(this);
		} else {
			if (!HandleFile()) RetryHandleFile();
		}
	} else {
		
		iWait->Reset();

		switch (current_state) {
		case IDLE:
		case CONNECTED:
		case RESULT_OK_WAIT:
			observer.error(this, -1002, _L("CHttp Inconsistent state"));
			break;
		case CONNECTING_SOCKET:
			observer.info(this, _L("sending header"));
			iSocket->RemoteName(iAddr); iAddrKnown=true;
			file.Write(head);
			iSocket->Send(head, 120);
			current_state=HEAD_TX;
			break;
		case HEAD_TX:
			observer.info(this, _L("sending body"));
			file.Write(iContents->Des().Left(100));
			iSocket->SendExpect(*iContents, _L8("^OK"), 
				_L8("^Error"), true, 300);
			current_state=BODY_TX;
			break;
		case BODY_TX:
			iRetryCount=0;

			current_state=RESULT_OK_WAIT;
			observer.info(this, _L("Intra-post wait"));
			iWait->Wait(1);
			break;
		default:
			break;
		}
		return;
	}
}

void CHttp::error(CBase* source, TInt code, const TDesC& reason)
{
	CALLSTACKITEM(_L("CHttp::error"));

	if (source==iInitiator) {
		observer.error(this, code, reason);
	} else {
		if (iRetryCount==MAX_RETRIES) {
			iRetryCount=0;
			iFileSize=0;
			iOffset=0;
			observer.error(this, code, reason);
		} else {
			++iRetryCount;
			observer.info(this, reason);
			observer.info(this, _L("retrying"));
			Connect(iIapId, iHost);
		}
	}
}

void CHttp::info(CBase* /*source*/, const TDesC& msg)
{
	CALLSTACKITEM(_L("CHttp::info"));

	observer.info(this, msg);
}

void CHttp::expired(CBase*)
{
	CALLSTACKITEM(_L("CHttp::expired"));

	switch(current_state) {
	case RESULT_OK_WAIT:
		current_state=CONNECTED;
		if (!HandleFile()) RetryHandleFile();
		break;
	case READING_FILE:
	case RETRY_HANDLE:
		if (iRetryCount==MAX_RETRIES) {
			iRetryCount=0;
			iFileSize=0;
			iOffset=0;
			observer.error(this, iLastError, _L("cannot open file"));
		} else {
			observer.info(this, _L("retrying"));
			if (current_state==READING_FILE) {
				if (ReadFile()) {
					iRetryCount=0;
					if (!HandleFile()) RetryHandleFile();
				}
			} else {
				if (!HandleFile()) RetryHandleFile();
			}
		}
		break;
	default:
		if (iRetryCount==MAX_RETRIES) {
			iRetryCount=0;
			iFileSize=0;
			iOffset=0;
			observer.error(this, -1, _L("timed out"));
		} else {
			++iRetryCount;
			observer.info(this, _L("retrying"));
			Connect(iIapId, iHost);
		}
		break;
	}
}

void CHttp::Store(const TDesC& filename, const TDesC& meta, HBufC8* Packet)
{
	CALLSTACKITEM(_L("CHttp::Store"));

	input.Close();
	iFilename=filename;
	iOffset=0;
	iRetryCount=0;
	iMeta=meta;
	iPacket=Packet;
	last_part=false;

	if (ReadFile()) {
		if (!HandleFile()) RetryHandleFile();
	} else {
		++iRetryCount;
		iWait->Wait(5);
	}
}

void CHttp::RetryHandleFile()
{
	++iRetryCount;
	current_state=RETRY_HANDLE;
	iWait->Wait(5);
}

bool CHttp::ReadFile()
{
	current_state=READING_FILE;
	iLastError=KErrNone;
	iLastError=input.Open(Fs(), iFilename, EFileRead|EFileShareAny);
	if (iLastError!=KErrNone) {
		return false;
	} 
	iLastError=input.Size(iFileSize);
	if (iLastError!=KErrNone) {
		input.Close();
		return false;
	} 
	return true;
}

void CHttp::AppendUrlEncoded(TDes& into, const TDesC& str)
{
	int i;
	for (i=0; i<str.Length(); i++) {
		TChar c(str[i]);
		TUint cval=(TUint)c & 0xff;
		if (c.IsAlphaDigit() && cval<0x7f) into.Append(c);
		else {
			TBuf<5> b;
			b.Format(_L("%%%02x"), cval);
			into.Append(b);
		}
	}
}

bool CHttp::HandleFile()
{
	TRAPD(err, HandleFileL());
	if (err==KErrNone) return true;
	TBuf<40> msg=_L("Error in HandleFileL: ");
	msg.AppendNum(err);
	observer.info(this, msg);
	return false;
}

void CHttp::HandleFileL()
{
	CALLSTACKITEM(_L("CHttp::HandleFileL"));

	TFileName name;
	TParse parse;
	parse.Set(iFilename, 0, 0);
	name=parse.NameAndExt();

	if (iRetryCount==0 || !iURL) {
		if (! iHost.Left(5).Compare(_L("https")) ) {
			MAXSIZE=7*1024;
		}
		if (last_part) {
			observer.success(this);
			return;
		}
		if (iOffset>=iFileSize) {
			input.Close();
			last_part=true;
		}

		bool first_part=false;
		if (iOffset==0) first_part=true;

		TInt size=0;
		if (!last_part) {
			size=iFileSize-iOffset;
			if (size>MAXSIZE) size=MAXSIZE;
		} else if (iPacket) {
			size=iPacket->Length();
		}

		if (!iContents || (iContents->Des().MaxSize()/2<size) ) {
			delete iContents; iContents=0;
			iContents=HBufC8::NewL(size);
		}
		iContents->Des().Zero();
		TPtr8 ptr=iContents->Des();
		if (!last_part) {
			input.Read(ptr, size);
		} else if (iPacket) {
			ptr=*iPacket;
		}
		
		head.Zero();

		head.Append(_L("POST "));
		TInt urllen=1;
		urllen+=iHost.Length()+10;
		urllen+=iPath.Length()+2;
		urllen+=name.Length()*3;
		urllen+=iMeta.Length()*3+2;
		if (!iURL || (iURL->Des().MaxSize()/2<urllen)) {
			delete iURL; iURL=0;
			iURL=HBufC::NewL(urllen);
		}
		iURL->Des().Zero();
		TBuf<1> op;
		if (first_part)
			op=_L("C");			
		else if (!last_part)
			op=_L("A");
		else 
			op=_L("P");

		iURL->Des().Append(iHost); iURL->Des().Append(_L("?"));
		iURL->Des().Append(op); iURL->Des().Append(_L(";"));
		TPtr16 d=iURL->Des();
		AppendUrlEncoded(d, iMeta); iURL->Des().Append(_L(";"));
		iURL->Des().AppendNum(iOffset); iURL->Des().Append(_L(";"));
		iURL->Des().Append(iPath); iURL->Des().Append(_L("/"));
		d=iURL->Des();
		AppendUrlEncoded(d, name);

		head.Append(*iURL);
		head.Append(_L(" HTTP/1.0\r\n"));

		// Content-type first in POST
		head.Append(_L("Content-Type: application/octet-stream\r\n"));
		// Content-length:
		head.Append(_L("Content-Length: "));
		head.AppendNum(size);
		head.Append(_L("\r\n\r\n"));
		iOffset+=size;

	} 
	TBuf<100> msg;
	msg.Format(_L("S %S p %d"), &name, TInt(iOffset/MAXSIZE));
	observer.info(this, msg);
	
	delete iSocket; iSocket=0;
#ifndef __S60V2__
	iSocket=CProtoSocket::NewL(*this, iOwnedServ);
#else
	iSocket=CProtoSocket::NewL(*this, iOwnedServ, iOwnedConnection);
#endif

#ifdef __WINS__
	static p=0; 
	p++;
	if (p % 10 == 2) User::Leave(-1029);
#endif

	if (!iAddrKnown) {
		TBuf<100> hostname;
		TInt first_slash=iHost.Locate('/');
		hostname=iHost.Mid(first_slash+2);
		hostname=hostname.Mid(0, hostname.Locate('/'));
		if (! iHost.Left(5).Compare(_L("https")) ) {
			iSocket->Connect(hostname, 443, 60, CSocketBase::ESSL);
		} else {
			iSocket->Connect(hostname, 80, 60, 0);
		}
	} else {
		if (! iHost.Left(5).Compare(_L("https")) ) {
			iSocket->Connect(iAddr, 60, CSocketBase::ESSL);
		} else {
			iSocket->Connect(iAddr, 60);
		}
	}
	current_state=CONNECTING_SOCKET;
}

void CHttp::ConstructL()
{
	CALLSTACKITEM(_L("CHttp::ConstructL"));

	iWait=CTimeOut::NewL(*this);

	User::LeaveIfError(iOwnedServ.Connect());
#ifndef __S60V2__
	iInitiator=CConnectionOpener::NewL(*this, iOwnedServ);
#else
	User::LeaveIfError(iOwnedConnection.Open(iOwnedServ));
	iInitiator=CConnectionOpener::NewL(*this, iOwnedServ, iOwnedConnection);
#endif

	User::LeaveIfError(file.Replace(Fs(), _L("c:\\http.txt"), EFileWrite|EFileShareAny));
}

CHttp::~CHttp()
{
	CALLSTACKITEM(_L("CHttp::~CHttp"));

	file.Close();

	delete iWait;
	delete iInitiator;
	delete iSocket;

	delete iContents;
	delete iURL;
#ifdef __S60V2__
	iOwnedConnection.Close();
#endif
	iOwnedServ.Close();
}

void CHttp::Connect(TUint32 IapID, const TDesC& UrlBase)
{	
	CALLSTACKITEM(_L("CHttp::Connect"));

	iHost=UrlBase;
	iIapId=IapID;
	current_state=CONNECTING;
	iAddrKnown=false;

	iInitiator->MakeConnectionL(IapID);
}

void CHttp::Close()
{
	CALLSTACKITEM(_L("CHttp::Close"));

	if (iSocket) iSocket->CloseImmediate();
	delete iContents; iContents=0;
	delete iURL; iURL=0;
	delete iSocket; iSocket=0;
	if (iInitiator && iInitiator->MadeConnection()) iInitiator->CloseConnection();
	observer.success(this);
}

void CHttp::Cwd(const TDesC& dir)
{
	CALLSTACKITEM(_L("CHttp::Cwd"));

	iPath=dir;
}
