package maito.browsing;

import maito.resource.*;
import maito.util.Tools;
import maito.util.DbTools;

import java.sql.*;
import java.util.Properties;
import java.util.Vector;

/**
* A class that takes care of all database related operations for the class maito.browsing.ResourceBrowser 
*  
* @author Antti Laitinen
*/
public class DatabaseManager {

    private Connection connection;
    
    private Properties config;
    
    private String[] graphs;
    
    private String currentGraph;
    
    private ResourceLoader loader;
    
    private String latestQuery;
    
    /**
     * Creates a new DatabaseManager object.
     * @throws SQLException
     * Thrown when initialization of the database connection fails
     */
    protected DatabaseManager() throws SQLException {
        
        //load config file
        this.config = Tools.loadProperties(Tools.PATH_DBCONFIG);
        if (config == null) {
            throw new SQLException("Could not load database configuration file");
        }
        
        this.connection = DbTools.createDbConnection(config);
        this.connection.setAutoCommit(true);
        
        this.graphs = this.getGraphDbNames();
        
        this.loader = new ResourceLoader(this.connection);
        
        this.latestQuery = null;
    }
    
    /**
     * Executes an SQL query in the database. Creates and returns Resource objects reflecting each row.  
     * @param sql
     * The whole query.
     * @param database
     * The name of the database where the query is executed in.
     * @return
     * An array of Resource objects each representing a single row. null if the query doesn't produce any Resource rows.
     */
    protected void executeSQL(String sql, String graph) throws SQLException {

        if(sql == null || graph == null) {
            return;
        }

        /*remove useless whitespace from the query*/
        sql = sql.trim();
        
        /*make sure that the query doesn't have a ; at end*/
        if(sql.endsWith(";")) {
            sql = new StringBuffer(sql).deleteCharAt(sql.lastIndexOf(";")).toString();
        }
        
        /*store the graph (same as databasename) that we are working on*/
        this.currentGraph = graph;
        
        /*select the database that we are going to use*/
        Statement stmt = this.connection.createStatement();
        stmt.execute("use " + graph + ";");
        
        /*Determine if the query will modify the database.
         *If it only reads the database it is handled differently. */
        boolean isQuery = sql.startsWith("select");
        
        if(isQuery) {
            this.latestQuery = sql;
            this.loader.executeSQL(sql);
        }
        else {
            this.latestQuery = null;
            stmt.execute(sql);
        }
    }
    
    protected void executeLatestQuery() throws SQLException {
        if(this.latestQuery != null) {
            this.loader.executeSQL(this.latestQuery);
        }
    }
    
    
    /**
     * Fetches the next resource produced by the latest query (by the method executeSQL()).
     * @return
     * A Resource object. null if the latest query didn't produce any Resources or if
     * all Resources are already fetched using this method.
     * @throws SQLException
     */
    protected Resource nextResource() throws SQLException {
        return this.loader.nextResource();
    }
    
    
    
    /**
     * Returns all available resource graphs (database names).
     * @return
     * An array of existing database names. If no resource graphs exist the array is empty.
     */
    protected String[] getGraphs() {
        return this.graphs;
    }
    
    /**
     * Fetches all relations for a single resource.
     * @param resource
     * @return
     * An array of the ResourceRelation object. null if no relations exist.
     * @throws SQLException
     */
    protected ResourceRelation[] getRelations(Resource resource) throws SQLException {
        
        String id = resource.getID();
        
        /*escape ' characters*/
        id = id.replaceAll("'", "''");
        
        Vector relations = new Vector();
        
        Statement stmt = this.connection.createStatement();
        
        stmt.execute("use " + this.currentGraph);
        
        ResultSet rs = stmt.executeQuery("select * from ResourceRelation where subject = '" + id + "';");
        
        while(rs.next()) {
             ResourceRelation relation = new ResourceRelation();
            
            relation.setSubject(rs.getString("subject"));
            relation.setObject(rs.getString("object"));
            relation.setRole(rs.getString("role"));
            
            relations.add(relation);
        }
        
        ResourceRelation[] relationArray = null;
        
        if(relations.size() > 0) {
            relationArray = new ResourceRelation[relations.size()];
            relationArray = (ResourceRelation[])relations.toArray(relationArray);
        }
        
        return relationArray;
    }
    
    
    
    /**
     * Fetches the names of all databases that contains a resource graph.
     * @return
     * An array of the names. If no resource graphs exist the array is empty.
     */
    private String[] getGraphDbNames() {
        
        String[] databases = new String[0];
        
        try {
            Statement stmt = this.connection.createStatement();
            
            ResultSet rs = stmt.executeQuery("show databases;");
            
            Vector v = new Vector();
            
            String dbNamePrefix = this.config.getProperty("dbname_resources");
            
            while(rs.next()) {
                String item = rs.getString(1);
                
                /*find out if database is a resource graph*/
                if(item.startsWith(dbNamePrefix)) {
                    v.add(item);
                }
            }
        
            rs.close();
            
            databases = new String[v.size()];
            
            for( int i = 0 ; i < databases.length ; i++) {
                databases[i] = (String)v.get(i);
            }
        }
        catch(SQLException e) {
            //do nothing. an empty array is returned.
        }        
        
        return databases;
    }
}
