package maito.browsing;

import maito.resource.*;
import maito.browsing.fileexport.*;

import java.util.*;
import java.sql.SQLException;

/**
* A class that can be used for creating a view into the data in the resource database. 
* Data is retrieved through sql queries. After a query is executed the data is available
* through the getResources() and getRelations() methods. Queries can also be used for deleting 
* data from the database. 
* 
* @author Antti Laitinen
*/
public class ResourceBrowser {
    
    private HashMap relations; //maps a resource's id to a ResourceRelation array
        
    private DatabaseManager dbManager;
    
    private FileExporter exporter;

    
    /**
     * Creates a new ResourceBrowser instance.
     * @throws RuntimeException
     * Thrown when a database connection cannot be created.
     */
    public ResourceBrowser() throws RuntimeException {

        try {
            this.dbManager = new DatabaseManager();
        }
        catch(SQLException e) {
            throw new RuntimeException("cannot connect to database");
        }
        
        this.relations = new HashMap();
        
        this.exporter = new FileExporterImpl();
    }
    
    /**
     * Returns all existing resource graphs.
     * @return
     * An array containing the database name of each graph. If no resource graphs exist the array is empty.
     */
    public String[] getGraphs() {
        return this.dbManager.getGraphs();
    }
    
    /**
     * Returns the next Resource object produced by the latest query.
     * @return
     * A Resource object. null if the latest query didn't produce any Resources or if
     * all Resources are already fetched using this method.
     */
    public Resource nextResource() throws SQLException{
        return this.dbManager.nextResource();
    }
    
    
    /**
     * Returns all file formats that are supported by the FileExporter.
     * @return
     * An array containg the names of the supported file formats.
     */
    public String[] getFileFormats() {
        return this.exporter.getFormats();
    }
    
    /**
     * Executes an SQL query in the resource database. The resulting data can be later retrieved using the getters in this class. 
     * @param sql
     * The SQL query to be executed. The query must be syntactically correct SQL. 
     * Table names must be in format: <code>database.table</code>.
     * @param graph
     * The name of the database where the query is executed. 
     * Must be one of the Strings returned by getGraphs().
     */
    public void executeSQL(String sql, String graph) throws SQLException{
        
        
        this.dbManager.executeSQL(sql, graph);
    }
    
    /**
     * Writes the current view into a file.
     * @param filename
     * The path and filename of the file.
     * @param format
     * The format of the file. Must one of the supported formats.
     * @return 
     * A boolean telling whether the file was saved successfully.
     */
    public boolean writeToFile(String filename, String format) {
        
        boolean success = false;
        
        try {
            this.dbManager.executeLatestQuery();
            
            Vector resources = new Vector();
            
            Resource resource = this.dbManager.nextResource(); 
            
            while(resource != null) {
                resources.add(resource);
                ResourceRelation[] rel = this.dbManager.getRelations(resource);
                if(rel != null) {
                    this.relations.put(resource.getID(),rel);
                }
                resource = this.dbManager.nextResource();
            }
            
            Resource[] resourceArray = new Resource[resources.size()];
            resourceArray = (Resource[])resources.toArray(resourceArray);
            
            success = this.exporter.writeFile(filename, format, resourceArray, this.getRelations() );
        
        }
        catch(SQLException e) {
            success = false;
        }
        return success;
    }
    

    /**
     * Returns all resource relations between all the resources in the current view. 
     * @return
     * An array containing the ResourceRelation objects or null if no relations exists.
     */
    private ResourceRelation[] getRelations() {
        
        /*there is one ResourceRelation[] per Resource*/
        Vector relationArrays = new Vector(this.relations.values());
        
        Vector allRelations = new Vector();
        
        for(int i = 0 ; i < relationArrays.size(); i++) {
            
            ResourceRelation[] tmp = (ResourceRelation[])relationArrays.get(i);
            
            for( int j = 0 ; j < tmp.length ; j++) {
                allRelations.add(tmp[j]);
            }
        }
        
        ResourceRelation[] result = new ResourceRelation[allRelations.size()];
        allRelations.copyInto(result);
        
        return result;
    }
    
    
    /*
     * test main
     */
    public static void main(String[] args) {
        
       
    }
}
