package maito.browsing.fileexport;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Properties;

import maito.resource.Resource;
import maito.resource.ResourceRelation;
import maito.util.Tools;

/**
 * A FileFormat implementation for CSV resource / relation lists.
 * 
 * @author Väinö Ala-Härkönen
 */
public class CSVResourceListFormat implements FileFormat {
    
    // the fields included
    private boolean includeID = false;
    private boolean includeKeyWords = false; 
    private boolean includeExtIDs = false;
    private String[] singleFields;
    
    // config file related
    private final String EXPORTFIELDCONFIG_PATH = "config" + File.separator + "csvlistexport.properties";
    private Properties exportFieldMap;
    
    // holders for the data
    private Resource[] resources = null;
    private ResourceRelation[] relations = null;    
    
    private boolean append = false;
    
    public CSVResourceListFormat() {
        exportFieldMap = Tools.loadProperties(EXPORTFIELDCONFIG_PATH);
        if (exportFieldMap == null) {
            System.out.println("ERROR: Could not load csvlistexport.properties");
        }
    }
    
    /**
     * Sets the resources that are to be written to the file.
     * @param resources
     * The resources that are to be written to the file. The
     * possible previous set will be replaced.
     */
    public void setResources(Resource[] resources) {
        this.resources = resources;
    }
    
    /**
     * Sets the relations that are to be written to the file.
     * @param relations
     * The relations that are to be written to the file. The
     * possible previous set will be replaced.
     */
    public void setRelations(ResourceRelation[] relations) {
        this.relations = relations;
    }

    /**
     * Writes the data set by setResources and setRelations to a file
     * that's in CSV list format: first row has headers of the fields
     * and the following rows have the data. The data fields are separated with
     * commas and enclosed in double quotes.
     * 
     * If resources or relations have been set, writes one file with the
     * non-null set. If both resources and relations have been set, writes
     * two separate files with suffixes -resources and -relations before the
     * first dot in the filename (ie. dump-resources.csv, dump-relations.csv).
     * 
     * The fields included in export are defined in configuration file
     * csvlistexport.properties.
     * 
     * @param filename 
     * the full path and name of the file to write the data to
     * @return 
     * true if succeeded, false if there was an error: either the 
     * file could not be written or source data was invalid.
     */
    public boolean writeFile(String filename) {        
        String resourcesFilename = null;
        String relationsFilename = null;
        
        if (exportFieldMap == null) {
            return false;
        }
        
        // check the properties - which fields should we include in the export?
        if (exportFieldMap.getProperty("includeID").trim().equals("1")) 
            includeID = true;
        
        if (exportFieldMap.getProperty("includeKeyWords").trim().equals("1")) 
            includeKeyWords = true;
        
        if (exportFieldMap.getProperty("includeExtIDs").trim().equals("1")) 
            includeExtIDs = true;
        
        String exportFields = exportFieldMap.getProperty("singleFields");
        if (exportFields != null) {
            singleFields = exportFields.split(",");
            for (int i = 0; i < singleFields.length; i++) {
                singleFields[i] = singleFields[i].trim();
                
            }
        }
        
        /* Determine whether we should save resources, relations or both
         * and construct file names accordingly
         */        
        if (this.resources != null && this.relations != null) {
            resourcesFilename = filename + "-resources";
            relationsFilename = filename + "-relations";
            int splitPoint = filename.lastIndexOf(".");
            if (splitPoint != -1) {
                resourcesFilename = filename.substring(0, splitPoint) + "-resources." + filename.substring(splitPoint+1, filename.length());
                relationsFilename = filename.substring(0, splitPoint) + "-relations." + filename.substring(splitPoint+1, filename.length());
            }
        }
        else if (this.resources != null)
            resourcesFilename = filename;
        else if (this.relations != null)
            relationsFilename = filename;
        else 
            return false;
        
        
        // Do the saving
        if (resourcesFilename != null) {
            boolean resSuccess = this.saveResources(resourcesFilename);
            if (resSuccess == false)
                return false;
        }
        if (relationsFilename != null) {
            boolean relSuccess = this.saveRelations(relationsFilename);
            if (relSuccess == false)
                return false;
        }
        
        return true;
    }
    

    /*
     * Saves resources to a file as a CSV list
     */
    private boolean saveResources(String resourcesFilename) {
        String thisField, temp;
        boolean notFirst = false; // whether the field is first in this row or not
        BufferedWriter file = null;
        
        try {
            file = new BufferedWriter(new OutputStreamWriter(
                    new FileOutputStream(resourcesFilename, append), "UTF-8"));
            
            /* 
             * Header
             */
            if (includeID) {
                file.write("\"Resource id\"");
                notFirst = true;
            }
            if (includeKeyWords) {
                if (notFirst)
                    file.write(",");
                file.write("\"KeyWords\"");
                notFirst = true;
            }
            if (includeExtIDs) {
                if (notFirst)
                    file.write(",");
                file.write("\"ExtIDs\"");
                notFirst = true;
            }
            for (int i = 0; i < singleFields.length; i++) {
                if (notFirst)
                    file.write(",");
                file.write("\"" + singleFields[i] + "\"");
                notFirst = true;
            }
            file.write("\n");
            notFirst = false;
            
            /*
             * Data rows
             */ 
            HashSet thisKeyWords, thisExtIDs;
            Iterator ite;
            for (int i = 0; i < resources.length; i++) {
                
                if (includeID) {
                    file.write("\"" + resources[i].getID() + "\"");
                    notFirst = true;
                }
                
                // all the keywords in one field, comma separated
                if (includeKeyWords) {
                    if (notFirst)
                        file.write(",");
                    thisKeyWords = resources[i].getKeywords();
                    ite = thisKeyWords.iterator();
                    thisField = "";
                    while (ite.hasNext()) {
                        temp = (String)ite.next();
                        temp = temp.replaceAll("\"", "\"\"");
                        thisField = thisField + ", " + temp;
                    }
                    file.write("\"" + thisField + "\"");
                    notFirst = true;
                }
                
                // all the external ids in one field, comma separated
                if (includeExtIDs) {
                    if (notFirst)
                        file.write(",");
                    thisExtIDs = resources[i].getExtIDs();
                    ite = thisExtIDs.iterator();
                    thisField = "";
                    while (ite.hasNext()) {
                        temp = (String)ite.next();
                        temp = temp.replaceAll("\"", "\"\"");
                        thisField = thisField + ", " + temp;
                    }
                    file.write("\"" + thisField + "\"");
                    notFirst = true;
                }

                // all the other fields one by one
                for (int j = 0; j < singleFields.length; j++) {
                    if (notFirst)
                        file.write(",");
                    thisField = resources[i].getSingleField(singleFields[j]);
                    if (thisField != null) {
                        thisField = thisField.replaceAll("\"", "\"\"");
                        file.write("\"" + thisField + "\"");
                    }
                    notFirst = true;
                }
                
                file.write("\n");
                notFirst = false;
            }
            
            file.close();
        }
        catch (Exception e) {
            System.out.println("ERROR in CSVResourceListFormat.saveResources: " + e);
            return false;
        }
        return true;
    }
    
    
    /*
     * Saves relations to a file as a CSV list
     */
    private boolean saveRelations(String relationsFilename) {
        BufferedWriter file = null;
        String thisSubject, thisRole, thisObject;
        
        try {
            file = new BufferedWriter(new OutputStreamWriter(
                    new FileOutputStream(relationsFilename, append), "UTF-8"));
            /* 
             * Header
             */
            file.write("Subject,Role,Object\n");
            
            /*
             * Subject - role - object -mappings one by one
             * Subject and object shouldn't have double quotes in them
             * in theory but we still escape them to be sure
             */
            for (int i = 0; i < relations.length; i++) {
                thisSubject = relations[i].getSubject();
                thisRole = relations[i].getRole();
                thisObject = relations[i].getObject();
                if (thisSubject == null)
                    thisSubject = "";
                else
                    thisSubject = "\"" + thisSubject.replaceAll("\"", "\"\"") + "\"";
                if (thisRole == null)
                    thisRole = "";
                else
                    thisRole = "\"" + thisRole.replaceAll("\"", "\"\"") + "\"";
                if (thisObject == null)
                    thisObject = "";
                else
                    thisObject = "\"" + thisObject.replaceAll("\"", "\"\"") + "\"";
                file.write(thisSubject + "," + thisRole + "," + thisObject + "\n");
                
            }
            file.close();
        }
        catch (Exception e) {
            System.out.println("ERROR in CSVResourceListFormat.saveRelations: " + e);
            return false;
        }        
        return true;
    }
    
}
