package maito.browsing.fileexport;

import java.io.File;
import java.util.Enumeration;
import java.util.Properties;

import maito.resource.Resource;
import maito.resource.ResourceRelation;
import maito.util.Tools;

/**
 * Implementation of the FileExporter spec - handles
 * the exporting of resource & relation data to different
 * file formats
 * 
 * @author Väinö Ala-Härkönen
 */
public class FileExporterImpl implements FileExporter {
    
    private final String FORMATCONFIG_PATH = "config" + File.separator + "exportformats.properties";
    private String fileFormatClass;
    private Properties formatMap;
    
    public FileExporterImpl () {
        formatMap = Tools.loadProperties(FORMATCONFIG_PATH);
        if (formatMap == null) {
            System.out.println("ERROR: Could not load exportformats.properties");
        }
    }

    /**
     * 
     * @param filename 
     * Name of the file to be written
     * @param format
     * Format of the file to be written - use getFormats() to find out which ones it supports
     * @param resources
     * Resources to be saved to the file
     * @param relations
     * Relations to be saved to the file
     * @return
     * true if save succeeded, false if there was an error
     */
    public boolean writeFile(String filename, String format, 
            Resource[] resources, ResourceRelation[] relations) {
        
        fileFormatClass = formatMap.getProperty(format);
        if (fileFormatClass == null)
            return false;
        FileFormat fileFormat = null;
        try {
            fileFormat = (FileFormat)Class.forName(fileFormatClass).newInstance();
        } 
        catch (Exception e) {
            System.out.println("Error in FileExporterImpl.writeFile: " + e);
            return false;
        }
        
        fileFormat.setResources(resources);
        fileFormat.setRelations(relations);        
        boolean success = fileFormat.writeFile(filename);
        
        return success;
    }

    /**
     * Returns a list of file export formats that are supported by the software.
     * @return
     * Identifiers of export formats that the software currently supports.
     */
    public String[] getFormats() {
        if (formatMap == null)
            return null;
        Enumeration formats = formatMap.propertyNames();
        String[] formatList = new String[formatMap.size()];
        for (int i = 0; i < formatList.length; i++) {
            formatList[i] = (String)formats.nextElement();
        }
        return formatList;
    }
    
}
