package maito.browsing.fileexport;

import java.io.BufferedWriter;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.util.HashMap;

import maito.resource.Resource;
import maito.resource.ResourceRelation;

/**
 * A FileFormat implementation for simple Pajek .net graphs
 * Uses the Resource IDs as the names in the Vertex list.
 * 
 * @author Väinö Ala-Härkönen
 */
public class PajekFormat implements FileFormat {
    
    private Resource[] resources;
    private ResourceRelation[] relations;
    
    private boolean append = false;

    /**
     * Sets the resources that are to be written to the file.
     * @param resources
     * The resources that are to be written to the file. The
     * possible previous set will be replaced.
     */
    public void setResources(Resource[] resources) {
        this.resources = resources;
    }
    
    /**
     * Sets the relations that are to be written to the file.
     * Relations that contain resources which aren't
     * in the resource array (set by setResources) won't be written to the file
     * @param relations
     * The relations that are to be written to the file. The
     * possible previous set will be replaced.
     */
    public void setRelations(ResourceRelation[] relations) {
        this.relations = relations;
    }

    /**
     * Writes the data set by setResources and setRelations to a file
     * that's in Pajek .net format.
     * @param filename 
     * the full path and name of the file to write the data to
     * @return 
     * true if succeeded, false if there was an error: either the 
     * file could not be written or source data was invalid.
     */
    public boolean writeFile(String fileName) {
        
        if (resources == null)
            return false;
        
        int currentSeqNumber = 0;
        String currentID = "";
        String currentSubj = "";
        String currentObj = "";
        String currentRole = "";
        String currentSubjID = "";
        String currentObjID = "";
        String currentRoleID = "";
        HashMap resourceMap = new HashMap(resources.length);
        
        try {
            BufferedWriter file = new BufferedWriter(new OutputStreamWriter(
                                  new FileOutputStream(fileName, append), "UTF-8"));
            
            // Vertex header
            file.write("*Vertices " + resources.length + "\r\n");
            
            // Vertices one by one
            for (int i = 0; i < resources.length; i++) {
                currentID = resources[i].getID();
                // Hide the special IDs
                if (currentID == null || currentID.equals("BIBREF") || currentID.equals("NONE"))
                    currentID = "";
                currentSeqNumber++;
                resourceMap.put(currentID, currentSeqNumber + "");
                file.write(currentSeqNumber + " \"" + currentID + "\"\r\n");
            }
            
            if (relations == null) {
                file.close();
                return false;
            }
            
            // Relations header
            file.write("*Arcs\r\n");
            
            // Relations one by one
            for (int i = 0; i < relations.length; i++) {
                currentSubj = relations[i].getSubject();
                currentObj = relations[i].getObject();
                currentRole = relations[i].getRole();
                currentSubjID = (String)resourceMap.get(currentSubj);
                currentObjID = (String)resourceMap.get(currentObj);
                // both subject and object exist in the vertices list, add the relation
                if (currentSubjID != null && currentObjID != null) {
                    // if role isn't a dummy one, we must add a relation through it
                    if (!currentRole.equals("BIBREF") && !currentRole.equals("NONE") && currentRole != null) {
                        currentRoleID = (String)resourceMap.get(currentRole);
                        if (currentRoleID != null) {
                            file.write(currentSubjID + " " + currentRoleID + "\r\n");
                            file.write(currentRoleID + " " + currentObjID + "\r\n");
                        }
                    }
                    // in case of dummy role, write a straight relation
                    else
                        file.write(currentSubjID + " " + currentObjID + "\r\n");                    
                }

            }
            
            file.close();
        }
        catch (Exception e) {
            System.out.println(e);
            return false;
        }
        return true;
    }
    
}
