/**
* ConsoleUI.java
*  
* 21.11.2005
* 
* This software is released under the GNU GPL license
*/
package maito.consoleui;

import maito.datacollecting.*;
import maito.integration.*;
import maito.LogListener;

import java.io.*;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.List;
import java.util.Iterator;
import java.util.HashMap;
import java.util.HashSet;

/**
 * A class that can be used as a simple text based user interface for data collecting and data integration. 
 * 
 * @see maito.datacollecting.DataCollector
 * @see maito.integration.Integrator
 * 
 * @author Antti Laitinen
 * @author Oskari Saarekas
 *
 */
public class ConsoleUI implements LogListener {
    
    private DataCollector collector;
    private Integrator integrator;

    private BufferedReader reader;
    
    /**
     * creates a new ConsoleUI object
     *
     */
    public ConsoleUI() {
        this.collector = new DataCollectorImpl("data", "config");
        this.collector.setLogListener(this);
        this.integrator = new IntegratorImpl();
        this.reader = new BufferedReader(new InputStreamReader(System.in));

    }
    
    public void logMessage(String message) {
        System.out.println(message);
    }
    
    /**
     * starts the UI
     *
     */
    public void start() {
        String input = null;
        System.out.println("MAITO 1.0 - under GPL\n");
        do {
            printMenu();
            input = getInput("enter choice (1 - 7) or hit enter to quit :");
            
            if(input.equals("1")) {
                listSources();
            }
            else if(input.equals("2")) {
                addSource();
            }
            else if(input.equals("3")) {
                removeSource();
            }
            else if(input.equals("4")) {
                updateSource();
            }
            else if(input.equals("5")) {
                listGraphs();
            }
            else if(input.equals("6")) {
                listGraphSources();
            }
            else if(input.equals("7")) {
                integrateSource();
            }
        }
        while(!input.equals(""));
    }
    
    private void printMenu() {
        System.out.println(
        "\n"+
        "* Data collecting *          * Integration *********\n"+
        "\n"+
        "1) list sources              5) list resource graphs\n"+
        "2) add a new source          6) list graph sources\n"+
        "3) remove a source           7) integrate a source\n"+
        "4) update a source\n"
        );
    }
    
    private boolean listSources() {
        System.out.println("All configured data sources :");
        DataSourceDescription[] sources = this.collector.getSources();
        
        int sourceCount = 0;
        if (sources != null) {
            sourceCount = sources.length;
        }
        
        if (sourceCount == 0) {
            System.out.println("no data sources");
        }
        else {
            for ( int i = 0 ; i < sourceCount ; i++) {
                System.out.println(i + ": " + sources[i]);
            }
        }
        
        System.out.println();
        return (sourceCount > 0);
    }
    
    private void addSource() {
        HashMap formats = this.collector.getSupportedTypes();
        if (formats == null || formats.size() == 0) {
            System.out.println("Error: No data source formats are supported");
            return;
        }
        
        System.out.println("Supported data source types :");
        System.out.println("---");
        HashSet types = new HashSet(formats.keySet());
        Iterator i = types.iterator();
        while (i.hasNext()) {
            System.out.println(i.next());
        }        
        System.out.println("---");
        
        String type = getInput("enter data source type :");
        if (!types.contains(type)) {
            System.out.println("Error: Invalid type");
            return;
        }
        
        String[] fs = (String[])formats.get(type);
        if (fs == null) {
            System.out.println("Error: Unable to get formats for this source type");
            return;
        }
        
        System.out.println("Data formats for this source type :");
        System.out.println("---");
        for (int j = 0; j < fs.length; ++j) {
            System.out.println(fs[j]);
        }
        System.out.println("---");
        System.out.println("(NOTE: oai_citeseer should only be used for citeseer");
        System.out.println("extended database dump XML files, the standard oai-pmh");
        System.out.println("response is DCXML and won't be recognized with oai_citeseer!)\n");        
        String format = getInput("enter data source format :");
        
        boolean formatFound = false;
        for (int j = 0; j < fs.length; ++j) {
            if (fs[j].equals(format)) {
                formatFound = true;
                break;
            }
        }
        
        if (!formatFound) {
            System.out.println("Error: Invalid data format");
            return;
        }
        
        String url = getInput("enter URL (http:// or file://) for the new data source :");
        URL URL;
        try {
            URL = new URL(url);
        }
        catch (MalformedURLException e) {
            System.out.println("Error: Invalid URL");
            return;
        }
        
        try {
            this.collector.addSource(null, type, URL, format);
            System.out.println("Data source successfully added. No data added yet - update the source to fetch and transform.");
        }
        catch(Exception e) {
            System.out.println("Error: Could not add new source");
            System.out.println(e.getMessage());
        }
    }
    
    private void removeSource() {
        if (!listSources()) {
            return;
        }
        String input = getInput("enter the number of source to remove, or * for all :");
        boolean success = false;
        
        if (input.equals("*")) {
            success = this.collector.removeSources(this.collector.getSources(),true);
        }
        else {
            try {
                int source = Integer.parseInt(input);
                DataSourceDescription tmpds[] = this.collector.getSources();
                DataSourceDescription ds[] = new DataSourceDescription[1];
                ds[0] = tmpds[source];
                success = this.collector.removeSources(ds, true);
            }
            catch (Exception e) {}
        }
        
        if(success) {
            System.out.println("sources removed successfully");
        }
        else {
            System.out.println("failed to remove one or more sources");
        }
    }
    
    private void updateSource() {
        if (!listSources()) {
            return;
        }
        String input = getInput("enter the number of source to update, or * for all :");

        DataSourceDescription[] ds = null;
        
        if (input.equals("*")) {
            ds = this.collector.getSources();
        }
        else {
            try {
                int source = Integer.parseInt(input);
                DataSourceDescription tmpds[] = this.collector.getSources();
                ds = new DataSourceDescription[1];
                ds[0] = tmpds[source];
            }
            catch (Exception e) {
                ds = null;
            }
        }
        
        if (ds == null) {
            System.out.println("Update failed");
            return;
        }
        
        this.collector.updateSources(ds);
        
        while(this.collector.workInProgress()) {
            try {
                Thread.sleep(500);
            }
            catch(Exception e) {
                e.printStackTrace();
            }
        }
    }
    
    private boolean listGraphs() {
        System.out.println("Existing resource graphs :");
        List graphs = this.integrator.getGraphs();
        
        if (graphs == null || graphs.isEmpty()) {
            System.out.println("no graphs");
            return false;
        }
        
        Iterator i = graphs.iterator();
        while (i.hasNext()) {
            System.out.println((String)i.next());
        }
        
        System.out.println();
        return true;
    }

    private void listGraphSources() {
        if (!listGraphs()) {
            return;
        }
        String input = getInput("enter the name of the graph to show sources for :");
        
        List sources = null;
        try {
            sources = this.integrator.getSources(input);
        }
        catch (IllegalArgumentException e) {
            System.out.println("invalid graph");
            return;
        }

        if (sources == null || sources.isEmpty()) {
            System.out.println("no sources");
            return;
        }

        Iterator i = sources.iterator();
        while (i.hasNext()) {
            String[] tempArr = (String[])i.next();
            System.out.println("- " + tempArr[0] + " (" + tempArr[1] + ")");
        }
        System.out.println();
    }
    
    private void integrateSource() {
        listGraphs();
        String graph = getInput("enter the name of the destination graph (new or existing) :");
        
        if (!listSources()) {
            return;
        }
        String input = getInput("enter the number of source to integrate, or * for all :");
        
        DataSourceDescription ds[] = null;
        
        if (input.equals("*")) {
            ds = this.collector.getSources();
        }
        else {
            try {
                int source = Integer.parseInt(input);
                DataSourceDescription tmpds[] = this.collector.getSources();
                ds = new DataSourceDescription[1];
                ds[0] = tmpds[source];
            }
            catch (Exception e) {
                ds = null;
            }
        }
        
        try {
            this.integrator.addGraph(graph);
        }
        catch (IllegalArgumentException e) {
            System.out.println("Invalid graph name");
            return;
        }
        
        try {
            this.integrator.integrate(ds, graph);
        }
        catch (IntegrationException e) {
            System.out.println("Integration failed: " + e.getMessage());
            return;
        }
        
        while(this.integrator.workInProgress()) {
            try {
                Thread.sleep(500);
            }
            catch(Exception e) {
                e.printStackTrace();
            }
        }
        System.out.println("Done\n");
    }
    
    private String getInput(String prompt) {
        System.out.println(prompt);
        System.out.print(">");
        String input = "";
        try {
            input = reader.readLine();
        }
        catch(Exception e) {
            e.printStackTrace();
            System.exit(1);
        }
        
        return input;
    }
    
    public static void main(String[] args) {
        new ConsoleUI().start();
    }
}
