package maito.datacollecting;

import maito.DataProcessor;

import java.net.URL;
import java.util.HashMap;

/**
 * An inteface for using the data collecting features of this program. 
 * This interface is designed to be used by the UI. 
 * @author Antti Laitinen
 *
 */

public interface DataCollector extends DataProcessor {
	
    /**
     * Adds a single source to this DataCollector's data sources that will be updateble in the future.
     * @param name 
     * A name/id for this source. This is the name that is visible to the user.
     * @param type
     * The type of this source. Must be one of the following:
     * <ul>
     * <li><code>"file"</code></li>
     * <li><code>"OAI-PMH"</code></li>
     * </ul>
     * @param location
     * The location where this source's data is retrieved from.
     * @param format
     * The data format. Must be one of the following:
     * <ul>
     * <li><code>"quick_format_name"</code></li>
     * <li><code>"quick_format_document"</code></li>
     * <li><code>"DCXML"</code></li>
     * <li><code>"oai_citeseer"</code></li>
     * </ul>
     * @return
     * A boolean value telling whether the given source was successfully added.
     * 
     * @throws IllegalArgumentException
     * Thrown when something is wrong with the parameters so that a new data source cannot be created.
     */
    public boolean addSource(String name, String type, URL location, String format) throws IllegalArgumentException;
    
    
    /**
     * Returns a description of each source that this DataCollector has.
     * @return
     * An array of DataSourceDescription objects. The array is empty if this DataCollector has no sources. 
     */
    public DataSourceDescription[] getSources();
    
    
	/**
     * Starts updating data sources. 
     * @param sources
     * The data sources that are to be updated.
	 */
	public void updateSources(DataSourceDescription[] sources);	
    
      
	/**
	 * Removes one or more data sources permanently. The data source will no longer be updated. All data from the database will be removed.
	 * @param sources
	 * The sources that are to be removed.
     * @param removeData
     * If true also all raw data is deleted from disk. If false the raw data is left alone. 
     * @return
     * A boolean value telling whether the sources were successfully removed.
	 */
	public boolean removeSources(DataSourceDescription[] sources, boolean removeData);

    
    /**
     * Returns all source types that are supported by this DataCollector. 
     * @return
     * A HashMap containing mappings from a transfer type to data format.
     * The key values are String objects containing a supported transfer type.
     * The keys map into String[] array objects that contain all supported data formats 
     * for the transfer type.
     * 
     * example:
     * "oaipmh" -> {"DCXML","oai_citeseer"}
     * (key "oaipmh" maps to an array that contains "DCXML" and "oai_citeseer")
     */
    public HashMap getSupportedTypes(); 
}
