package maito.datacollecting;

import maito.LogListener;
import maito.util.*;

import java.net.URL;
import java.util.*;
import java.text.SimpleDateFormat;
import java.io.File;
import java.io.FileInputStream;
import javax.xml.parsers.*;
import org.w3c.dom.*;
import java.sql.*;


/**
 *  
 * @author Antti Laitinen
 */
public class DataCollectorImpl implements DataCollector {

    private Vector updaterThreads; // contains UpdateThread objects (an inner class)

    private File[] sourceConfFiles;

    private File sourceTypesConfFile;

    private String dataDir;

    private Properties dbSettings;

    private LogListener logListener;

    private HashMap sources; // maps data source IDs to DataSource objects

    private Vector errors; // Contains descriptions of errors as String objects

    private DocumentBuilderFactory factory;

    private DocumentBuilder builder;

    private Connection dbConnection;

    /* data source configuration file specific settings. */

    private final String CONF_FILE_IDENTIFIER = ".source";

    private final String SOURCEID_XPATH = "/source/id";

    private final String DATASOURCE_XPATH = "/source/datasource/class";

    private final String RECORDPARSER_XPATH = "/source/recordparser/class";

    private final String RECORDCONSTRUCTOR_XPATH = "/source/recordconstructor/class";

    private final String TRANSFORMER_XPATH = "/source/transformer/class";

    private final String ORIGINALLOCATION_XPATH = "/source/location/original";

    /* other settings */

    private final String RAWDATA_FILE_IDENTIFIER = ".rawdata";

    private final String DBCONFIG_FILENAME = "dbconfig.properties";

    private final String SOURCETYPESXML_FILENAME = "sourcetypes.xml";

    private final String DBNAME_KEY = "dbname_rawdata";

    private final String DATE_FORMAT = "yyyy-MM-dd";

    /**
     * Creates a new DataCollectorImpl instance. Reads all configuration files
     * and prepares data sources for updating.
     * 
     * @param dataDir
     *            The directory that contains configuration files for each data
     *            source.
     * @param configDir
     *            The directory that contains configuration files used by the
     *            whole program (for example database configuration).
     * @throws RuntimeException
     *             Thrown if something goes wrong in the initialization.
     */
    public DataCollectorImpl(String dataDir, String configDir)
            throws RuntimeException {
        this.dataDir = dataDir;
        this.errors = new Vector();
        this.updaterThreads = new Vector();

        /* confirm access to configuration directory */
        if (!new File(configDir).isDirectory()) {
            throw new IllegalArgumentException(
                    "cannot access configuration files: " + configDir
                            + " is not a directory");
        }

        /* load the database configuration */
        this.dbSettings = Tools.loadProperties(configDir + File.separator
                + this.DBCONFIG_FILENAME);

        if (this.dbSettings == null) {
            throw new RuntimeException("failed to load database configuration");
        }

        /* initialize the database if needed */
        try {
            this.initDatabase();
        } catch (SQLException e) {
            throw new RuntimeException("database initialization failed: "
                    + e.getMessage());
        }

        /* create a connection to the database */
        try {
            this.dbConnection = DbTools.createDbConnection(this.dbSettings,
                    this.dbSettings.getProperty(this.DBNAME_KEY));
            this.dbConnection.setAutoCommit(true);
        } catch (SQLException e) {
            throw new RuntimeException("cannot create database connection: "
                    + e.getMessage());
        }

        /* confirm access to source types configuration file */
        this.sourceTypesConfFile = new File(configDir + File.separator
                + this.SOURCETYPESXML_FILENAME);

        if (!this.sourceTypesConfFile.exists()) {
            throw new RuntimeException("configuration file "
                    + this.SOURCETYPESXML_FILENAME + " was not found");
        }

        /* prepare for xml parsing */
        try {
            this.factory = DocumentBuilderFactory.newInstance();
            this.builder = factory.newDocumentBuilder();
        } catch (Exception e) {
            throw new RuntimeException("cannot prepare for xml parsing: "
                    + e.getMessage());
        }

        /*
         * read existing data source configuration files and create DataSource
         * objects
         */
        File dir = new File(dataDir);

        if (!dir.isDirectory()) {
            throw new IllegalArgumentException(dataDir
                    + " should be a directory");
        }

        this.sourceConfFiles = this.getConfFiles(dir);

        this.sources = this.createDataSources(this.sourceConfFiles);

        if(this.sources == null) {
            throw new RuntimeException("failed to initialize data sources");
        }
        
        /**
         * FIXME If a source doesn't exist in the database then it's added
         * there. Since the format of each data source is only stored in the
         * database, the format is set to 'unknown'.
         */

        Set existingSources = this.sources.keySet();
        Iterator iter = existingSources.iterator();

        for (int i = 0; i < existingSources.size(); i++) {

            String existingSource = (String) iter.next();
            try {
                if (!this.sourceExistsInDatabase(existingSource)) {
                    /* NOTE: url and location are assumed to be the same */
                    this.addSourceToDatabase(existingSource, "unknown",
                            existingSource);
                }
            } catch (SQLException e) {
                throw new RuntimeException("database error: " + e.getMessage());
            }
        }
    }

    public HashMap getSupportedTypes() {

        HashMap types = new HashMap();

        String xPathTransferTypes = "/sourcetypes/rule/@transfertype";

        String[] transferTypes = null;

        try {
            transferTypes = XMLTools.getAllContents(this.builder,
                    new FileInputStream(this.sourceTypesConfFile),
                    xPathTransferTypes);

            for (int i = 0; i < transferTypes.length; i++) {

                if (!types.containsKey(transferTypes[i])) {

                    String[] formats = XMLTools.getAllContents(this.builder,
                            new FileInputStream(this.sourceTypesConfFile),
                            "/sourcetypes/rule[@transfertype = '"
                                    + transferTypes[i] + "']/@sourceformat");

                    types.put(transferTypes[i], formats);

                }
            }
        } catch (Exception e) {
            // TODO handle exception
            e.printStackTrace();
        }

        return types;
    }

    public boolean addSource(String name, String type, URL location,
            String format) throws RuntimeException {

        /*The path to the data source's data*/
        String path = this.getPath(location);
        
        /*if the source is a file see that it exists and is readable*/
        if(location.getProtocol().equalsIgnoreCase("file")) {
            File testFile = new File(path);
            
            if(!testFile.exists()) {
                throw new RuntimeException("the file " + path + " doesn't exist");
            }
            
            if(!testFile.canRead()) {
                throw new RuntimeException("the file " + path + " is not readable");
            }
        }
        
        /* The path is also used as an id for data sources */
        String id = path; 


        /* see that a data source with the given URL doesn't already exist. */
        try {
            if (this.sourceExistsInDatabase(id)) {
                throw new RuntimeException("source with location " + id
                        + " already exists");
            }
        } catch (SQLException e) {
            throw new RuntimeException("database error: " + e.getMessage());
        }

        /* update database with the new source */
        try {
            this.addSourceToDatabase(id, format, path);
        } catch (SQLException e) {
            throw new RuntimeException(
                    "unable to update database with data source " + id
                            + "\nreason: " + e.getMessage());
        }

        /* create a configuration file for the new data source */
        File confFile = this.createSourceConfFile(type, path, format);

        /* Add the new data source to the collection in memory */
        // TODO do this in a nicer way
        File[] param = new File[1];
        param[0] = confFile;
        HashMap newSource = this.createDataSources(param);
        this.sources.putAll(newSource);

        return true;
    }

    public DataSourceDescription[] getSources() {

        Vector descriptions = new Vector();

        Set sourceIDs = this.sources.keySet();
        
        Iterator iter = sourceIDs.iterator();

        String id = null;

        while (iter.hasNext()) {
            id = (String) iter.next();

            /*initial values if no data is found*/
            String updated = "no successful update";
            String modified = "not modified";
            long size = -1;
            String format = "no format";
            boolean integrated = false;

            try {
                /* find out data source's details from the database */
                Statement stmt = this.dbConnection.createStatement();
                ResultSet rs = stmt.executeQuery("select * from DataSource where id='"
                                + id + "';");

                if (rs.next()) {
                    
                    String value = rs.getString("updated"); 
                    if(value != null) {
                        updated = value;
                    }
                    
                    value = rs.getString("modified"); 
                    if(value != null) {
                        modified = value;
                    }
                    
                    size = rs.getLong("lineCount");
                    
                    value = rs.getString("format"); 
                    if(value != null) {
                        format = value;
                    }

                    rs.close();
                }
                /* Get all this source's records which are integrated somewhere */
                rs = stmt.executeQuery("select integratedTo from DataRecord where source='"
                                + id + "' and integratedTo IS NOT NULL;");
                /*
                 * This source is seen as integrated if at least one integrated
                 * record exists
                 */
                integrated = rs.next();
                
                stmt.close();
                rs.close();
                
            } catch (SQLException e) {
                e.printStackTrace();
                this.addError("database error: couldn't read properties for data source: " + id);
            }

            descriptions.add(new DataSourceDescription(id, updated, modified,
                    size, format, integrated));
        }

        DataSourceDescription[] array = new DataSourceDescription[descriptions.size()];

        for (int i = 0; i < array.length; i++) {
            array[i] = (DataSourceDescription) descriptions.get(i);
        }

        return array;
    }

    public void updateSources(DataSourceDescription[] sources) {

        /* remove any errors from previous updates */
        this.errors.clear();

        Set sourceIDs = this.sources.keySet();
        Iterator iter = sourceIDs.iterator();

        for (int i = 0; i < this.sources.size(); i++) {

            String id = (String) iter.next();

            for (int j = 0; j < sources.length; j++) {

                if (sources[j].getId().equals(id)) {
                    DataSource source = (DataSource) this.sources.get(id);

                    UpdateThread updater = new UpdateThread(this, source, id);
                    this.updaterThreads.add(updater);
                    updater.start();

                    this.addLogMessage("updating source: " + id);
                }
            }
        }
    }

    public boolean removeSources(DataSourceDescription[] sources,
            boolean removeData) {

        for (int i = 0; i < sources.length; i++) {

            String id = sources[i].getId();

            System.out.println("removing source with id: " + id);
            
            /* remove source from memory */
            if (this.sources.get(id) != null) {
                this.sources.remove(id);
            }

            /* remove the configuration file */
            File confFile = new File(this.getConfFileName(id));

            if (confFile.exists()) {
                confFile.delete();
            }

            try {
                //Statement stmt = this.dbConnection.createStatement();
                Connection con = DbTools.createDbConnection(this.dbSettings,
                        this.dbSettings.getProperty(this.DBNAME_KEY));
                con.setAutoCommit(true);
                Statement stmt = con.createStatement();
                stmt.execute("delete from Statement where source='" + id
                                + "';");
                stmt.execute("delete from DataRecord where source='" + id
                        + "';");
                stmt.execute("delete from DataSource where id='" + id + "';");
                stmt.close();
                con.close();
            }
            catch (SQLException e) {
                this.addError("couldn't remove data source from the database: "
                        + e.getMessage());
                return false;
            }

            if (removeData) {
                File rawdataFile = new File(this.getRawdataFileName(id));

                if (rawdataFile.exists()) {
                    rawdataFile.delete();
                }
            }

            this.addLogMessage("removed data source: " + id);
        }

        return true;
    }

    
    public boolean workInProgress() {
        return this.updaterThreads.size() > 0;
    }

    
    public String[] getCurrentTasks() {

        String[] tasks = new String[this.updaterThreads.size()];

        for (int i = 0; i < tasks.length; i++) {
            tasks[i] = ((UpdateThread) this.updaterThreads.get(i))
                    .getTaskDescription();
        }
        return tasks;
    }

    
    public String[] getErrors() {
        String[] array = new String[this.errors.size()];
        for (int i = 0; i < array.length; i++) {
            array[i] = (String) this.errors.get(i);
        }
        return array;
    }

    
    public void setLogListener(LogListener listener) {
        this.logListener = listener;
    }

    /**
     * Converts an URL to a path which can be passed to a DataSource implementation.
     * @param location
     * @return
     */
    private String getPath(URL location) {
        String path = null;

        if (location.getProtocol().equalsIgnoreCase("file")) {
            path = location.getPath();

            /*this is a special hack that allows files to be entered in the form:
             *file://somefile.txt
             *These files are then searched from the current working directory*/
            if(path.equals("")) {
                path = location.getHost();
            }
                
        } else {
            path = location.toString();
        }

        return path;
    }

    private void addSourceToDatabase(String id, String format, String location)
            throws SQLException {

        String problem = this.validateSourceAttributes(id, format, location);
        if (problem != null) {
            throw new SQLException(problem);
        }

        Statement stmt = this.dbConnection.createStatement();
        String insert = "insert into DataSource values (" + "'" + id + "'," /* id */
                + "'" + format + "'," /* format */
                + "'" + location + "'," /* location */
                + "0," /* lineCount */
                + "NULL," /* updated */
                + "NULL" /* modified */
                + ");";
        stmt.execute(insert);
        stmt.close();
    }

    /**
     * This method validates data source attributes so that they can be fed to
     * the database.
     * 
     * @param id
     *            The id field of the data source.
     * @param format
     *            The format field of the data source.
     * @param location
     *            The location field of the data source.
     * @return null if no problems are found. If one or more of the attributes
     *         is unfit for the database, returns a String describing the
     *         problem.
     */
    private String validateSourceAttributes(String id, String format,
            String location) {

        String problem = null;

        try {
            Statement stmt = this.dbConnection.createStatement();

            /* check that the given attributes are not too long */

            ResultSet rs = stmt.executeQuery("select * from DataSource;");
            ResultSetMetaData metadata = rs.getMetaData();

            int maxLength = 0;

            for (int i = 1; i <= metadata.getColumnCount(); i++) {
                if (metadata.getColumnName(i).equals("id")) {
                    maxLength = metadata.getColumnDisplaySize(i);
                    if (id.length() > maxLength) {
                        problem = "data source's URL is too long";
                        break;
                    }
                }
                if (metadata.getColumnName(i).equals("location")) {
                    maxLength = metadata.getColumnDisplaySize(i);
                    if (location.length() > maxLength) {
                        problem = "data source's URL is too long";
                        break;
                    }
                }
            }
            
            stmt.close();
            rs.close();
            
        } catch (SQLException e) {
            problem = "unable to read database metadata";
            e.printStackTrace();
        }

        return problem;
    }

    
    private boolean sourceExistsInDatabase(String id) throws SQLException {
        Statement stmt = this.dbConnection.createStatement();
        ResultSet rs = stmt.executeQuery("select id from DataSource where id='"
                + id + "';");

        boolean exists = rs.next();

        stmt.close();
        rs.close();

        return exists;
    }

    /**
     * Creates a new data source configuration file for a new source. The file
     * is written to disk.
     * 
     * @param type
     * @param location
     * @param format
     * @return A File object presenting the file that was written on disk. null
     *         if the file couldn't be created.
     */
    private File createSourceConfFile(String type, String location,
            String format) {

        String dataSource = null;
        String recordParser = null;
        String recordConstructor = null;
        String transformer = null;

        /* resolve the classes used for this data source
         *  from the configuration file sourcetypes.xml */
        try {
            String ruleXPath = "/sourcetypes/rule[@sourceformat=\"" + format
                    + "\" and @transfertype=\"" + type + "\"]";

            dataSource = XMLTools.getNodeContent(this.builder,
                    new FileInputStream(this.sourceTypesConfFile), ruleXPath
                            + "/datasource");
            recordParser = XMLTools.getNodeContent(this.builder,
                    new FileInputStream(this.sourceTypesConfFile), ruleXPath
                            + "/recordparser");
            recordConstructor = XMLTools.getNodeContent(this.builder,
                    new FileInputStream(this.sourceTypesConfFile), ruleXPath
                            + "/recordconstructor");
            transformer = XMLTools.getNodeContent(this.builder,
                    new FileInputStream(this.sourceTypesConfFile), ruleXPath
                            + "/transformer");
        } catch (Exception e) {
            this
                    .addError("couldn't create data source configuration file for: "
                            + location);
            return null;
        }

        /* create the source conf file as a Document object */
        Document doc = this.builder.newDocument();

        Element root = doc.createElement("source");
        doc.appendChild(root);

        root.appendChild(doc.createElement("id")).appendChild(
                doc.createTextNode(location.toString()));
        root.appendChild(doc.createElement("location")).appendChild(
                doc.createElement("original")).appendChild(
                doc.createTextNode(location));
        root.appendChild(doc.createElement("datasource")).appendChild(
                doc.createElement("class")).appendChild(
                doc.createTextNode(dataSource));
        root.appendChild(doc.createElement("recordparser")).appendChild(
                doc.createElement("class")).appendChild(
                doc.createTextNode(recordParser));
        root.appendChild(doc.createElement("recordconstructor")).appendChild(
                doc.createElement("class")).appendChild(
                doc.createTextNode(recordConstructor));
        root.appendChild(doc.createElement("transformer")).appendChild(
                doc.createElement("class")).appendChild(
                doc.createTextNode(transformer));

        String filename = this.getConfFileName(location.toString());

        File file = XMLTools.writeDocumentToFile(doc, filename);

        return file;
    }

    
    /**
     * Return a name for a configuration file. The data source's id determines the filename. 
     * @param sourceID
     * @return
     * A String containg the filename. No path is included.
     */
    private String getConfFileName(String sourceID) {
        return this.dataDir
                + File.separator
                + sourceID.replaceAll("[^:]*://", "").replaceAll("/", "_")
                        .replaceAll("\\\\", "_").replaceAll(":", "_")
                + this.CONF_FILE_IDENTIFIER;
    }

    
    /**
     * Return a name for a raw data file. The data source's id determines the filename. 
     * @param sourceID
     * @return
     * A String containg the filename. No path is included.
     */
    private String getRawdataFileName(String sourceID) {
        return this.dataDir + File.separator
                + sourceID.replaceAll("[^:]*://", "").replaceAll("/", "_")
                + this.RAWDATA_FILE_IDENTIFIER;
    }

    /**
     * Reads all data source configuration files from a directory.
     * 
     * @param directory
     *            The directory that is searched for data source configuration files.
     * @return All files that are recognized as data source configuration files.
     */
    private File[] getConfFiles(File directory) {

        File[] files = directory.listFiles();

        Vector confFiles = new Vector();

        for (int i = 0; i < files.length; i++) {
            if (files[i].getName().endsWith(this.CONF_FILE_IDENTIFIER)) {
                confFiles.add(files[i]);
            }
        }

        File[] result = new File[confFiles.size()];
        result = (File[])confFiles.toArray(result);
        
        return result;
    }

    /**
     * Creates DataSource objects and all other objects needed for each data
     * source.
     * 
     * @param confFiles
     * The configuration files describing each Data Source.
     * @return A HashMap containing the DataSource objects. On failure null.
     */
    private HashMap createDataSources(File[] confFiles) {

        HashMap dataSources = new HashMap();

        for (int i = 0; i < confFiles.length; i++) {
            try {
                /* The source's ID */
                String sourceID = XMLTools.getNodeContent(this.builder,
                        new FileInputStream(confFiles[i]), this.SOURCEID_XPATH);

                /* check that the source exists in the database. */
                /*
                 * if(!this.sourceExistsInDatabase(sourceID)) { throw new
                 * RuntimeException("source " + sourceID + " has a configuration
                 * file but is missing from the database"); }
                 */

                /* class names for different implementations */
                String dataSourceClass = XMLTools.getNodeContent(this.builder,
                        new FileInputStream(confFiles[i]),
                        this.DATASOURCE_XPATH);
                String recordParserClass = XMLTools.getNodeContent(
                        this.builder, new FileInputStream(confFiles[i]),
                        this.RECORDPARSER_XPATH);
                String recordConstructorClass = XMLTools.getNodeContent(
                        this.builder, new FileInputStream(confFiles[i]),
                        this.RECORDCONSTRUCTOR_XPATH);
                String transformerClass = XMLTools.getNodeContent(this.builder,
                        new FileInputStream(confFiles[i]),
                        this.TRANSFORMER_XPATH);
                String dataSourceLocation = XMLTools.getNodeContent(
                        this.builder, new FileInputStream(confFiles[i]),
                        this.ORIGINALLOCATION_XPATH);

                /* these are used when giving parameters for constructors */
                Class[] parameterClasses = null;
                Object[] parameters = null;

                /* create a transformer */
                Transformer transformer = null;
                transformer = (Transformer) Class.forName(transformerClass)
                        .newInstance();

                /* create a DataStorage */
                File rawdataFile = new File(this.getRawdataFileName(sourceID));

                // Connection con = DbTools.createDbConnection(this.dbSettings,
                // this.dbSettings.getProperty(this.DBNAME_KEY));
                DataStorage dataStorage = new DataStorage(sourceID,
                        this.dbConnection, rawdataFile, transformer);

                /* create a RecordConstructor */
                RecordConstructor recordConstructor = null;
                recordConstructor = (RecordConstructor) Class.forName(
                        recordConstructorClass).newInstance();

                /* create a RecordParser */
                RecordParser recordParser = null;
                parameterClasses = new Class[2];
                parameterClasses[0] = Class
                        .forName("maito.datacollecting.RecordConstructor");
                parameterClasses[1] = Class
                        .forName("maito.datacollecting.DataStorage");

                parameters = new Object[2];
                parameters[0] = recordConstructor;
                parameters[1] = dataStorage;

                recordParser = (RecordParser) Class.forName(recordParserClass)
                        .getConstructor(parameterClasses).newInstance(
                                parameters);

                /* create a DataSource */
                parameterClasses = new Class[2];
                parameterClasses[0] = Class.forName("java.util.Properties");
                parameterClasses[1] = Class
                        .forName("maito.datacollecting.RecordParser");

                parameters = new Object[2];
                Properties dataSourceParams = new Properties();
                dataSourceParams.put(Tools.DATASOURCE_PARAM_LOCATION,
                        dataSourceLocation);
                String updated = this.getUpdated(sourceID);
                if (updated != null) {
                    dataSourceParams.put(Tools.DATASOURCE_PARAM_UPDATED,
                            updated);
                }
                parameters[0] = dataSourceParams;
                parameters[1] = recordParser;

                Object source = Class.forName(dataSourceClass).getConstructor(
                        parameterClasses).newInstance(parameters);

                dataSources.put(sourceID, source);
            } catch (Exception e) {
                this.addError("unable to init data sources: " + e.getMessage());
                return null;
            }
        }

        return dataSources;
    }

    /**
     * Retrieves the date of last successful update of a data source.
     * 
     * @param dataSourceID
     *            The requested data source
     * @return The date as a string in the format yyyy-MM-dd
     */
    private String getUpdated(String dataSourceID) throws SQLException {

        String updated = null;

        Statement stmt = this.dbConnection.createStatement();
        ResultSet rs = stmt
                .executeQuery("select updated from DataSource where id = '"
                        + dataSourceID + "';");

        if (rs.next()) {
            updated = rs.getString("updated");
        }

        stmt.close();
        
        if (rs != null) {
            rs.close();
        }

        return updated;
    }

    private synchronized void addError(String error) {
        this.errors.add(error);
        if (this.logListener != null) {
            this.logListener.logMessage(error);
        }
    }

    private synchronized void addLogMessage(String message) {
        if (this.logListener != null) {
            this.logListener.logMessage(message);
        }
    }

    /**
     * Creates the needed database and it's tables if they do not exist.
     * 
     * @throws SQLException
     */
    private void initDatabase() throws SQLException {

        this.dbConnection = DbTools.createDbConnection(this.dbSettings);

        String dbName = this.dbSettings.getProperty(this.DBNAME_KEY);

        Statement stm = this.dbConnection.createStatement();
        ResultSet rs = stm.executeQuery("SHOW DATABASES LIKE '" + dbName + "'");
        if (!rs.next()) { // database not there - create it

            this.addLogMessage("initializing database " + dbName);

            String sqlScript = Tools.readFile(Tools.PATH_RAWDATA_SQL);
            if (sqlScript == null) {
                throw new SQLException("Error loading sql script file: "
                        + Tools.PATH_RAWDATA_SQL);
            }
            stm.executeUpdate("CREATE DATABASE IF NOT EXISTS " + dbName
                    + " CHARACTER SET utf8");
            stm.executeUpdate("USE " + dbName);

            // execute each sql command from script
            int previous = 0;
            for (int i = sqlScript.indexOf(";", previous); i != -1; i = sqlScript
                    .indexOf(";", previous)) {

                String query = sqlScript.substring(previous, i);
                stm.execute(query);
                previous = i + 1;
            }
        } // else database already created
        if (rs != null)
            rs.close();
        if (stm != null)
            stm.close();

        this.dbConnection.close();
    }

    /**
     * This method is called by running UpdateThread instances when they finish
     * updating.
     * 
     * @param datasourceId
     * @param updater
     *            The UpdateThread object that handled the update.
     * @param withoutErrors
     */
    private synchronized void updateFinished(String dataSourceID,
            UpdateThread updater, boolean finishedWithoutError) {

        if (finishedWithoutError) {
            // get the timestamp for current date
            String timestamp = new SimpleDateFormat(this.DATE_FORMAT)
                    .format(new java.util.Date());
            String update = "update DataSource set updated = '" + timestamp
                    + "' where id='" + dataSourceID + "';";
            
            try {
                
                Connection con = DbTools.createDbConnection(this.dbSettings,
                        this.dbSettings.getProperty(this.DBNAME_KEY));
                
                Statement stmt = con.createStatement();
                //Statement stmt = this.dbConnection.createStatement();
                stmt.execute(update);
                stmt.close();
                con.close();
            } catch (SQLException e) {
                this.addError("couldn't set update date for data source: "
                        + dataSourceID);
                e.printStackTrace();
            }
        }
        this.addLogMessage("update finished for data source: " + dataSourceID);
        
        this.updaterThreads.remove(updater);
    }

    /**
     * A thread for updating a data source.
     * 
     * @author Antti Laitinen
     * 
     */
    class UpdateThread extends Thread {

        private DataCollectorImpl collector;

        private DataSource source;

        private String sourceId;

        public UpdateThread(DataCollectorImpl collector, DataSource source,
                String sourceId) {
            this.collector = collector;
            this.source = source;
            this.sourceId = sourceId;
        }

        public void run() {
            boolean finishedWithoutError = true;

            try {
                this.source.update();
            } 
            catch (DataSourceException e) {
                this.collector.addError("error while updating source "
                        + sourceId + ": " + e.getMessage());
                finishedWithoutError = false;
            } 
            finally {
                this.collector.updateFinished(this.sourceId, this,
                        finishedWithoutError);
            }
        }

        public String getTaskDescription() {
            return "updating data source " + this.sourceId;
        }
    }
}
