package maito.datacollecting;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.BufferedWriter;
import java.io.IOException;

import java.util.Vector;
import java.util.Iterator;

import java.math.BigInteger;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.ResultSet;
import java.sql.SQLException;

import maito.util.DbTools;


/**
 * A class that takes care of storing the data it receives from a RecordParser. 
 * The original raw data is stored into a file. The data is also transformed using a
 * Transformer implementation and then the transformed record is stored in the database. 
 * @author Oskari Saarekas 
 * 
 * @see maito.datacollecting.Transformer
 * @see maito.datacollecting.RecordParser
 * 
 */
public class DataStorage {

    private String sourceID;
    private Connection conn;
    private File file;
    private Transformer transformer;
    
    /**
     * @param sourceID
     * ID of the datasource. The ID must exist in the DataSource table, or the
     * class will not work properly.
     * @param conn
     * Connection object to the database. The class will not work if the
     * object is invalid.
     * @param file
     * File object that indicates the file where the plaintext records are
     * stored. The class will not work properly if invalid object is passed.
     */
    public DataStorage(String sourceID, Connection conn, File file, Transformer transformer) {
        this.sourceID = sourceID;
        this.conn = conn;
        this.file = file;
        this.transformer = transformer;
    }
    
    /*
     * Checks if an identical record already exists in the DB.
     * Does this by retrieving a record with the same ID and
     * datasource ID, then uses the Record's equals -method
     * to check the similarity.
     */
    private boolean identicalRecordExists(Record record) {
        Statement stmt = null;
        ResultSet rs = null;
        
        try {
            this.conn.setAutoCommit(true);
            stmt = conn.createStatement();
            
            String sqlRecID = DbTools.SQLstr(record.getID(), Record.IDLENGTH);
            String sqlSrcID = DbTools.SQLstr(this.sourceID, DataSourceDescription.IDLENGTH);
            
            rs = stmt.executeQuery("SELECT id, source FROM DataRecord WHERE " +
                                   "id=" + sqlRecID + " AND source=" + sqlSrcID);
            if (!rs.next()) {
                return false;
            }
            
            rs = stmt.executeQuery("SELECT name, value FROM Statement WHERE " +
                                   "record=" + sqlRecID + " AND source=" + sqlSrcID);

            Record fromDB = new Record(record.getID());
            while (rs.next()) {
                fromDB.setField(rs.getString("name"), rs.getString("value"));
            }
            
            if (!record.equals(fromDB)) {
                return false;
            }
        }
        catch (SQLException e) {
            return false;
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch (SQLException e) {}
            }
            if (stmt != null) {
                try {
                    stmt.close();
                } catch (SQLException e) {}
            }
        }
        
        return true;
    }
    
    /**
     * Adds a new record to the file and to the database.
     *
     * @param record The record to be added as a Record object.
     * @param original The record in its original String form.
     *
     * @return
     * True if the record is successfully added to the database
     * and the file seems to be writable, or an identical record
     * already exists in the database.
     */
    public boolean addRecord(Record record, String original) {
        if (this.sourceID == null || this.conn == null ||
            this.file == null || this.transformer == null) {
            return false;
        }
        
        //transforms record into standard format
        try {
        	record = transformer.transform(record);
        }
        catch(TransformException te) {
        	//write log here
        	return false;
        }
        
        if (this.identicalRecordExists(record)) {
            return true;
        }

        /*
         * Check that record can be written to the file
         */
        BufferedWriter fileBuffer = null;
        try {
            fileBuffer = new BufferedWriter(
                         new OutputStreamWriter(
                         new FileOutputStream(this.file, true), "UTF-8"));
        }
        catch (IOException e) {
            if (fileBuffer != null) {
                try {
                    fileBuffer.close();
                } catch (IOException e2) {}
            }
            return false;
        }
        
        /*
         * Count how many lines the plaintext record spans
         */
        int recordLines = 1;
        int i = -1;
        while ((i = original.indexOf('\n', i + 1)) != -1) {
            ++recordLines;
        }
        
        Statement stmt = null;
        ResultSet rs = null;
        
        try {
            this.conn.setAutoCommit(true);
            stmt = conn.createStatement();
            
            String sqlRecID = DbTools.SQLstr(record.getID(), Record.IDLENGTH);
            String sqlSrcID = DbTools.SQLstr(this.sourceID, DataSourceDescription.IDLENGTH);
            
            /*
             * Remove possible outdated copy of the record
             */
            stmt.execute("DELETE FROM DataRecord WHERE id=" + sqlRecID);
            
            /*
             * Retrieve the number of lines in the plaintext record file
             */
            rs = stmt.executeQuery("SELECT linecount FROM DataSource " +
                                   "WHERE id=" + sqlSrcID);
            
            BigInteger dsLineCount = null;
            if (rs.next()) {
                if (rs.getString("linecount") == null) {
                    dsLineCount = new BigInteger("0");
                } else {
                    dsLineCount = new BigInteger(rs.getString("linecount"));
                }
            }
            if (dsLineCount == null) {
                dsLineCount = new BigInteger("0");
            }
            
            /*
             * Calculate number of the record's first line in the plaintext file
             */
            BigInteger firstLine = new BigInteger(dsLineCount.toString());
            firstLine = firstLine.add(BigInteger.ONE);
            /*
             * Calculate the number of lines in the plaintext file
             * It's also the number of the record's last line
             */
            dsLineCount = dsLineCount.add(new BigInteger("" + recordLines));
            
            this.conn.setAutoCommit(false);
            
            stmt = this.conn.createStatement();
            stmt.execute("INSERT INTO DataRecord SET " +
                         "id=" + sqlRecID + "," +
                         "source=" + sqlSrcID + "," +
                         "firstLineNumber=" + firstLine + "," +
                         "lastLineNumber=" + dsLineCount);
            
            Iterator fields = record.getFieldNames().iterator();
            while (fields.hasNext()) {
                String fieldName = (String)fields.next();
                Vector values = record.getField(fieldName);
                for (i = 0; i < values.size(); ++i) {
                    String fieldValue = (String)values.elementAt(i);
                    stmt.execute("INSERT INTO Statement SET " +
                        "name=" + DbTools.SQLstr(fieldName, Record.STMTKEYLEN) + "," +
                        "value=" + DbTools.SQLstr(fieldValue, Record.STMTVALLEN) + "," +
                        "record=" + sqlRecID + ",source=" + sqlSrcID);
                }
            }
            
            stmt.execute("UPDATE DataSource SET modified = curdate()," +
                         "linecount=" + dsLineCount + " WHERE id=" + sqlSrcID);

            this.conn.commit();
        }
        catch (SQLException e) {
            try {
                if (!this.conn.getAutoCommit()) {
                    this.conn.rollback();
                }
            }
            catch (SQLException e2) {}
            return false;
        }
        finally {
            if (rs != null) {
                try {
                    rs.close();
                } catch(Exception e) {}
            }
            if (stmt != null) {
                try {
                    stmt.close();
                } catch(Exception e) {}
            }
        }

        /*
         * Write record to the plaintext file
         */
        try {
            fileBuffer.write(original + "\n");
        }
        catch (IOException e) {
        }
        finally {
            if (fileBuffer != null) {
                try {
                    fileBuffer.close();
                } catch (IOException e) {}
            }
        }

        return true;
    }
}
