/*
* Record.java
*
* This software is released under the GNU GPL license
*/

package maito.datacollecting;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Vector;
import java.util.Iterator;
import java.util.NoSuchElementException;

public class Record {

    public static final int IDLENGTH = 200;
    public static final int STMTKEYLEN = 255;
    public static final int STMTVALLEN = 255;

    private HashMap fields;
    private String recordID;
    private String integratedTo;

    /**
    * Class constructor
    *
    * @param newID 
    * The ID to be set for the new record, must not be null
    * (In case of null value, ID is assumed to be an empty String)
    */
    public Record(String newID){
        if (newID != null) {
            recordID = newID;
        }
        else recordID = "";
        fields = new HashMap();
        integratedTo = null;
    }
    
    /** 
    * Returns a list of keys in this record
    *
    * @return a HashSet of all keys in this Record
    */
    public HashSet getFieldNames() {
        return new HashSet(fields.keySet());
    }
    
    /** 
    * Gets the value(s) of the field with the specified key.
    * Record keys are Strings and values are Vectors containing Strings
    *
    * @param fieldName The key of the required field
    * @return The required value, null if field not found
    */
    public Vector getField (String fieldName) {
        return (Vector) fields.get(fieldName);
    }
    
    /** 
    * Gets the ID of this record
    *
    * @return The ID of this record
    */
    public String getID() {
        return recordID;
    }
    
    /** 
    * Adds a value to a field. If specified key already exists,
    * adds a new String value to the existing Vector, otherwise creates a new one
    *
    * @param fieldKey A key to be inserted or modified, must not be null
    * @param fieldValue A value to be inserted
    * @return true if value stored, false if there was an error
    */ 
    public boolean setField (String fieldKey, String fieldValue) {
        if ((fieldKey) == null) {
            return false;
        }
        else if (fields.containsKey(fieldKey)) { // Vector already exists
            Vector tempVector = (Vector)fields.get(fieldKey);
            tempVector.addElement(fieldValue);
            fields.put(fieldKey, tempVector);
            return true;
        }
        else { // Create a new Vector
            Vector newVector = new Vector(5); // Most keys should not have even this many entries - keyword list being the exception
            newVector.addElement(fieldValue);
            fields.put(fieldKey, newVector);
            return true;
        }
    }
    
    /**
     * Deletes the field and all its values. Use with caution.
     * 
     * @param fieldKey the field to be deleted
     */
    public void deleteField(String fieldKey) {
        fields.remove(fieldKey);
    }
    
    /**
    * Two records are considered the same only if they have the same id. 
    *
    * @return true if two records are equal, false if not
    */
/*    public boolean equals (Record record) {
        if (record == null) {
            return false;
        }
        return this.getID().equals(record.getID());
    }*/
    
    public boolean equals (Record record) {
        if (record == null) {
            return false;
        }
        if (!this.getID().equals(record.getID())) {
            return false;
        }
        if (this.getFieldNames().size() != record.getFieldNames().size()) {
            return false;
        }
            
        Iterator fields = this.getFieldNames().iterator();
        while (fields.hasNext()) {
            String fieldName = (String)fields.next();
            Vector v1 = this.getField(fieldName);
            Vector v2 = record.getField(fieldName);
            
            if (v2 == null) {
                return false;
            }
            if (v1.size() != v2.size()) {
                return false;
            }
            
            /*
             * Creates a copy of the compared Vector.
             * Removes each element found in the comparison
             * Vector from the copy Vector, and checks if
             * it's empty (Record contents are the same).
             */
            
            Iterator i1 = v1.iterator();
            Vector cmp = (Vector)v2.clone();
            try {
                while (i1.hasNext()) {
                    String value = (String)i1.next();
                    Iterator i2 = cmp.iterator();
                    while (i2.hasNext()) {
                        String cmpValue = (String)i2.next();
                        if (cmpValue == null) {
                            if (value == null)
                            {
                                i2.remove();
                                break;
                            }
                        }
                        else if (cmpValue.equals(value)) {
                            i2.remove();
                            break;
                        }
                    }
                }
            }
            catch (NoSuchElementException e) {
                return false; //Not covered
            }
            catch (IllegalStateException e) {
                return false; //Not covered
            }
            
            if (cmp.size() > 0) {
                return false;
            }
        }
        
        return true;
    }
    
    public void setIntegratedTo(String s) {
        this.integratedTo = s;
    }
    
    /**
     * @return empty string if <code>integratedTo</code> is null
     */
    public String getIntegratedTo() {
        return this.integratedTo == null ? "" : this.integratedTo;
    }
}

