package maito.datacollecting.file;

import java.io.*;
import java.net.URI;
import java.util.Properties;

import maito.datacollecting.DataSource;
import maito.datacollecting.DataSourceException;
import maito.datacollecting.RecordParser;
import maito.util.Tools;

/**
* Reads a file in lines and passes them to a
* RecordParser for parsing. The buffering is done
* in the parser.
* 
* @author Väinö Ala-Härkönen
* 
* This software is released under the GNU GPL license
*/
public class FileDataSource implements DataSource {
    
    private BufferedReader fread;
    private RecordParser recordParser;
    private String baseURL; 
    // private String updated;
    private char[] cbuf;
    private String thisLine;
    
    private final int BLOCK_SIZE = 1024;

    public FileDataSource (Properties parameters, RecordParser recordParser) throws DataSourceException {
        
        if (parameters == null) {
            throw new DataSourceException("maito.datacollecting.file.FileDataSource: null parameter set");
        }
        
        this.baseURL = parameters.getProperty(Tools.DATASOURCE_PARAM_LOCATION); 
        if (this.baseURL == null) {
            throw new DataSourceException("maito.datacollecting.file.FileDataSource: parameter " + Tools.DATASOURCE_PARAM_LOCATION + " missing");
        }
        
        // this.updated = parameters.getProperty("updated");
        
        if (recordParser != null)
            this.recordParser = recordParser;
        else
            throw new DataSourceException("maito.datacollecting.file.FileDataSource: recordParser not defined");
    }
        
    /**
     * Reads the source file in blocks and passes them on to the specified recordParser
     * When the end of the file is reached, sends a null block.
     * 
     */
    public void update() throws DataSourceException {
        // this.updateInLines();
        this.updateInBlocks();
    }

    // Does the file reading line by line
    private void updateInLines() throws DataSourceException {
        try {
            URI baseURI = new URI (baseURL);
            File thisFile = new File(baseURI);
            fread = new BufferedReader(new InputStreamReader(new FileInputStream(thisFile), "UTF-8"));
        } 
        catch (Exception e) {
            throw new DataSourceException("maito.datacollecting.file.FileDataSource: error opening file: " + baseURL);
        }
        
        try {
            while ((thisLine = fread.readLine()) != null) {
                recordParser.putData(thisLine);
                // System.out.println(thisLine);
            }
            recordParser.putData(null); // the file has ended
        } 
        catch (IOException e) { // never reached in practise?
            throw new DataSourceException("maito.datacollecting.file.FileDataSource: error reading file " + baseURL);
        }        
    }

    // Does the file reading in blocks    
    private void updateInBlocks() throws DataSourceException {
        cbuf = new char[BLOCK_SIZE];
        
        try {
            if (baseURL.startsWith("file://"))
                baseURL = baseURL.substring(7,baseURL.length());
            //URI baseURI = new URI (baseURL);
            //System.out.println(baseURI);
            File thisFile = new File(baseURL);
            fread = new BufferedReader(new InputStreamReader(new FileInputStream(thisFile), "UTF-8"));
            //fread = new BufferedReader(new InputStreamReader(new FileInputStream(baseURL), "UTF-8"));
        } 
        catch (Exception e) {            
            throw new DataSourceException("maito.datacollecting.file.FileDataSource: error opening file: " + baseURL + ", " + e.getMessage());            
        }
        
        try {
            while ((fread.read(cbuf, 0, BLOCK_SIZE)) != -1) {
                recordParser.putData(new String(cbuf));
                //System.out.println(cbuf);
            }
        } 
        catch (IOException e) { // never reached in practise?
            throw new DataSourceException("maito.datacollecting.file.FileDataSource: error reading file " + baseURL);
        }
        
    }

}
