package maito.datacollecting.oaipmh;

import maito.datacollecting.*;
import maito.util.Tools;
import java.util.Properties;

/**
 * A class that represents a data source using OAI-PMH to read data from a network repository. 
 * @author Antti Laitinen
 *
 */
public class OAIPMHDataSource implements DataSource {

	private RecordParser recordParser;
	
	private String baseURL;
	private String metadataPrefix = "oai_dc";
	private String updated;
	
	private final int PART_SIZE = 100;
	
	
	/**
	 * 
	 * @param parameters
	 * Must contain the following parameters:
	 * <ul>
	 * <li>location (base URL to the remote OAI-PMH service)</li>
	 * </ul>
	 * The following parameters are optional:
	 * <ul>
	 * <li>updated (The date when this source was last updated. If this parameter is not present, then the )</li>
	 * </ul>
	 */
	public OAIPMHDataSource(Properties parameters, RecordParser recordParser) throws DataSourceException {
	
		if(parameters == null) {
            throw new DataSourceException("parameters missing");
        }
        
		this.baseURL = parameters.getProperty(Tools.DATASOURCE_PARAM_LOCATION); 
		if (this.baseURL == null) {
			throw new DataSourceException("parameter baseURL missing");
		}
		
		
		this.updated = parameters.getProperty(Tools.DATASOURCE_PARAM_UPDATED);
		
		this.recordParser = recordParser;
		
		if(this.recordParser == null) {
			throw new DataSourceException("parameter recordParser is null");
		}
	}
	
	
	/**
	 * Reads data from the OAI-PMH repository. 
	 * If the data is successfully received it is then sent to the RecordParser.
	 */
	public void update() throws DataSourceException {
		ListRecords data = null;
		try
		{
			data = new ListRecords(this.baseURL, this.metadataPrefix, this.updated);
		}
		catch(OAIPMHException e) {
			throw new DataSourceException(e.getMessage());
		}
		
        try {
                
    		String part = data.nextPart(PART_SIZE); 
    		
    		while(part != null)
    		{
    			this.recordParser.putData(part);
    			part = data.nextPart(PART_SIZE);
    		}
        }
        catch(OAIPMHException e) {
            throw new DataSourceException(e.getMessage());
        }
	}
}
