package maito.datacollecting.oaipmh;

import maito.datacollecting.*;

/**
* Parses records from an incoming OAI-PMH stream
* NOTE: this actually works with any other character stream as
* long as the content is in a format similar to OAI-PMH / DC XML
* 
* @version 1.0
* @author Antti Laitinen
* @author Väinö Ala-Härkönen
* 
* 15.11.2005
* 
* This software is released under the GNU GPL license
*/
public class OAIPMHRecordParser implements RecordParser {
	
	private RecordConstructor recordConstructor;
	
	private DataStorage dataStorage;
	
	private StringBuffer buffer;
	
	public OAIPMHRecordParser(RecordConstructor recordConstructor, DataStorage dataStorage) {
		
		this.recordConstructor = recordConstructor; 
		this.dataStorage = dataStorage;
		this.buffer = new StringBuffer();
	}
	
	/**
     * Adds the part to the internal buffer and checks out if there's a full record in
     * the buffer already. If a full record is found, constructs a Record with
     * the specified RecordConstructor and passes it on to the <code>DataStorage</code>.
     * If several full Records are found in buffer, does the same to them all.
     * 
     * @param part 
     * The next part of the record data
	 */
	public void putData(String part) {
	
		if (part != null)
            this.buffer.append(part);
		
		String recordString;
        
//       process until the buffer has no more full records...
		while((recordString = extractRecord()) != null) 
		{
             // System.out.println(recordString);
			try {
				Record record = this.recordConstructor.constructRecord(recordString);
                 if (record != null)
                     this.dataStorage.addRecord(record, recordString);
			}
			catch(Exception e) {
                 System.out.println(e);
                 e.printStackTrace();
			    throw new IllegalArgumentException();
			}
		}
	}

	/**
	 * Extracts the first full record from the string buffer.
     * 
	 * @return
	 * Returns the extracted record as a String. If a complete record is not found returns null.
	 */
	private String extractRecord() {
        String result = "";
        
        String startTag = "<record>";
        String endTag = "</record>";
        
        int startIndex = this.buffer.indexOf(startTag); 
        int endIndex = this.buffer.indexOf(endTag);
        
        if (startIndex >= 0 && endIndex >= 0) {
            result = this.buffer.substring(startIndex, endIndex + endTag.length());
            this.buffer.delete(startIndex, endIndex + endTag.length());
            return new String(result);
        }
        else 
            return null;
	}
}
