package maito.datacollecting.quickformat;

import maito.datacollecting.DataStorage;
import maito.datacollecting.Record;
import maito.datacollecting.RecordConstructor;
import maito.datacollecting.RecordParser;

/**
* Parses records from an incoming quickformat stream
* 
* @version 1.0
* @author Väinö Ala-Härkönen
*/
public class QuickformatRecordParser implements RecordParser {
    
    /*
     * These are the format identifiers we're accepting
     * These are used for prefix matching so when adding new ones,
     * make sure they're ordered accordingly
     */
    private static final String[] VALIDFORMATIDS = 
    {"1",  "2.1", "2.2", "2", "3.1", "3", "4.1", "4.2", "4.3", "4.4", "4.5", "4"};

    private RecordConstructor recordConstructor;
    
    private DataStorage dataStorage;
    
    private StringBuffer buffer, recordBuffer;
        
    private String format = null;
    private String sourceName = null;
    
    private boolean EOF = false;
    
    public QuickformatRecordParser(RecordConstructor recordConstructor, DataStorage dataStorage) {
        
        this.recordConstructor = recordConstructor; 
        this.dataStorage = dataStorage;
        this.buffer = new StringBuffer();
        this.recordBuffer = new StringBuffer();
    }
    
    /**
     * Adds the part to the internal buffer and checks out if there's a full record in
     * the buffer already. If a full record is found, constructs a Record with
     * the specified RecordConstructor and passes it on to the <code>DataStorage</code>.
     * If several full Records are found in buffer, does the same to them all.
     * NOTE: Your data source must send a null string as end of data signal so the
     * last record will get processed
     * 
     * @param part 
     * The next part of the record data or null if end-of-data
     */
    public void putData(String part) {
        
        if (part != null)
            this.buffer.append(part);
        else {
            EOF = true;
            this.buffer.append("\n");
        }

        this.parseRecord();
    }

    
    /*
     * Splits the input into lines and calls parseLine to process them
     */
    private void parseRecord() {
        int lineEnd;
        while (((lineEnd = buffer.indexOf("\n")) != -1))
            parseLine(lineEnd);
    }
    
    /*
     * Parses the lines in quickformat data into something meaningful.
     * Constructs and adds full records.
     */
    private void parseLine(int lineEnd) {
        
        // extract the line...
        String thisLine = buffer.substring(0, lineEnd);
        buffer.delete(0, lineEnd + 1);
        
        thisLine = thisLine.trim();
        //System.out.println("new line: " + thisLine);
        
        // ignore comments and empty lines
        if (thisLine.startsWith("#") || thisLine.length() == 0) {
            /*
             * If end-of-data encountered and there's an unfinished
             * document record in the buffer, construct and add it
             */
            if (EOF && (buffer.length() == 0)) {
                if (format.equals("1") && (recordBuffer.length() != 0)) { 
                    String thisRecord = new String(recordBuffer);
                    //System.out.println("START RECORD" + thisRecord + "END RECORD");
                    Record record = this.recordConstructor.constructRecord(thisRecord);
                    if (record != null)
                        this.dataStorage.addRecord(record, thisRecord);
                }
            }
        }
        
        // first line should be the format
        else if (format == null) {
            for (int i = 0; i < VALIDFORMATIDS.length; i++) {
                if (thisLine.equals(VALIDFORMATIDS[i])) {
                    // System.out.println("format: " + thisLine);
                    format = thisLine;
                }
            }
            if (format == null) // no valid format found
                throw new IllegalArgumentException("No valid format tag found in quickformat data");
        }
        
        // second line should be the source name
        else if (sourceName == null) {
            sourceName = thisLine;
        }
        
        /* 
         * the next lines should be the data...
         */
        
        // document format, multiple lines separated
        else if (format.equals("1")) {
            /*
             * record separator found or end of file reached and the whole input 
             * buffer is already processed
             */
            if (thisLine.equals("---") || (EOF && (buffer.length() == 0))) {
                if (EOF) { // add the very last line, it belongs to this record
                    recordBuffer.append(thisLine + "\n");
                }
                // if there's already a full record in the buffer (this isn't the first one), process it
                if (recordBuffer.length() != 0) {
                    String thisRecord = new String(recordBuffer);
                    //System.out.println("START RECORD" + thisRecord + "END RECORD");
                    Record record = this.recordConstructor.constructRecord(thisRecord);
                    if (record != null)
                        this.dataStorage.addRecord(record, thisRecord);
                    recordBuffer.delete(0, recordBuffer.length());
                }
            }
            
            /* 
             * add the new line to the record buffer with a newline
             */
            recordBuffer.append(thisLine + "\n");
      }
        
        // name format, one line per one record
        else {
            // System.out.println("NAME RECORD: " + thisLine);
            // NOTE: the format is appended to the start of the record data so the constructor knows which tag to use
            Record record = this.recordConstructor.constructRecord(format + thisLine);
            if (record != null)
                this.dataStorage.addRecord(record, thisLine);
        }
        
    }
    


}
