package maito.integration;

import maito.DataProcessor;
import maito.LogListener;
import maito.datacollecting.DataSourceDescription;

/**
 * An inteface for using the integration features of this program. 
 * This interface is designed to be used by the UI.
 * 
 * @author Antti Laitinen
 *
 */
public interface Integrator extends DataProcessor {

    /**
     * Adds a new graph to this Integrator. Actual integration is not 
     * done until the <code>integrate</code>-method is called.
     *
     * @param graphName 
     * The name of the new resource graph
     * @throws IllegalArgumentException
     * Thrown when the graph with the given name already exists.
     */
    public void addGraph(String graphName) throws IllegalArgumentException;
    
    
    /**
     * Returns an array of all resource graphs known to this Integrator. 
     *
     * @return 
     * A List of resource graph name Strings. 
     * If no graphs exist an empty List is returned.
     */
     public java.util.List getGraphs();
     
     
     /**
      * Returns the descriptions of the sources that are integrated into a graph.
      * @param graph
      * The graph that's source are to be returned.
      * @return
      * A list of String arrays where the arrary[0] displays the source name.
      * The array[1] displays integration date. 
      * If no sources are found returns an empty List.
      * @throws IllegalArgumentException
      * Thrown if something is wrong with the <code>graph</code> parameter.
      */
     public java.util.List getSources(String graph) throws IllegalArgumentException;
    
     
    /**
     * Starts the creation of a new resource graph in the integration subsystem.
     * 
     * @param sources
     * The data sources that are integrated.
     * @param graph
     * The name of the graph that is to be integrated.
     * @throws IntegrationException
     * Thrown if something goes wrong during the integration process.
     */
    public void integrate(DataSourceDescription[] sources, String graph) throws IntegrationException;

    
    /**
    * Sets a listener for log messages sent by this Integrator.
    *
    * @param graph
    * The resource graph that's log messages are to be listened to. If <code>null</code> all messages are listened to. 
    * @param listener
    * The object that listens to this Integrator's log messages.
    */
    public String setLogListener(String graph, LogListener listener);
}
