package maito.integration;

import maito.datacollecting.Record;
import maito.resource.Resource;
import maito.util.Tools;
import maito.util.DbTools;

import java.util.Date;
import java.util.Iterator;
import java.util.Properties;
import java.util.Vector;
import java.sql.*;

/**
 * @author Tuomas Tanner
 * @author Oskari Saarekas
 * 
 * This software is released under the GNU GPL license
 */
public class IntegratorHelper {

    private Properties docFieldmap;
    private Properties actorFieldmap;
    private Properties quickNameFieldmap;
    
    public IntegratorHelper(Properties docs, Properties actors,
                            Properties quickName) {

        this.docFieldmap = docs;
        this.actorFieldmap = actors;
        this.quickNameFieldmap = quickName;
    }

    /**
     * @param sourceID ID of the source of the parameter record
     * @param rec the Record to be integrated
     * @con DB Connection to the tables of this resource graph
     */
    public void integrateDocument(Connection con, String sourceID,
                                  Record rec) throws IntegrationException {
        String curDocID = null;
        Resource curDocument = null;
        
        /*
         * Get an existing similar document resource,
         * or have a new one created
         */
        String canoName = getFirstVal(rec.getField("KanoNimeke"));
        if (canoName == null || canoName.equals("")) {
            canoName = Tools.canonizeGeneric(getFirstVal(rec.getField("Nimeke")));
            if (canoName == null) {
                canoName = "";
            }
            rec.setField("KanoNimeke", canoName);
        }
        curDocument = this.getResource(con, "Document", null,
                                       createDocID(rec), canoName);
        curDocID = curDocument.getID();
        
        curDocument.setOverwrite(false);
        curDocument.setField("Records." + sourceID + "¤" + rec.getID(),
                             DbTools.formatDate(new Date()));
        
        /*
         * Add all document-related fields from the record
         * to this document resource, and finally save it
         */
        Iterator iter = rec.getFieldNames().iterator();
        while(iter.hasNext()) {
            String fieldName = (String)iter.next();
            String mappedField = docFieldmap.getProperty(fieldName);
            if (mappedField == null) { //Record field not used in document
                continue;
            }
            
            Vector val = rec.getField(fieldName);
            Iterator vIter = val.iterator();
            while (vIter.hasNext()) {
                String tmpvalue = (String)vIter.next();
                debug("Set field " + fieldName + " > " + mappedField + 
                        " - value: " + tmpvalue + ", status: " +
                        curDocument.setField(mappedField, tmpvalue));
            }
            
            /*
             * Create additional keywords from canonized name
             */
            String[] kws = canoName.split(" ");
            if (kws != null) {
                for(int i = 0; i < kws.length; ++i) {
                    curDocument.setField("KeyWords.keyword", kws[i]);
                }
            }
        }

        /*
         * Insert dummy fields to make sure that DB specs are met
         * (not necessary anymore) (note that overwrite = false)
         */
        curDocument.setField("Resource.name", "");
        curDocument.setField("Resource.canonicalName", "");
        curDocument.save(con);

        
        /*
         * Create Actors and Channels from record
         */
        iter = rec.getFieldNames().iterator();
        while(iter.hasNext()) {
            String fieldName = (String)iter.next();
            String mappedField = actorFieldmap.getProperty(fieldName);
            if (mappedField == null) { //not an actor or channel field
                continue;
            }
            
            //0 = Actor or Channel, 1 = type, 2 = role type 
            String mappings[] = mappedField.split(",");
            
            Vector val = rec.getField(fieldName);
            Iterator vIter = val.iterator();
            while (vIter.hasNext()) { //process all values 
                String tmpID = null;
                String tmpValue = (String)vIter.next();
                String canonVal = Tools.canonizeGeneric(tmpValue);
                Resource tmpResource = null;

                tmpResource = this.getResource(con, mappings[0], mappings[1],
                                               tmpValue, canonVal);
                tmpID = tmpResource.getID();
                
                //set info from record and save the resource to db
                tmpResource.setField("Resource.name", tmpValue);
                tmpResource.setField("Resource.canonicalName", canonVal);
                tmpResource.setField(mappings[0] + ".type", mappings[1]);
                tmpResource.save(con);
                
                createRelation(con, tmpID, curDocID, mappings[2], false);
            }
        }
    }
    
    /**
     * @param sourceID ID of the source of the parameter record
     * @param rec the Record representing a single record in the
     * quickformat sourcefile
     * @con DB Connection to the tables of this resource graph
     */
    public void integrateNameQuickFormat(Connection con, String sourceID,
                                         Record rec) throws
                                         IntegrationException {
        /*
         * Extract possible Hallinnoija related elements from
         * the Record - we'll deal with them later
         */
        String hallinnoija = null;
        Vector values = rec.getField("OrganisaatioHallinnoija");
        if (values != null) {
            hallinnoija = getFirstVal(values);
            rec.deleteField("OrganisaatioHallinnoija");
        }
        
        /*
         * Determine the type, name and canonical name of
         * this resource, based on the source Record's properties
         * and load an existing resource, or create a new one
         */
        Resource resource = null;
        String resType = null;
        String subType = null;
        String resName = null;
        String canoName = null;
        Iterator iter = rec.getFieldNames().iterator();
        
        while (iter.hasNext()) {
            String fieldName = (String)iter.next();
            String mappedField = this.quickNameFieldmap.getProperty(fieldName);
            if (mappedField == null) {
                continue;
            }
            
            if (mappedField.startsWith("Resource.name,")) {
                values = rec.getField(fieldName);
                if (values != null) {
                    //0 = Resource field, 1 = Table name, 2 = subtype 
                    String mappings[] = mappedField.split(",");

                    resType = mappings[1];
                    subType = mappings[2];
                    resName = getFirstVal(values);
                    canoName = Tools.canonizeGeneric(resName);
                    resource = this.getResource(con, resType, subType,
                                                resName, canoName);
                    break;
                }
            }
        }
        
        if (resource == null) {
            return;
        }

        /*
         * Add some new properties for the resource,
         * overwriting any existing old ones, and finally
         * save the Resource
         */
        resource.setOverwrite(true);
        resource.setField("Resource.name", resName);
        resource.setField(resType + ".type", subType);
        resource.setField("Records." + sourceID + "¤" + rec.getID(),
                          DbTools.formatDate(new Date()));

        iter = rec.getFieldNames().iterator();
        while (iter.hasNext()) {
            String fieldName = (String)iter.next();
            String mappedField = this.quickNameFieldmap.getProperty(fieldName);
            if (mappedField == null) {
                continue;
            }
            
            int comma = mappedField.indexOf(",");
            if (comma > 0) {
                mappedField = mappedField.substring(0, comma);
            }
            
            values = rec.getField(fieldName);
            Iterator vIter = values.iterator();
            while (vIter.hasNext()) {
                String tmpvalue = (String)vIter.next();
                debug("Set field " + fieldName + " > " + mappedField + 
                      " - value: " + tmpvalue + ", status: " +
                      resource.setField(mappedField, tmpvalue));
            }
        }
        
        resource.save(con);
        
        /*
         * Add the Hallinnoija along with the proper relation
         */
        if (hallinnoija != null) {
            debug("Creating Hallinnoija " + hallinnoija);
            canoName = Tools.canonizeGeneric(hallinnoija);
            Resource halRes = this.getResource(con, "Actor", "Organisaatio",
                                               hallinnoija, canoName);
            /*
             * Update record reference and store back to DB
             */
            halRes.setField("Resource.name", hallinnoija);
            halRes.setField("Actor.type", "Organisaatio");
            halRes.setField("Records." + sourceID + "¤" + rec.getID(),
                            DbTools.formatDate(new Date()));
            halRes.save(con);
            
            /*
             * Create the relation between the Hallinnoija and
             * the regular resource which was built from the
             * record data earlier in this method
             */
            createRelation(con, halRes.getID(), resource.getID(), "Hallinnoija", false);
        }
    }
         
    /**
     * @param sourceID ID of the source of the parameter record
     * @param records a Vector filled with Records to be processed
     * @con DB Connection to the tables of this resource graph
     */
    public void resolveDocumentReferences(Connection con, Record rec,
                                          String sourceID) throws
                                          IntegrationException {
        Statement stm = null;
        ResultSet rs = null;
        String query = "";

        Vector values;

        try {
            stm = con.createStatement();
            
            /*
             * Repeat for fields Viitattu and Viittaava
             */
            for (int i = 0; i < 2; ++i) {
                values = rec.getField(i == 0 ? "Viitattu" : "Viittaava");
                if (values == null) {
                    continue;
                }
            
                /*
                 * Search for referenced documents by extended IDs
                 */
                Iterator iter = values.iterator();
                while (iter.hasNext()) {
                    String relatedRes = (String)iter.next();
                    query = "SELECT id FROM Resource, ExtIDs" +
                            " WHERE id=resource AND type='Document'" +
                            " AND extID=" + DbTools.SQLstr(relatedRes, Resource.IDLENGTH);
                    rs = stm.executeQuery(query);
                
                    /*
                     * Create relations for every found document
                     * If none found, create a new document and relation
                     */
                    if (rs.next()) {
                        do {
                            createRelation(con, createDocID(rec),
                                           rs.getString(1), "BIBREF", i == 1);
                        } while(rs.next());
                    }
                    else {
                        Resource newRes = new Resource(relatedRes, "Document");
                        relatedRes = newRes.findUniqueID(con);
                        newRes.setField("ExtIDs.extID", relatedRes);
                        newRes.setField("Records." + sourceID + "¤" +
                            rec.getID(), DbTools.formatDate(new Date()));
                        newRes.save(con);
                        createRelation(con, createDocID(rec),
                                       relatedRes, "BIBREF", i == 1);
                    }
                }
            }
        }
        catch (SQLException e) {
            System.out.println("resolveDocRefs failed with record: " +
                rec.getID() + "\nSQLex: " + e.getMessage());
            System.out.println("Continuing anyway...");
        }
        finally {
            try {
                if (rs != null)
                    rs.close();
            } catch (SQLException e) {}
            try {
                if (stm != null)
                    stm.close();
            } catch (SQLException e) {}
        }
    }


    /**
     * Retrieves a Resource (by types and canoName) from the DB,
     * or creates a new Resource if none exist. If a new Resource is
     * created, parameters are used to pre-fill some fields.
     *
     * @param con Connection to the DB. Make sure it's not null!
     * @param resourceType Actor, Channel, Document or Role
     * @param subType detailed type of the resource. Make sure it
     *   corresponds to the DB table column specifications!
     * @param newResID ID of the resource if a new one is created.
     *   Note: if the ID is already in use, a number will be
     *   automatically appended to create a unique ID
     * @param canoName Canonized name
     *
     * @returns A Resource that matches the parameters given
     */
    public Resource getResource(Connection con, String resourceType,
                                String subType, String newResID,
                                String canoName)
                                throws IntegrationException {

        Statement stm = null;
        ResultSet rs = null;
        String query = "";
        
        try {
            stm = con.createStatement();
            query = "SELECT Resource.id FROM Resource, " + resourceType +
                    " WHERE Resource.id=" + resourceType + ".id AND" +
                    " canonicalName='" + DbTools.escSQL(canoName) + "'";
            if (subType != null) {
                query += " AND " + resourceType + ".type='" + subType + "'";
            }
            debug(query);
            rs = stm.executeQuery(query);

            String resourceID = null;
            if (rs.next()) {
                resourceID = rs.getString(1);
            }
            
            /*
             * If an existing resource was found, we'll load it
             * and return it, otherwise create a new one
             */
            Resource resource = null;
            if (resourceID != null) {
                resource = new Resource(resourceID, resourceType);
                boolean loadStatus = resource.load(con);
                debug(resourceType + " resource loaded: " + resourceID + ", status: " + loadStatus);
            }
            else {
                resource = new Resource(newResID, resourceType);
                resourceID = resource.findUniqueID(con);
                resource.setField("Resource.canonicalName", canoName);
                if (subType != null) {
                    resource.setField(resourceType + ".type", subType);
                }
                debug("Created new " + resourceType + " resource, id=" + resourceID);
            }
        
            return resource;
        }
        catch (SQLException e) {
            throw new IntegrationException("getRelation failed SQLex(" +
                e.getMessage() + ")\nPrevious query: " + query);
        }
        finally {
            try {
                if (rs != null)
                    rs.close();
            } catch (SQLException e) {}
            try {
                if (stm != null)
                    stm.close();
            } catch (SQLException e) {}
        }
    }


    /**
     * Creates a relation between two resources, and saves it
     * in the DB's ResourceRelation table, unless one already exists.
     *
     * The specified role resource is created, unless one already
     * exists. Role types 'NONE' and 'BIBREF' are special values
     * for Channel/Document and Document/Document relations
     * respectively. These should be pre-inserted in the DB during
     * the DB table setup.
     *
     * @param con Connection to the DB. Make sure it's not null!
     * @param subject ID of the resource that points
     * @param object ID of the resource that is being pointed
     * @param roleType type of the relation, must be one of the
     *   enumerated values for DB's Role.type column
     * @param reversed set true to reverse the direction of the
     *   relation (subject becomes object and vice versa)
     */
    public void createRelation(Connection con, String subject,
                               String object, String roleType,
                               boolean reversed)
                               throws IntegrationException {

        if (subject == null || object == null || roleType == null) {
            return;
        }
        
        if (reversed) {
            String temp = object;
            object = subject;
            subject = temp;
        }
        
        String SQLsubject = DbTools.SQLstr(subject, Resource.IDLENGTH);
        String SQLobject = DbTools.SQLstr(object, Resource.IDLENGTH);
        
        boolean autoCommit = true;
        Statement stm = null;
        ResultSet rs = null;
        String query = "";
            
        try {
            autoCommit = con.getAutoCommit();
            con.setAutoCommit(false);
            stm = con.createStatement();
        
            /*
             * Role type NONE for Channel - Document relations
             * Role type BIBREF for Document - Document relations
             */
            if (roleType.equals("NONE") || roleType.equals("BIBREF")) {
                query = "SELECT * FROM ResourceRelation WHERE role='" + roleType + 
                        "' AND subject=" + SQLsubject + " AND object=" + SQLobject;
                debug(query);
                rs = stm.executeQuery(query);
                /*
                 * Save the relation, if it doesn't already exist
                 */
                if (!rs.next()) {
                    query = "INSERT INTO ResourceRelation SET role='" + roleType + 
                            "',subject=" + SQLsubject + ",object=" + SQLobject;
                    debug(query);
                    stm.executeUpdate(query);
                }
            }
            else {
                /*
                 * Check if a suitable role already exists
                 * If not, create a new one
                 */
                query = "SELECT Role.id FROM Role, ResourceRelation " +
                        "WHERE Role.id=ResourceRelation.role AND object=" + 
                        SQLobject + " AND Role.type='" + roleType + "'";
                rs = stm.executeQuery(query);
                debug(query);
            
                String roleID = null;
                if (rs.next()) {
                    roleID = rs.getString(1);
                }
                else {
                    Resource newRole = new Resource(roleType + "_" + object, "Role");
                    roleID = newRole.findUniqueID(con);
                    newRole.setField("Resource.name", "");
                    newRole.setField("Resource.canonicalName", "");
                    newRole.setField("Role.type", roleType);
                    newRole.save(con);
                }

                /*
                 * Save the relation, if it doesn't already exist
                 */
                query = "SELECT Role.id FROM Role, ResourceRelation " +
                        "WHERE Role.id=ResourceRelation.role AND object=" + 
                        SQLobject + " AND Role.type='" + roleType + "' " +
                        "AND subject=" + SQLsubject;
                rs = stm.executeQuery(query);
                if (!rs.next()) {
                    query = "INSERT INTO ResourceRelation SET role=" + 
                            DbTools.SQLstr(roleID, Resource.IDLENGTH) + 
                            ",subject=" + SQLsubject + ",object=" + SQLobject;
                    debug(query);
                    stm.executeUpdate(query);
                }
            }
        
            con.commit();
        }
        catch (SQLException e) {
            throw new IntegrationException("createRelation failed SQLex(" +
                e.getMessage() + ")\nPrevious query: " + query);
        }
        finally {
            try {
                con.setAutoCommit(autoCommit);
            } catch(SQLException e) {}
            try {
                if (rs != null)
                    rs.close();
            } catch (SQLException e) {}
            try {
                if (stm != null)
                    stm.close();
            } catch (SQLException e) {}
        }
    }
    
    /**
     * Creates an ID for a document-type resource.
     * The ID is derived from the fields of the
     * parameter-Record
     */
    public static String createDocID(Record rec) {
        String tmpId = null;
        
        String[] fieldnames = {"HenkilöTekijä", "OrganisaatioTekijä", "JokuTekijä"};
        String author = null;
        for(int i = 0; i < fieldnames.length && author == null; ++i) {
            author = getFirstVal(rec.getField(fieldnames[i]));
        }
        if (author != null && author.length() >= 3)
            tmpId = author.substring(0, 3);
        else 
            tmpId = "xxx";
            
        String year = getFirstVal(rec.getField("Julkaisupäivämäärä"));
        if (year != null && year.length() >= 4) 
            tmpId += year.substring(2, 4);
        else
            tmpId += "XX";
            
        String title = getFirstVal(rec.getField("Nimeke"));
        if (title != null && title.length() > 3) {
            String titleWords[] = title.split(" ");
            String titleWord = null;
            for (int i = 0; i < titleWords.length; ++i) {
                if (titleWords[i].length() > 3) {
                    titleWord = titleWords[i];
                    break;
                }
            }
            if (titleWord == null) {
                titleWord = "xxxx";
            }
            tmpId += titleWord;
        }
        else
            tmpId += "xxxx";
        
        /*
         * If all fields were blank, create a six-letter random ID
         */
        if (tmpId.equals("xxxXXxxxx")) {
            tmpId = "";
            for (int i = 0; i < 6; ++i) {
                int increment = (int)(Math.random() * 25);
                char nextChar = (char)('a' + increment);
                tmpId += "" + nextChar;
            }
        }
        return tmpId;
    }

    /**
     * Returns the first String of a Vector, or null
     * if not possible
     */
    public static String getFirstVal(Vector v) {
        if (v == null) {
            return null;
        }
        try {
            return (String)v.get(0);
        }
        catch (ArrayIndexOutOfBoundsException e) {
            return null;
        }
    }

    private static void debug(String s) {
//        System.out.println("Debug IntegratorHelper: " + s);
    }
}